/******************************************************************************
File:        eloIoctl.h
Purpose:     Windows CE. DeviceIoControl declarations for Elo TouchSystems 
touchscreen drivers. Applications and drivers exchange information through the 
classes defined here. Applications may use these directly but doing so requires 
opening an interface to the specific touch device (CreateFile) and passing the 
object through DeviceIoControl. The EloApi library has already done this, 
defining an API class for each IOCTL class. Except for the few cases of 
variable size IOCTL class, instantiating the API class fully instantiates the 
IOCTL class. Most applications should use the API classes.

Copyright 2008 Elo TouchSystems
All rights reserved

Updated:    2008-09-04
******************************************************************************/
#ifndef ELOIOCTL_H
#define ELOIOCTL_H

#include <eloCodes.h>

#ifdef EloApi_Static
	#define T_EloApi
#else
	#ifdef EloApi_Export
		#define T_EloApi __declspec(dllexport)
	#else 
		#define T_EloApi __declspec(dllimport)
	#endif
#endif

enum { IocGet, IocSet };

class T_EloApi IocHdr
{
public:
    ELO_ERR mEloErr;        
    IocHdr( void ) { mEloErr = EloErrSuccess; }
};

#define IdxFlag(I) ( 1 << (I) )

// ===================== Simple Command Arguments ============================
// These define argument values for commands that use a single ULONG as input 
// to the driver and return from it, so an Ioc class is not needed. The 
// EloApi::Command class contains an interface function for each of these 
// commands.
// ............................................................................

enum rotateCmd { // Argument to IOCTL_ELO_ROTATE
    ORIENTATION_0,
    ORIENTATION_90,
    ORIENTATION_180,
    ORIENTATION_270,
    ROTATE_VIDEO = 0x100, // Flag to tell driver to rotate the display.
};

enum OnHold { // Bitmap for IOCTL_ELO_ON_HOLD and IocHoldParms.mOnHold
    ON_HOLD_EVENT =             1, 
    ON_HOLD_NO_EVENT =          2, 
    ON_HOLD_SHOW =              4,
    ON_HOLD_NO_SHOW =           8,
    ON_HOLD_AUTOUNTOUCH =       0x10, 
    ON_HOLD_NO_AUTOUNTOUCH =    0x20,
    ON_HOLD_GLOBAL =            0x40,
    ON_HOLD_TEMP =              0x80,
}; 
// Each of the three onHold features-- event, show, and untouch-- is turned off/on
// or no change according to two flags. The current onHold condition (after any
// changes due to this command) is returned by set flags, i.e. ON_HOLD_EVENT,  
// ON_HOLD_SHOW, and ON_HOLD_AUTOUNTOUCH. To just read the current state, pass
// an argument of 0. ON_HOLD_GLOBAL disables hold event for all touchscreens
// without changing each one's individual setting. ON_HOLD_GLOBAL+ON_HOLD_EVENT 
// removes this. ON_HOLD_GLOBAL+ON_HOLD_TEMP disables hold event for all until
// the next untouch (or ON_HOLD_GLOBAL+ON_HOLD_EVENT)

enum TouchSound { // For IOCTL_ELO_SOUND
    TOUCH_SOUND_OFF,
    TOUCH_SOUND_ON,
    TOUCH_SOUND_UPDATE,
    IS_TOUCH_SOUND_ON,
};

enum OnTouch { // Bitmap for IOCTL_ELO_ON_TOUCH
    ON_TOUCH_NOTHING =      0x0000,
    ON_TOUCH_SYSMOUSE =     0x0001, // Fully adjusted mouse event to system
    ON_TOUCH_APIMOUSE =     0x0002, // Fully adjusted mouse event through API
    ON_TOUCH_APIADJ =       0x0004, // Touchscreen domain (not translated) with adjustments.
    ON_TOUCH_APITRANS =     0x0008, // Translated to mouse but not adjusted 
                                    // (acceleration and linearity)
    ON_TOUCH_APIRAW =       0x0010, // Raw touchscreen coordinates
    ON_TOUCH_CLEANING =     0x0020, // No response until wakeup on hold.
    ON_TOUCH_IGNORENEXT =   0x0040, // Ignore all touch stream events up to and 
                // including the next new touch or untouch. Auto-clear at event.

// Event filters. These apply only to API. All events are passed to system if ON_TOUCH_SYSMOUSE
    ON_TOUCH_DOWN =         0x2000, 
    ON_TOUCH_STREAM =       0x4000,
    ON_TOUCH_UP =           0x8000,
// EloTch.mOnTouch is USHORT LOWORD of the ULONG API argument. HIWORD is used for
// API commands.
};  
// ON_TOUCH_APIRAW and ON_TOUCH_SYSMOUSE can be combined with any one of the 
// other three but ON_TOUCH_APIMOUSE, ON_TOUCH_APITRANS, and ON_TOUCH_APITRANS 
// are mutually exclusive. ON_TOUCH_APITRANS produces the same result as 
// ON_TOUCH_APIMOUSE with acceleration and linearity correction turned off. If 
// ON_TOUCH_SYSMOUSE and ON_TOUCH_APITRANS are used together, the mouse event 
// will not have acceleration or linearity correction.
#define ON_TOUCH_ALLEVENTS (ON_TOUCH_DOWN + ON_TOUCH_STREAM + ON_TOUCH_UP)
#define ON_TOUCH_ANY_API ( ON_TOUCH_APIMOUSE + ON_TOUCH_APIADJ + ON_TOUCH_APITRANS + ON_TOUCH_APIRAW )
#define OnTouchOnlyRaw(T) (((T) & ~ON_TOUCH_ALLEVENTS) == ON_TOUCH_APIRAW )
#define OnTouchAnyApi(T) (((T) & ON_TOUCH_ANY_API ) != 0 )

enum { ON_TOUCH_SET, ON_TOUCH_GET }; // IOCTL_ELO_ON_TOUCH HIWORD( ULONG ) commands.

enum SaveCalTo { // Bitmap for IOCTL_ELO_SAVECAL
    SAVECAL_DEFAULT =       1,  // Driver read CalStore to determine registry and/or controller.
    SAVECAL_REG =           2,  // Write to registry.
    SAVECAL_CONTROLLER =    4,  // Write to controller.
    SAVECAL_EDGES =         8,  // Write edge calibration parameters to registry (no controller option).
};

enum DevLink { LINK_UNKNOWN, LINK_SERIAL, LINK_USB }; // IOCTL_ELO_GETLINKTYPE

enum DevIdCmd { DEVIDCMD_WR, DEVIDCMD_RESET, DEVIDCMD_RD }; // IOCTL_ELO_APIDEVID 

// ============================================================================
    
enum { ELOMM_CLICK_ON_TOUCH, ELOMM_CLICK_ON_UNTOUCH, ELOMM_EMULATE_MOUSE }; // MouseMode

// --------------------------IOCTL_ELO_CFGITEM -----------------------------
// Get or Set one configuration item. Item is identified by registry item 
// ordinal defined in eloRegNames.h in the range EloRegIdxPriority through 
// EloRegIdxCalStore. Get reads the current working value without reference to 
// the registry. Set sets the working value, writes the new value to the 
// registry, and may perform item-specific actions. 
// See eloApi.h: EloApi::CmdEloCfg, EloApi::setCfgItem and getCfgItem
class T_EloApi IocCfgItem : public IocHdr
{
public:
    USHORT  mOp; // IocSet/Get
    int     mId; // registry ordinal 
    DWORD   mDat;
};

//------------------------IOCTL_ELO_HOLD_PARMS---------------------------------
// See eloApi.h: EloApi::HoldParms
enum { 
    // Individual feature indices. These index mParms.
    HPIDX_TICK,             // Hold timer tick period in ms
    HPIDX_JIGGLE,           // Hold perimeter in absolute touch units.
    HPIDX_UNTOUCH_TICKS,    // Number of ticks until forced untouch.
    HPIDX_EVENT_TICKS,      // Tick count at which to generate hold event (RCOH or other)
    HPIDX_SHOW_TICKS,       // Tick count at which begin to provide hold visual feedback.
    // mParms[ HPIDX_SHOW_TICKS ] > mParms[ HPIDX_EVENT_TICKS ] signifies no feedback.
    HPIDX_SHOW_STEP,        // Feedback step size in video pixels.

    // Command pseudo-indices. Produce mCmd flags but no elements of mParms.
    HPIDX_ONHOLD, 
    HPIDX_GET		// Get all current parameters and flags. 
};  // Indices for accessing HoldParms.mParms array and for creating
// corresponding HoldParms.mCmd bit-mapped flags (using IdxFlag). 

class T_EloApi HoldParms
{
public:
    USHORT  mP[6];  // HPIDX_TICK, HPIDX_JIGGLE, HPIDX_UNTOUCH_TICKS, 
                   // HPIDX_EVENT_TICKS, HPIDX_SHOW_TICKS, HPIDX_SHOW_STEP
    USHORT& tick( void )            { return mP[ HPIDX_TICK ]; } 
    USHORT& jiggle( void )          { return mP[ HPIDX_JIGGLE ]; } 
    USHORT& untouchTicks( void )    { return mP[ HPIDX_UNTOUCH_TICKS ]; } 
    USHORT& eventTicks( void )      { return mP[ HPIDX_EVENT_TICKS ]; } 
    USHORT& showTicks( void )       { return mP[ HPIDX_SHOW_TICKS ]; } 
    USHORT& showStep( void )        { return mP[ HPIDX_SHOW_STEP ]; } 
};

const HoldParms DefaultHoldParms = { 
    100,    // TICK
    1000,   // JIGGLE
    0,      // UNTOUCH_TICKS
    8,      // EVENT_TICKS
    1,      // SHOW_TICKS
    10      // SHOW_STEP
};

class T_EloApi IocHoldParms : public IocHdr
{
public:
    HoldParms mParms;
    USHORT  mCmd; // Bit-mapped flags IdxFlag[ HPIDX_TICK...HPIDX_SHOW_STEP].
    USHORT  mOnHold; // OnHold enable/disable flags. Only positives 
    // (ON_HOLD_EVENT, ON_HOLD_SHOW, ON_HOLD_AUTOUNTOUCH) if ON_HOLD_GET.
};

// IOCTL_ELO_ON_HOLD is Simple Command
// IOCTL_ELO_SOUND is Simple Command
// IOCTL_ELO_ON_TOUCH is Simple Command

// ------------------------ IOCTL_ELO_MOUSELIMITS -------------------------
// See eloApi.h: EloApi::CmdMouseLimits
enum { EloLimitGet, EloLimitSetRel, EloLimitSetAbs };
class T_EloApi MouseLimits
{
public:
    long edge[4]; // Index by VID_LEFT, VID_TOP, VID_RIGHT, VID_BOTTOM 
};

class T_EloApi IocMouseLimits 
{   
public:
    USHORT mOp; // EloLimitGet, EloLimitSetRel, EloLimitSetAbs 
    MouseLimits limit;
};

// ---------------------IOCTL_ELO_REGKEYNAME ----------------------------
#define MAX_ELOREGKEYNAME 100
class T_EloApi IocRegKeyName : public IocHdr
{
public:
    bool    mOp;  // IocSet/Get 
    TCHAR	mName[ MAX_ELOREGKEYNAME + 1 ];
};

// -------------------IOCTL_ELO_SMARTSET---------------------------------------
// See eloApi.h: class EloSmartSet
enum { SS_TRANSFER_NORMAL, SS_TRANSFER_NOWAIT, 
SS_TRANSFER_SHORTWAIT, SS_TRANSFER_LONGWAIT, SS_TRANSFER_WAITFOREVER 
}; // TransferStatus.mCmd

typedef struct { 
    short   mCnt; 
    USHORT  mCmd; // SS_TRANSFER_x
    DWORD   mErr; // return from communication function.
}   TranStat;

class T_EloApi EloTrans
{
public:
    UCHAR       mBuf[10];
    TranStat    mStat;
};

class T_EloApi IocSmartSet : public IocHdr
{
public:
    EloTrans    mIn;
    EloTrans    mOut;
};

//=============================================================================
//                                 CALIBRATION
//=============================================================================

typedef struct { USHORT mX; USHORT mY; } RawPoint;

// ............. Edge indices in video and touchscreen domains ...................
enum { VID_LEFT, VID_TOP, VID_RIGHT, VID_BOTTOM };          // Video domain
enum { TSEDGE_XLO, TSEDGE_YLO, TSEDGE_XHI, TSEDGE_YHI };    // Touchscreen domain 

//............. Corner indices in video and touchscreen domains .............
enum { VID_LEFTTOP, VID_RIGHTTOP, VID_LEFTBOTTOM, VID_RIGHTBOTTOM }; // Video domain
 
// Video domain corner indices based on top/bottom as b1, left/right as b0
enum { VID_CORNER_LEFT = 1, VID_CORNER_TOP = 2 };
#define IsLeftCorner(I) (( I & VID_CORNER_LEFT ) == 0 )
#define IsTopCorner(I)  (( I & VID_CORNER_TOP ) == 0 )

enum { TSCORNER_YLXL, TSCORNER_YLXH, TSCORNER_YHXL, TSCORNER_YHXH }; // TS domain

// TS domain corner indices based on Y as b1, X as b0.
enum { TS_CORNER_X = 1, TS_CORNER_Y = 2 };
#define IsXlCorner(I)   ((I & TS_CORNER_X) == 0 )
#define IsYlCorner(I)   ((I & TS_CORNER_Y) == 0 )

const long VirtualVideoX = 65535L;
const long VirtualVideoY = 65535L;

//-------------------------- IOCTL_ELO_GETPOINT -------------------------------
// See eloApi.h: EloApi::CmdGetPt
enum { TCH_FAIL, TCH_DISCONNECT, TCH_NOEVENT, TCH_TOUCH, TCH_CLICK, TCH_HOLD };
class T_EloApi ApiTchEvent  
{
public:
    USHORT  mTch; // TCH_FAIL, TCH_DISCONNECT, TCH_NOEVENT, etc.
    USHORT  mX;
    USHORT  mY;
    USHORT  mZ;
    long    mTransX;
    long    mTransY;
    USHORT  mApiDevId;
    ApiTchEvent() { mTch = TCH_NOEVENT; }
    void    reset( void ) { mTch = TCH_NOEVENT; }
    bool    isError( void ) { return mTch < TCH_NOEVENT; }
    bool    isTouch( void ) { return mTch > TCH_NOEVENT; }
};

class T_EloApi IocGetPt : public IocHdr
{
public:
    bool    mBlock; // In. If true then block in device driver waiting for event.
    ApiTchEvent mEvent;
};

// =============================== 3-PT CALIBRATION ===========================
typedef long T_CAL;

enum { ORIENT_INVX = 1, ORIENT_INVY = 2, ORIENT_SWAPXY = 4 };

typedef struct {
    T_CAL   mOffX;	// Touchscreen offset from video normalized to 65536 in video X axis.
    T_CAL   mOffY;	// " Y axis.
    T_CAL   mVidDx; // Video calibration distance in X axis normalized to 65536.
    T_CAL   mVidDy; // " in Y axis.
// mTsDx and mTsDy are relative to the video X and Y axes during calibration. They
// may not match the touchscreen controller's orientation sense.
    T_CAL   mTsDx;	// Touchscreen calibration distance in TS units in video X axis. 
    T_CAL   mTsDy;	// " in video Y axis.
    USHORT  mSwapXy;
    USHORT  mCalcTvRel( void ) { return  ( mSwapXy ? ORIENT_SWAPXY : 0 ) | 
        ( mTsDy < 0 ? ORIENT_INVY : 0 ) | ( mTsDx < 0 ? ORIENT_INVX : 0 ); }
}   Cal3Data;

typedef struct {
    USHORT  mTvRelation; // Index/selector indicating touch to video orientation.
    USHORT  mBorder[4]; // TSEDGE_XLO, TSEDGE_YLO, TSEDGE_XHI, TSEDGE_YHI
}	Cal3Ex;

class T_EloApi Cal3DatEx
{
public:
	Cal3Data	mData;
	Cal3Ex		mEx;
};

// ------------------------- IOCTL_ELO_CAL3DAT ----------------------------------
// See eloApi.h: EloApi::CmdCal 
class T_EloApi IocCal3Dat : public IocHdr
{   
public:
    USHORT mOp; // IocSet/Get
   Cal3DatEx dat;
};

// =========================== EDGE CALIBRATION ===============================
typedef short T_ACC_RATE;
typedef USHORT T_ACC_POS;

enum { LINCOR_UNKNOWN, LINCOR_NOTREQUIRED, LINCOR_SEGMENTED }; // EdgeCal.mType
class T_EloApi EdgeCal
{
public:
    USHORT      mType;
    T_ACC_RATE  mLcAcc[4]; // Portion of linearity compensation represented by acceleration.
    USHORT      mCntPts[4]; // Number of inflection points each edge list.
};

//----------------------- IOCTL_ELO_EDGECAL ---------------------------------
// See eloApi.h: EloApi::CmdEdgeCal
class T_EloApi IocEdgeCal : public IocHdr
{   
public:
    static int mByteCnt( int ptCnt ) { return 
        sizeof( IocEdgeCal ) + ( ptCnt - 1 ) * sizeof( RawPoint ); }

    USHORT      mSize;  // The total size of this IocEdgeCal instance
    // includes mSize through extensible mPts. Calculate by 
    // sizeof( IocEdgeCal ) + ( count points - 1 ) * sizeof( RawPoint ).
    EdgeCal     mEdgeCal;
    USHORT      mOp;		// IocGet/Set
    USHORT      mPtsByteCnt; // Size of mPts
    RawPoint    mPts[1]; // First element of extensible array of inflection points.
};

// =================== LINEARITY CORRECTION AND ACCELERATION ==================

// ------------------------- IOCTL_ELO_LINCOR ----------------------------------
// See eloApi.h: EloApi::CmdLinCor
class T_EloApi LinCor
{
public:
    UCHAR       mOn[4]; // TSEDGE_XLO,...
    T_ACC_RATE  mLcAdj[4]; // TSEDGE_XLO,... acceleration adjustment.
    // See CornerAcc.mXoff and mYoff for edge calibration ideal position.
};

enum {  
    LINCOR_GET = 0, 
    LINCOR_SET_ON = 1, 
    LINCOR_SET_ADJ = 2 
};  // IocLinCor.mOp Get or bit-mapped set.

class T_EloApi IocLinCor
{   
public:
    USHORT  mOp; // LINCOR_GET, LINCOR_SET_ON, LINCOR_SET_ADJ
    LinCor  mLinCor;
};

// ------------------------- IOCTL_ELO_ACCEL ----------------------------------
// See eloApi.h: EloApi::CmdAccel
// An edge with linearity correction always has edge acceleration, using a method
// that is different from the standard edge acceleration. This method uses the
// same mPos and mRate members of EloAcc as standard acceleration but mMethod is
// ignored. 
enum { ACC_OFF, ACC_ON, ACC_DEFAULT }; // EloAcc.mMethod
#define ACC_MULT 256 

class T_EloApi EloAcc
{
public:
// All arrays indexed in TS domain: TSEDGE_XLO, TSEDGE_YLO, TSEDGE_XHI, TSEDGE_YHI 
    UCHAR       mMethod[4]; // ACC_NONE, ACC_DEFAULT, ACC_ON. 
    T_ACC_POS   mPos[4];        
    T_ACC_RATE  mRate[4];   
};

enum {  GET_ACCEL =         0, 
        SET_ACCEL_METHOD =  1, 
        SET_ACCEL_RATE =    2, 
        SET_ACCEL_POS =     4 }; // IocAccel.mOp

class T_EloApi IocAcc 
{   
public:
    USHORT  mOp; // GET_ACCEL...SET_ACCEL_POS
    EloAcc  mAcc;
};

// ------------------------- IOCTL_ELO_CORNERACC --------------------------------
// See eloApi.h: EloApi::CmdCornerAcc
enum { CACC_GET = 0, CACC_SET_ACC = 1, CACC_SET_ENABLE = 2, CACC_ON = 4 };
class T_EloApi CornerAcc
{
public:
    USHORT  mRange[4];  // Size of capture range in TS points. Indexed in TS 
    // domain TSCORNER_YLXL, TSCORNER_YLXH, TSCORNER_YHXL, TSCORNER_YHXH
    USHORT  mXtarget;   // Edge offset to target destination in touchscreen 
    // domain. Shared with linearity correction. X value is currently used 
    // for both X and Y.
};

class T_EloApi IocCornerAcc
{
public:
    USHORT      mOp; // CACC_GET or bitmap of CACC_SET_ACC | CACC_SET_ENABLE |
    // CACC_ENABLE. For CACC_GET, driver returns op = 0 if not enabled else non-0.
    CornerAcc   mAcc;
};

// -------------------------- IOCTL_ELO_DEVCFG -----------------------------
class T_EloApi DevCfg
{
public:
    EdgeCal     mEdgeCal;
    EloAcc      mAcc;
    CornerAcc   mCornerAcc;
    LinCor      mLinCor;
    MouseLimits	mMouseLimits;   // long edge[4] index by VID_LEFT, VID_TOP, 
	                                // VID_RIGHT, VID_BOTTOM
    DWORD       mPriority;
    DWORD       mHighPriority;
    DWORD       mMouseMode;
    DWORD       mAntiBounce;
    TCHAR       mPortName[13]; // "USB" or serial name specified in Registry, e.g. "COM1:"

	USHORT		mVendorID;
	USHORT		mProductID;
	BOOL		mSupportSmartset;
};

class T_EloApi IocDevCfg
{   
public:
    USHORT  mOp; // IocSet/Get
    DevCfg	mDevCfg;
};

#endif  // ndef ELOIOCTL_H



