/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.viewmodel

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.util.Log
import androidx.lifecycle.ViewModel
import com.eloview.sdk.accountManager.AccountManager
import com.eloview.sdk.packageManager.Package
import com.eloview.sdk.systemManager.ServiceConnectionCallback
import com.eloview.sdk.testapp.GlobalClass
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow

data class PackageActivityState(
    var serviceConnected: Boolean = false,
    var output: String = "",
    var silentInstallPackagePath: String = "/storage/emulated/0/Download/example.apk",
    var silentUninstallPackage: String = "",
    var stopPackageName: String = "",
    var enablePackageName: String = "",
    var disablePackageName: String = "",
    var packageStatusPackageName: String = "",
    var whiteListPackagePermission: String = "",
    var packagePermission: String = "",
    var packageName: String = "",
    var clearPackagePermission: String = "",
    var enablePackageWhitelisting: String = "",
    var showSilentUninstallDialog: Boolean = false,
    var showStopPackageDialog: Boolean = false,
    var showEnablePackageDialog: Boolean = false,
    var showDisablePackageDialog: Boolean = false,
    var showGetPackageStatusDialog: Boolean = false,
    var showAddWhiteListPackagePermissionDialog: Boolean = false,
    var showSetPackageNameAndPermissionDialog: Boolean = false,
    var showClearPackagePermissionDialog: Boolean = false,
    var showEnablePackageWhitelistingDialog: Boolean = false,
    var showAutoStartAppAndLockItDialog: Boolean = false,
    var autoStartAppAndLockItPackageNameHint: String = "com.example.example",
    var autoStartAppAndLockItOldPasswordHint: String = "Old Password",
    var autoStartAppAndLockItNewPasswordHint: String = "New Password",
    var autoStartAppAndLockItPackageName: String = "",
    var autoStartAppAndLockItOldPassword: String = "",
    var autoStartAppAndLockItNewPassword: String = "",
    var showRemoveTaskAutoStartAndLockingDialog: Boolean = false,
    var temporaryUnpinCheckBox: Boolean = false,
    var permanentUnpinCheckBox: Boolean = false,
    var removeTaskAutoStartAndLockingPassword: String = "",
    var enablePackageWhitelistingBlockPackageCheckBox: Boolean = false
)

class PackageActivityViewModel(val applicationContext: Context, preview: Boolean = false) :
    ViewModel(),
    ServiceConnectionCallback {
    companion object {
        private const val TAG = "PackageActivityViewModel"
        var isPreview: Boolean = false
    }

    val packageServiceManager = if (preview) {
        isPreview = true
        null
    } else {
        isPreview = false
        Package()
    }

    val packageHandler: Handler = Handler(Looper.getMainLooper()) { msg ->
        val b = msg.data
        val key = msg.what.toString()
        var `val` = b.getString(key)
        when (msg.what) {
            AccountManager.TOKEN_VERIFY_FAIL -> {
                Log.v(TAG, "Invalid accessToken")
                `val` = "INVALID TOKEN\n$`val`"
                unbindService()
            }

            Package.INSTALL_LISTENER_REGISTERED -> {
                Log.v(TAG, "INSTALL_LISTENER_REGISTERED")
                `val` = "INSTALL_LISTENER_REGISTERED\n$`val`"
            }

            Package.UNINSTALL_LISTENER_REGISTERED -> {
                Log.v(TAG, "UNINSTALL_LISTENER_REGISTERED")
                `val` = "UNINSTALL_LISTENER_REGISTERED\n$`val`"
            }

            Package.INSTALL_LISTENER_UNREGISTERED -> {
                Log.v(TAG, "INSTALL_LISTENER_UNREGISTERED")
                `val` = "INSTALL_LISTENER_UNREGISTERED\n$`val`"
            }

            Package.UNINSTALL_LISTENER_UNREGISTERED -> {
                Log.v(TAG, "UNINSTALL_LISTENER_UNREGISTERED")
                `val` = "UNINSTALL_LISTENER_UNREGISTERED\n$`val`"
            }

            AccountManager.TOKEN_VERIFY_SUCCESS -> {
                Log.v(TAG, "Token available")
                `val` = "Token: $`val`"
            }

            Package.LISTENER_REGISTRATION_ERROR -> {
                Log.d(
                    TAG,
                    "LISTENER_REGISTRATION_ERROR: \n$`val`"
                )
                `val` = "Internal Error : $`val`"
            }

            Package.INSTALLER_RESULT_CODE -> {
                Log.d(TAG, "INSTALLER_RESULT_CODE: \n$`val`")
                `val` = "INSTALLER_RESULT_CODE: $`val`"
            }

            Package.PACKAGE_STATUS_CHANGE_LISTENER_CODE -> {
                Log.d(
                    TAG,
                    "PACKAGE_STATUS_CHANGE_LISTENER_CODE: $`val`"
                )
                `val` = "PACKAGE_STATUS_CHANGE_LISTENER_CODE: \n$`val`"
            }

            Package.PACKAGE_STATUS_CHANGE_CODE -> {
                Log.d(
                    TAG,
                    "PACKAGE_STATUS_CHANGE_CODE: $`val`"
                )
                `val` = "PACKAGE_STATUS_CHANGE_CODE: \n$`val`"
            }

            Package.SET_WHITELIST_APP_PERMISSION_RESULT_CODE -> {
                Log.d(
                    TAG,
                    "SET_WHITELIST_APP_PERMISSION_RESULT_CODE: \n$`val`"
                )
                `val` = "SET_WHITELIST_APP_PERMISSION_RESULT_CODE: $`val`"
            }

            Package.SET_WHITELIST_RULE_RESULT_CODE -> {
                Log.d(
                    TAG,
                    "SET_WHITELIST_RULE_RESULT_CODE: \n$`val`"
                )
                `val` = "SET_WHITELIST_RULE_RESULT_CODE: $`val`"
            }

            Package.PACKAGE_STATUS_RESULT_CODE -> {
                Log.d(
                    TAG,
                    "PACKAGE_STATUS_RESULT_CODE: \n$`val`"
                )
                `val` = "PACKAGE_STATUS_RESULT_CODE $`val`"
            }
        }
        if (`val` != null) {
            setOutput(`val`)
        }
        false
    }

    private val _packageActivityState = MutableStateFlow(PackageActivityState())
    val packageActivityState: StateFlow<PackageActivityState> = _packageActivityState.asStateFlow()

    private fun setServiceConnected(serviceConnected: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(serviceConnected = serviceConnected)
    }

    fun getServiceConnected(): Boolean {
        return _packageActivityState.value.serviceConnected
    }

    fun bindService() {
        packageServiceManager?.bindService(
            applicationContext,
            this,
            GlobalClass.globalViewModel.getAccessToken()
        )
    }

    private fun unbindService() {
        packageServiceManager?.unbindService(applicationContext)
        setServiceConnected(false)
    }

    override fun onServiceConnected() {
        setServiceConnected(true)
        Log.d(TAG, "onServiceConnected: Service connected")
    }

    override fun onServiceDisconnected() {
        setServiceConnected(false)
        Log.d(TAG, "onServiceDisconnected: Service disconnected")
    }

    override fun onTokenVerifyFailed() {
        setServiceConnected(false)
        Log.e(TAG, "onTokenVerifyFailed: Token verify failed")
    }

    fun setSilentInstallPackagePath(path: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(silentInstallPackagePath = path)
    }

    fun getSilentInstallPackagePath(): String {
        return _packageActivityState.value.silentInstallPackagePath
    }

    fun setOutput(output: String) {
        _packageActivityState.value = _packageActivityState.value.copy(output = output)
    }

    fun getOutput(): String {
        return _packageActivityState.value.output
    }

    fun setSilentUninstallPackage(packageName: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(silentUninstallPackage = packageName)
    }

    fun getSilentUninstallPackage(): String {
        return _packageActivityState.value.silentUninstallPackage
    }

    fun setStopPackageName(packageName: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(stopPackageName = packageName)
    }

    fun getStopPackageName(): String {
        return _packageActivityState.value.stopPackageName
    }

    fun setEnablePackageName(packageName: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(enablePackageName = packageName)
    }

    fun getEnablePackageName(): String {
        return _packageActivityState.value.enablePackageName
    }

    fun setDisablePackageName(packageName: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(disablePackageName = packageName)
    }

    fun getDisablePackageName(): String {
        return _packageActivityState.value.disablePackageName
    }

    fun setPackageStatusPackageName(packageName: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(packageStatusPackageName = packageName)
    }

    fun getPackageStatusPackageName(): String {
        return _packageActivityState.value.packageStatusPackageName
    }

    fun setWhiteListPackagePermission(packageName: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(whiteListPackagePermission = packageName)
    }

    fun getWhiteListPackagePermission(): String {
        return _packageActivityState.value.whiteListPackagePermission
    }

    fun setPackagePermission(permission: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(packagePermission = permission)
    }

    fun getPackagePermission(): String {
        return _packageActivityState.value.packagePermission
    }

    fun setPackageName(packageName: String) {
        _packageActivityState.value = _packageActivityState.value.copy(packageName = packageName)
    }

    fun getPackageName(): String {
        return _packageActivityState.value.packageName
    }

    fun setClearPackagePermission(packageName: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(clearPackagePermission = packageName)
    }

    fun getClearPackagePermission(): String {
        return _packageActivityState.value.clearPackagePermission
    }

    fun setEnablePackageWhitelisting(packageName: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(enablePackageWhitelisting = packageName)
    }

    fun getEnablePackageWhitelisting(): String {
        return _packageActivityState.value.enablePackageWhitelisting
    }

    fun setShowSilentUninstallDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showSilentUninstallDialog = show)
    }

    fun getShowSilentUninstallDialog(): Boolean {
        return _packageActivityState.value.showSilentUninstallDialog
    }

    fun setShowStopPackageDialog(show: Boolean) {
        _packageActivityState.value = _packageActivityState.value.copy(showStopPackageDialog = show)
    }

    fun getShowStopPackageDialog(): Boolean {
        return _packageActivityState.value.showStopPackageDialog
    }

    fun setShowEnablePackageDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showEnablePackageDialog = show)
    }

    fun getShowEnablePackageDialog(): Boolean {
        return _packageActivityState.value.showEnablePackageDialog
    }

    fun setShowDisablePackageDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showDisablePackageDialog = show)
    }

    fun getShowDisablePackageDialog(): Boolean {
        return _packageActivityState.value.showDisablePackageDialog
    }

    fun setShowGetPackageStatusDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showGetPackageStatusDialog = show)
    }

    fun getShowGetPackageStatusDialog(): Boolean {
        return _packageActivityState.value.showGetPackageStatusDialog
    }

    fun setShowAddWhiteListPackagePermissionDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showAddWhiteListPackagePermissionDialog = show)
    }

    fun getShowAddWhiteListPackagePermissionDialog(): Boolean {
        return _packageActivityState.value.showAddWhiteListPackagePermissionDialog
    }

    fun setShowSetPackageNameAndPermissionDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showSetPackageNameAndPermissionDialog = show)
    }

    fun getShowSetPackageNameAndPermissionDialog(): Boolean {
        return _packageActivityState.value.showSetPackageNameAndPermissionDialog
    }

    fun setShowClearPackagePermissionDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showClearPackagePermissionDialog = show)
    }

    fun getShowClearPackagePermissionDialog(): Boolean {
        return _packageActivityState.value.showClearPackagePermissionDialog
    }

    fun setShowEnablePackageWhitelistingDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showEnablePackageWhitelistingDialog = show)
    }

    fun getShowEnablePackageWhitelistingDialog(): Boolean {
        return _packageActivityState.value.showEnablePackageWhitelistingDialog
    }

    fun setShowAutoStartAppAndLockItDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showAutoStartAppAndLockItDialog = show)
    }

    fun getShowAutoStartAppAndLockItDialog(): Boolean {
        return _packageActivityState.value.showAutoStartAppAndLockItDialog
    }

    fun setAutoStartAppAndLockItPackageName(packageName: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(autoStartAppAndLockItPackageName = packageName)
    }

    fun getAutoStartAppAndLockItPackageName(): String {
        return _packageActivityState.value.autoStartAppAndLockItPackageName
    }

    fun setAutoStartAppAndLockItOldPassword(password: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(autoStartAppAndLockItOldPassword = password)
    }

    fun getAutoStartAppAndLockItOldPassword(): String {
        return _packageActivityState.value.autoStartAppAndLockItOldPassword
    }

    fun setAutoStartAppAndLockItNewPassword(password: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(autoStartAppAndLockItNewPassword = password)
    }

    fun getAutoStartAppAndLockItNewPassword(): String {
        return _packageActivityState.value.autoStartAppAndLockItNewPassword
    }

    fun setShowRemoveTaskAutoStartAndLockingDialog(show: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(showRemoveTaskAutoStartAndLockingDialog = show)
    }

    fun getShowRemoveTaskAutoStartAndLockingDialog(): Boolean {
        return _packageActivityState.value.showRemoveTaskAutoStartAndLockingDialog
    }

    fun setTemporaryUnpinCheckBox(checked: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(temporaryUnpinCheckBox = checked)
    }

    fun getTemporaryUnpinCheckBox(): Boolean {
        return _packageActivityState.value.temporaryUnpinCheckBox
    }

    fun setPermanentUnpinCheckBox(checked: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(permanentUnpinCheckBox = checked)
    }

    fun getPermanentUnpinCheckBox(): Boolean {
        return _packageActivityState.value.permanentUnpinCheckBox
    }

    fun setRemoveTaskAutoStartAndLockingPassword(password: String) {
        _packageActivityState.value =
            _packageActivityState.value.copy(removeTaskAutoStartAndLockingPassword = password)
    }

    fun getRemoveTaskAutoStartAndLockingPassword(): String {
        return _packageActivityState.value.removeTaskAutoStartAndLockingPassword
    }

    fun setEnablePackageWhitelistingBlockPackageCheckBox(checked: Boolean) {
        _packageActivityState.value =
            _packageActivityState.value.copy(enablePackageWhitelistingBlockPackageCheckBox = checked)
    }

    fun getEnablePackageWhitelistingBlockPackageCheckBox(): Boolean {
        return _packageActivityState.value.enablePackageWhitelistingBlockPackageCheckBox
    }
}