/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.viewmodel

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.util.Log
import androidx.lifecycle.ViewModel
import com.eloview.sdk.accountManager.AccountManager
import com.eloview.sdk.networkManager.Network
import com.eloview.sdk.systemManager.ServiceConnectionCallback
import com.eloview.sdk.testapp.GlobalClass
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow

data class NetworkActivityState(
    var serviceConnected: Boolean = false,
    var output: String = "",
    var networkConfigurationWiFiLanToggle: Boolean = true, // true == wifi, false == lan
    var networkSecurityTypeSelected: String = "WPA",
    var ssidName: String = "",
    var alias: String = "",
    var password: String = "",
    var proxyHost: String = "",
    var proxyPort: String = "",
    var staticIp: String = "",
    var gateway: String = "",
    var prefix: String = "",
    var dns1: String = "",
    var dns2: String = "",
    var avoidPoorConnection: Boolean = true,
    var defaultExclude: Boolean = true,
    var showNetworkConfigurationDialog: Boolean = false,
    var showRemoveCertificateDialog: Boolean = false,
    var removeAlias: String = "",
    var ethState: Boolean = false,
)

class NetworkActivityViewModel(val applicationContext: Context, preview: Boolean = false) :
    ViewModel(),
    ServiceConnectionCallback {
    companion object {
        private const val TAG = "NetworkActivityViewModel"
        var isPreview: Boolean = false
    }

    val networkServiceManager = if (preview) {
        isPreview = true
        null
    } else {
        isPreview = false
        Network()
    }

    private val _networkActivityState = MutableStateFlow(NetworkActivityState())
    val networkActivityState: StateFlow<NetworkActivityState> = _networkActivityState.asStateFlow()

    val networkHandler = Handler(Looper.getMainLooper()) { msg ->
        val b = msg.data
        val key = msg.what.toString()
        var `val` = b.getString(key)
        when (msg.what) {
            Network.SET_NETWORK_RESPONSE_CODE -> Log.i(
                TAG,
                "SET_NETWORK_RESPONSE_CODE = $`val`"
            )

            Network.GET_NETWORK_RESPONSE_CODE -> {
                Log.i(TAG, "GET_NETWORK_RESPONSE_CODE = $`val`")
                `val` = "GET_NETWORK_RESPONSE_CODE \n\n$`val`"
            }

            Network.GET_WIFI_MAC_ADDRESS_CODE -> {
                Log.i(TAG, "GET_WIFI_MAC_ADDRESS_CODE = $`val`")
                `val` = "GET_WIFI_MAC_ADDRESS_CODE \n\n$`val`"
            }

            Network.GET_ETH_MAC_ADDRESS_CODE -> {
                Log.i(TAG, "GET_ETH_MAC_ADDRESS_CODE = $`val`")
                `val` = "GET_ETH_MAC_ADDRESS_CODE \n\n$`val`"
            }

            AccountManager.TOKEN_VERIFY_FAIL -> {
                Log.e(TAG, "TOKEN_VERIFY_FAIL = $`val`")
                `val` = "INVALID TOKEN\n$`val`"
            }

            AccountManager.TOKEN_VERIFY_SUCCESS -> {
                Log.i(TAG, "TOKEN_VERIFY_SUCCESS = $`val`")
                `val` = "TOKEN VERIFIED\n$`val`"
            }

            Network.SET_EST_CERTIFICATE_CODE -> {
                Log.i(TAG, "SET_EST_CERTIFICATE_CODE = $`val`")
                `val` = "SET_EST_SERVER_CODE\n$`val`"
            }

            Network.INSTALL_CERTIFICATE_CODE -> {
                Log.i(TAG, "INSTALL_CERTIFICATE_CODE = $`val`")
                `val` = "INSTALL_CERTIFICATE_CODE\n$`val`"
            }
        }
        if (`val` != null) {
            setOutput(`val`)
        }
        false
    }

    private fun setServiceConnected(serviceConnected: Boolean) {
        _networkActivityState.value =
            _networkActivityState.value.copy(serviceConnected = serviceConnected)
    }

    fun getServiceConnected(): Boolean {
        return _networkActivityState.value.serviceConnected
    }

    fun bindService() {
        networkServiceManager?.bindService(
            applicationContext,
            this,
            GlobalClass.globalViewModel.getAccessToken()
        )
    }

    fun unbindService() {
        networkServiceManager?.unbindService(applicationContext)
        setServiceConnected(false)
    }

    override fun onServiceConnected() {
        setServiceConnected(true)
        Log.d(TAG, "onServiceConnected: Service connected")
    }

    override fun onServiceDisconnected() {
        setServiceConnected(false)
        Log.d(TAG, "onServiceDisconnected: Service disconnected")
    }

    override fun onTokenVerifyFailed() {
        setServiceConnected(false)
        Log.e(TAG, "onTokenVerifyFailed: Token verify failed")
    }

    fun getNetworkConfigurationWiFiLanToggle(): Boolean {
        return _networkActivityState.value.networkConfigurationWiFiLanToggle
    }

    fun setNetworkConfigurationWiFiLanToggle(networkConfigurationWiFiLanToggle: Boolean) {
        _networkActivityState.value =
            _networkActivityState.value.copy(networkConfigurationWiFiLanToggle = networkConfigurationWiFiLanToggle)
    }

    fun getNetworkSecurityType(): String {
        return _networkActivityState.value.networkSecurityTypeSelected
    }

    fun setNetworkSecurityType(networkSecurityTypeSelected: String) {
        _networkActivityState.value =
            _networkActivityState.value.copy(networkSecurityTypeSelected = networkSecurityTypeSelected)
    }

    fun getSsidName(): String {
        return _networkActivityState.value.ssidName
    }

    fun setSsidName(ssidName: String) {
        _networkActivityState.value = _networkActivityState.value.copy(ssidName = ssidName)
    }

    fun getAlias(): String {
        return _networkActivityState.value.alias
    }

    fun setAlias(alias: String) {
        _networkActivityState.value = _networkActivityState.value.copy(alias = alias)
    }

    fun getPassword(): String {
        return _networkActivityState.value.password
    }

    fun setPassword(password: String) {
        _networkActivityState.value = _networkActivityState.value.copy(password = password)
    }

    fun getProxyHost(): String {
        return _networkActivityState.value.proxyHost
    }

    fun setProxyHost(proxyHost: String) {
        _networkActivityState.value = _networkActivityState.value.copy(proxyHost = proxyHost)
    }

    fun getProxyPort(): String {
        return _networkActivityState.value.proxyPort
    }

    fun setProxyPort(proxyPort: String) {
        _networkActivityState.value = _networkActivityState.value.copy(proxyPort = proxyPort)
    }

    fun getStaticIp(): String {
        return _networkActivityState.value.staticIp
    }

    fun setStaticIp(staticIp: String) {
        _networkActivityState.value = _networkActivityState.value.copy(staticIp = staticIp)
    }

    fun getGateway(): String {
        return _networkActivityState.value.gateway
    }

    fun setGateway(gateway: String) {
        _networkActivityState.value = _networkActivityState.value.copy(gateway = gateway)
    }

    fun getPrefix(): String {
        return _networkActivityState.value.prefix
    }

    fun setPrefix(prefix: String) {
        _networkActivityState.value = _networkActivityState.value.copy(prefix = prefix)
    }

    fun getDns1(): String {
        return _networkActivityState.value.dns1
    }

    fun setDns1(dns1: String) {
        _networkActivityState.value = _networkActivityState.value.copy(dns1 = dns1)
    }

    fun getDns2(): String {
        return _networkActivityState.value.dns2
    }

    fun setDns2(dns2: String) {
        _networkActivityState.value = _networkActivityState.value.copy(dns2 = dns2)
    }

    fun getAvoidPoorConnection(): Boolean {
        return _networkActivityState.value.avoidPoorConnection
    }

    fun setAvoidPoorConnection(avoidPoorConnection: Boolean) {
        _networkActivityState.value =
            _networkActivityState.value.copy(avoidPoorConnection = avoidPoorConnection)
    }

    fun getDefaultExclude(): Boolean {
        return _networkActivityState.value.defaultExclude
    }

    fun setDefaultExclude(defaultExclude: Boolean) {
        _networkActivityState.value =
            _networkActivityState.value.copy(defaultExclude = defaultExclude)
    }

    fun getShowNetworkConfigurationDialog(): Boolean {
        return _networkActivityState.value.showNetworkConfigurationDialog
    }

    fun setShowNetworkConfigurationDialog(showNetworkConfigurationDialog: Boolean) {
        _networkActivityState.value =
            _networkActivityState.value.copy(showNetworkConfigurationDialog = showNetworkConfigurationDialog)
    }

    fun getShowRemoveCertificateDialog(): Boolean {
        return _networkActivityState.value.showRemoveCertificateDialog
    }

    fun setShowRemoveCertificateDialog(showRemoveCertificateDialog: Boolean) {
        _networkActivityState.value =
            _networkActivityState.value.copy(showRemoveCertificateDialog = showRemoveCertificateDialog)
    }

    fun getRemoveAlias(): String {
        return _networkActivityState.value.removeAlias
    }

    fun setRemoveAlias(removeAlias: String) {
        _networkActivityState.value = _networkActivityState.value.copy(removeAlias = removeAlias)
    }

    fun setEthState(ethState: Boolean) {
        _networkActivityState.value = _networkActivityState.value.copy(ethState = ethState)
    }

    fun getEthState(): Boolean {
        return _networkActivityState.value.ethState
    }

    fun setOutput(output: String) {
        _networkActivityState.value = _networkActivityState.value.copy(output = output)
    }

    fun getOutput(): String {
        return _networkActivityState.value.output
    }

    fun isEthernetEnabled(): Boolean {
        try {
            if (networkServiceManager != null) {
                return networkServiceManager.ethState
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return false // Ethernet not found or not up
    }
}