/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.viewmodel

import android.content.Context
import android.util.Log
import androidx.lifecycle.ViewModel
import com.eloview.sdk.memoryManager.Memory
import com.eloview.sdk.systemManager.ServiceConnectionCallback
import com.eloview.sdk.testapp.GlobalClass
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow

data class MemoryActivityState(
    var serviceConnected: Boolean = false,
    var output: String = ""
)

class MemoryActivityViewModel(val applicationContext: Context, preview: Boolean = false) :
    ViewModel(),
    ServiceConnectionCallback {
    companion object {
        private const val TAG = "MemoryActivityViewModel"
        var isPreview: Boolean = false
    }

    val memoryServiceManager = if (preview) {
        isPreview = true
        null
    } else {
        isPreview = false
        Memory()
    }

    private val _memoryActivityState = MutableStateFlow(MemoryActivityState())
    val memoryActivityState: StateFlow<MemoryActivityState> = _memoryActivityState.asStateFlow()

    private fun setServiceConnected(serviceConnected: Boolean) {
        _memoryActivityState.value =
            _memoryActivityState.value.copy(serviceConnected = serviceConnected)
    }

    fun getServiceConnected(): Boolean {
        return _memoryActivityState.value.serviceConnected
    }

    fun bindService() {
        memoryServiceManager?.bindService(
            applicationContext,
            this,
            GlobalClass.globalViewModel.getAccessToken()
        )
    }

    fun unbindService() {
        memoryServiceManager?.unbindService(applicationContext)
    }

    override fun onServiceConnected() {
        setServiceConnected(true)
        Log.d(TAG, "onServiceConnected: Service connected")
    }

    override fun onServiceDisconnected() {
        setServiceConnected(false)
        Log.d(TAG, "onServiceDisconnected: Service disconnected")
    }

    override fun onTokenVerifyFailed() {
        setServiceConnected(false)
        Log.e(TAG, "onTokenVerifyFailed: Token verify failed")
    }

    fun setOutput(output: String) {
        _memoryActivityState.value = _memoryActivityState.value.copy(output = output)
    }

    fun getOutput(): String {
        return _memoryActivityState.value.output
    }
}