/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.viewmodel

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.util.Log
import androidx.lifecycle.ViewModel
import com.eloview.sdk.infoManager.Info
import com.eloview.sdk.systemManager.ServiceConnectionCallback
import com.eloview.sdk.testapp.GlobalClass
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow

data class InfoActivityState(
    var serviceConnected: Boolean = false,
    var output: String = "",
)

class InfoActivityViewModel(val applicationContext: Context, preview: Boolean = false) :
    ViewModel(),
    ServiceConnectionCallback {
    companion object {
        private const val TAG = "InfoActivityViewModel"
        var isPreview: Boolean = false
    }

    val infoServiceManager = if (preview) {
        isPreview = true
        null
    } else {
        isPreview = false
        Info()
    }

    private val _infoActivityState = MutableStateFlow(InfoActivityState())
    val infoActivityState: StateFlow<InfoActivityState> = _infoActivityState.asStateFlow()

    private fun setServiceConnected(serviceConnected: Boolean) {
        _infoActivityState.value =
            _infoActivityState.value.copy(serviceConnected = serviceConnected)
    }

    fun getServiceConnected(): Boolean {
        return _infoActivityState.value.serviceConnected
    }

    fun bindService() {
        infoServiceManager?.bindService(
            applicationContext,
            this,
            GlobalClass.globalViewModel.getAccessToken()
        )
    }

    fun unbindService() {
        infoServiceManager?.unbindService(applicationContext)
    }

    override fun onServiceConnected() {
        setServiceConnected(true)
        Log.d(TAG, "onServiceConnected: Service connected")
    }

    override fun onServiceDisconnected() {
        setServiceConnected(false)
        Log.d(TAG, "onServiceDisconnected: Service disconnected")
    }

    override fun onTokenVerifyFailed() {
        setServiceConnected(false)
        Log.e(TAG, "onTokenVerifyFailed: Token verify failed")
    }

    val mDeviceInfoReceiver: BroadcastReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            Log.v(TAG, " onReceive DeviceInfoReceiver called")
            val outputStr = StringBuilder()
            val bundle = intent.extras
            if (bundle != null) {
                val keys = bundle.keySet()
                val it: Iterator<String> = keys.iterator()
                Log.e(TAG, "Dumping Intent start")
                while (it.hasNext()) {
                    val key = it.next()
                    Log.e(TAG, "[" + key + "=" + bundle.getString(key) + "]")
                    outputStr.append("mDeviceInfoReceiver key:= ").append(key).append(" val:=")
                        .append(
                            bundle.getString(key)
                        ).append("\n")
                }
                Log.e(TAG, "Dumping Intent end")
            }
            setOutput(outputStr.toString())
        }
    }

    fun setOutput(output: String) {
        _infoActivityState.value = _infoActivityState.value.copy(output = output)
    }

    fun getOutput(): String {
        return _infoActivityState.value.output
    }

}