/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.viewmodel

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.util.Log
import androidx.lifecycle.ViewModel
import com.eloview.sdk.accountManager.AccountManager
import com.eloview.sdk.eloview.EloView
import com.eloview.sdk.systemManager.ServiceConnectionCallback
import com.eloview.sdk.testapp.GlobalClass
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow

data class EloActivityState(
    var output: String = "",
    var serviceConnected: Boolean = false,
    var showOfflinePlayContentDialog: Boolean = false,
    var offlinePlayContentPackageName: String = "",
)

class EloActivityViewModel(val applicationContext: Context, preview: Boolean = false) :
    ViewModel(),
    ServiceConnectionCallback {
    companion object {
        private const val TAG = "EloActivityViewModel"
        var isPreview: Boolean = false
        const val QS_BT_WIFI_ENABLE = 1;
        const val QS_BT_WIFI_DISABLE = 2;
        const val QS_BT_WIFI_ALREADY_ENABLE = 3;
        const val QS_BT_WIFI_ALREADY_DISABLE = 4;
    }

    val eloviewServiceManager = if (preview) {
        isPreview = true
        null
    } else {
        isPreview = false
        EloView()
    }

    val eloViewHandler: Handler = Handler(Looper.getMainLooper()) { msg ->
        val b = msg.data
        val key = msg.what.toString()
        var `val` = b.getString(key)
        when (msg.what) {
            AccountManager.TOKEN_VERIFY_FAIL -> {
                `val` = "INVALID TOKEN\n$`val`"
                setOutput(`val`)
                unbindService()
            }

            AccountManager.TOKEN_VERIFY_SUCCESS -> {
                `val` = "VALID TOKEN\n$`val`"
                setOutput(`val`)
            }

            EloView.OFFLINE_PLAY_CONTENT_CODE, EloView.DEVICE_CLOUD_STATUS_CODE, EloView.QS_WIFI_RESPONSE_CODE, EloView.QS_BT_RESPONSE_CODE, EloView.GET_PLAY_CONTENT, EloView.GET_PLAY_CONTENT_STATUS, EloView.GET_DEVICE_ONLINE_STATUS -> `val`?.let {
                setOutput(
                    it
                )
            }

            else -> setOutput("Action Not Supported")
        }
        false
    }

    private val _eloActivityState = MutableStateFlow(EloActivityState())
    val eloActivityState: StateFlow<EloActivityState> = _eloActivityState.asStateFlow()

    fun setOutput(output: String) {
        _eloActivityState.value = _eloActivityState.value.copy(output = output)
    }

    fun getOutput(): String {
        return _eloActivityState.value.output
    }

    fun setShowOfflinePlayContentDialog(showOfflinePlayContentDialog: Boolean) {
        _eloActivityState.value =
            _eloActivityState.value.copy(showOfflinePlayContentDialog = showOfflinePlayContentDialog)
    }

    fun getShowOfflinePlayContentDialog(): Boolean {
        return _eloActivityState.value.showOfflinePlayContentDialog
    }

    fun setOfflinePlayContentPackageName(offlinePlayContentPackageName: String) {
        _eloActivityState.value =
            _eloActivityState.value.copy(offlinePlayContentPackageName = offlinePlayContentPackageName)
    }

    fun getOfflinePlayContentPackageName(): String {
        return _eloActivityState.value.offlinePlayContentPackageName
    }

    fun bindService() {
        eloviewServiceManager?.bindService(
            applicationContext,
            this,
            GlobalClass.globalViewModel.getAccessToken()
        )
    }

    private fun unbindService() {
        try {
            eloviewServiceManager?.unbindService(applicationContext)
        } catch (e: Exception) {
            Log.w(TAG, "unbindService: $e")
        }
    }

    private fun setServiceConnected(serviceConnected: Boolean) {
        _eloActivityState.value =
            _eloActivityState.value.copy(serviceConnected = serviceConnected)
    }

    public fun getServiceConnected(): Boolean {
        return _eloActivityState.value.serviceConnected
    }

    override fun onServiceConnected() {
        setServiceConnected(true)
        Log.d(TAG, "onServiceConnected: Service connected")
    }

    override fun onServiceDisconnected() {
        setServiceConnected(false)
        Log.d(TAG, "onServiceDisconnected: Service disconnected")
    }

    override fun onTokenVerifyFailed() {
        setServiceConnected(false)
        Log.e(TAG, "onTokenVerifyFailed: Token verify failed")
    }

    fun bluetoothStatus(status: Int?): String {
        Log.d(TAG, "bluetoothStatus: isQSBTEnabled $status")
        return when (status) {
            QS_BT_WIFI_ALREADY_ENABLE -> "Quick settings bluetooth option is already enabled"
            QS_BT_WIFI_ALREADY_DISABLE -> "Quick settings bluetooth option is already disabled"
            QS_BT_WIFI_ENABLE -> "Quick settings Bluetooth enabled"
            QS_BT_WIFI_DISABLE -> "Quick settings Bluetooth disabled"
            else -> "Some error occurred"
        }
    }

    fun wifiStatus(status: Int?): String {
        Log.d(TAG, "wifiStatus: isQSWiFiEnabled $status")
        return when (status) {
            QS_BT_WIFI_ALREADY_ENABLE -> "Quick settings Wi-Fi option is already enabled"
            QS_BT_WIFI_ALREADY_DISABLE -> "Quick settings Wi-Fi option is already disabled"
            QS_BT_WIFI_ENABLE -> "Quick settings Wi-Fi enabled"
            QS_BT_WIFI_DISABLE -> "Quick settings Wi-Fi disabled"
            else -> "Some error occurred"
        }
    }
}