/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.viewmodel

import android.app.Activity
import android.content.Context
import android.content.SharedPreferences
import android.os.Handler
import android.os.Looper
import android.os.Message
import android.provider.Settings
import android.util.Log
import android.widget.Toast
import androidx.compose.runtime.State
import androidx.compose.runtime.mutableIntStateOf
import androidx.lifecycle.ViewModel
import com.eloview.sdk.accountManager.AccountManager
import com.eloview.sdk.testapp.AccountManagerActivity
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.R
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import java.util.Properties

data class AccountManagerState(
    var output: String = "",
    var accessToken: String = "",
    var email: String = "",
    var password: String = "",
    var clientId: String = "",
    var secretKey: String = "",
    var emailPass: Boolean = false,
    var clientSecret: Boolean = false,
    var verifyFromFile: Boolean = false,
    var tokenFromApi: Boolean = false,
    var isResealed: Boolean = false,
    var isPolaris: Boolean = false,
    var textEmailEnabled: Boolean = true,
    var textPasswordEnabled: Boolean = true,
    var buttonGetTokenEnabled: Boolean = true,
    var buttonVerifyTokenEnabled: Boolean = true,
    var textClientIdEnabled: Boolean = true,
    var textSecretKeyEnabled: Boolean = true,
    var buttonGetAndVerifyTokenEnabled: Boolean = true,
    var buttonLoadAndVerifyTokenEnabled: Boolean = true,
    var buttonLogoutEnabled: Boolean = false,
    var textNoteEnabled: Boolean = false,
    var spinnerEnabled: Boolean = true
)

class AccountManagerViewModel(private val applicationContext: Context, preview: Boolean = false) :
    ViewModel() {

    private val sharedPreferences: SharedPreferences =
        applicationContext.getSharedPreferences("AppPreferences", Context.MODE_PRIVATE)

    // State for selected index
    private val _selectedIndex = mutableIntStateOf(getSavedSelectedIndex())
    val selectedIndex: State<Int> = _selectedIndex

    companion object {
        private const val TAG = "AccountManagerViewModel"
    }

    lateinit var accessTokenHandler: Handler
    private val _accountManagerState = MutableStateFlow(AccountManagerState())
    val deviceManagerState: StateFlow<AccountManagerState> = _accountManagerState.asStateFlow()
    fun clearOutput() {
        _accountManagerState.update { it.copy(output = "") }
    }

    fun setOutput(output: String) {
        _accountManagerState.update { it.copy(output = output) }
    }

    fun setEmail(email: String) {
        _accountManagerState.update { it.copy(email = email) }
        GlobalClass.globalViewModel.setEmail(email)
    }

    fun setPassword(password: String) {
        _accountManagerState.update { it.copy(password = password) }
        GlobalClass.globalViewModel.setPassword(password)
    }

    fun setClientId(clientId: String) {
        _accountManagerState.update { it.copy(clientId = clientId) }
        GlobalClass.globalViewModel.setClientId(clientId)
    }

    fun setSecretKey(secretKey: String) {
        _accountManagerState.update { it.copy(secretKey = secretKey) }
        GlobalClass.globalViewModel.setSecretKey(secretKey)
    }

    private fun setResealed(isResealed: Boolean) {
        _accountManagerState.update { it.copy(isResealed = isResealed) }
        GlobalClass.globalViewModel.setIsResealed(isResealed)
    }

    private fun setVerifyFromFile(verifyFromFile: Boolean) {
        _accountManagerState.update { it.copy(verifyFromFile = verifyFromFile) }
    }

    internal fun setAccessToken(accessToken: String) {
        _accountManagerState.update { it.copy(accessToken = accessToken) }
        GlobalClass.globalViewModel.setAccessToken(accessToken)
    }

    internal fun setEmailPass(emailPass: Boolean) {
        _accountManagerState.update { it.copy(emailPass = emailPass) }
        GlobalClass.globalViewModel.setEmailPass(emailPass)
    }

    internal fun setClientSecret(clientSecret: Boolean) {
        _accountManagerState.update { it.copy(clientSecret = clientSecret) }
        GlobalClass.globalViewModel.setClientSecret(clientSecret)
    }

    internal fun setTokenFromApi(tokenFromApi: Boolean) {
        _accountManagerState.update { it.copy(tokenFromApi = tokenFromApi) }
        GlobalClass.globalViewModel.setTokenFromApi(tokenFromApi)
    }

    private fun setIsPolaris(isPolaris: Boolean) {
        _accountManagerState.update { it.copy(isPolaris = isPolaris) }
        GlobalClass.globalViewModel.setIsPolaris(isPolaris)
    }

    init {
        accessTokenHandler = Handler(Looper.getMainLooper()) { msg ->
            val b = msg.data
            val key = msg.what.toString()
            var `val` = b.getString(key, "")
            Log.d(TAG, "handleMessage: $`val`")
            when (msg.what) {
                AccountManager.ACCESS_TOKEN_VALID -> {
                    setAccessToken(`val`)
                    setEmailPass(true)
                    setClientSecret(false)
                    setTokenFromApi(false)
                    Log.d(
                        TAG,
                        "handleMessage: accessToken >> " + _accountManagerState.value.accessToken
                    )
                    if (_accountManagerState.value.verifyFromFile) {
                        setVerifyFromFile(false)
                        AccountManager.instance.verifyEloToken(
                            applicationContext,
                            _accountManagerState.value.accessToken,
                            accessTokenHandler
                        )
                    } else {
                        setOutput(`val`)
                    }
                }

                AccountManager.TOKEN_VERIFY_FAIL -> {
                    Log.d(TAG, "handleMessage: Invalid Token")
                    `val` = "INVALID TOKEN\n$`val`"
                    setOutput(`val`)
                }

                AccountManager.ACCESS_TOKEN_INVALID -> {
                    Log.d(TAG, "handleMessage: Failed to get aN access accessToken")
                    setOutput(`val`)
                }

                AccountManager.TOKEN_VERIFY_SUCCESS -> {
                    Log.d(TAG, "handleMessage: accessToken >> $`val`")
                    AccountManagerActivity.isTokenVerified = true
                    GlobalClass.globalViewModel.setAuthorized(true)
                    setOutput(`val`)
                }

                AccountManager.OAUTH_TOKEN_VALID -> {
                    setClientSecret(true)
                    setEmailPass(false)
                    setTokenFromApi(false)
                    Log.d(TAG, "handleMessage: oAuthToken >> $`val`")
                    setAccessToken(`val`)
                    AccountManager.instance.verifyEloToken(
                        applicationContext,
                        _accountManagerState.value.accessToken,
                        accessTokenHandler
                    )
                }

                AccountManager.OAUTH_TOKEN_INVALID -> {
                    Log.d(TAG, "handleMessage: Failed to get an oAuth accessToken")
                    setOutput(`val`)
                }

                AccountManager.API_ACCESS_TOKEN_VALID -> {
                    setClientSecret(false)
                    setEmailPass(false)
                    setTokenFromApi(true)
                    Log.d(TAG, "handleMessage: ApiAccessToken >> $`val`")
                    setOutput(`val`)
                    setAccessToken(`val`)
                    AccountManager.instance.verifyEloToken(
                        applicationContext,
                        _accountManagerState.value.accessToken,
                        accessTokenHandler
                    )
                }

                AccountManager.API_ACCESS_TOKEN_INVALID -> Log.d(
                    TAG,
                    "handleMessage: Failed to get ApiAccessToken"
                )

                AccountManager.GENERIC_ERROR -> setOutput(`val`)
            }
            true
        }
        if (!preview) {
            val isResealed = (Settings.Secure.getString(applicationContext.contentResolver, "elo_is_reseal").lowercase() == "false").not()
            setResealed(isResealed)
            if (isResealed) {
                _accountManagerState.update { it.copy(spinnerEnabled = false) }
                _accountManagerState.update { it.copy(textNoteEnabled = false) }
            }
        }
    }

    private val oAuthHandler: Handler = Handler(Looper.getMainLooper()) { msg: Message ->
        val b = msg.data
        val key = msg.what.toString()
        val `val` = b.getString(key, "")
        Log.d(TAG, "handleMessage: $`val`")
        when (msg.what) {
            AccountManager.SETUP_OAUTH_SUCCESS -> {
                _accountManagerState.update { it.copy(buttonGetTokenEnabled = true) }
                _accountManagerState.update { it.copy(buttonLogoutEnabled = true) }
                Log.d(TAG, "handleMessage: SETUP_OAUTH_SUCCESS$`val`")
            }

            AccountManager.SETUP_OAUTH_ERROR -> {
                _accountManagerState.update { it.copy(buttonGetTokenEnabled = true) }
                _accountManagerState.update { it.copy(buttonLogoutEnabled = false) }
                Log.d(TAG, "handleMessage: SETUP_OAUTH_ERROR$`val`")
            }
        }
        true
    }

    fun loadEnvironmentProperties(index: Int, selectedItem: String, activity: Activity) {
        val prop = when (index) {
            1 -> "sdk_qa.properties"
            2 -> "sdk_sfp.properties"
            3 -> "sdk_polaris_elodev.properties"
            4 -> "sdk_polaris_stg.properties"
            5 -> "sdk_polaris_production.properties"
            6 -> "sdk_polaris_qa.properties"
            else -> "sdk.properties"
        }
        if (prop != AccountManager.getProperty()) {
            _accountManagerState.update { it.copy(accessToken = "") }
        }
        if (selectedItem.lowercase().contains("polaris")) {
            setIsPolaris(true)
            AccountManager.instance.selectEnvironment(
                activity.applicationContext,
                prop,
                false
            )
            _accountManagerState.update { it.copy(textEmailEnabled = false) }
            _accountManagerState.update { it.copy(textPasswordEnabled = false) }
            _accountManagerState.update { it.copy(buttonLogoutEnabled = false) }
            _accountManagerState.update { it.copy(buttonGetTokenEnabled = false) }
            _accountManagerState.update { it.copy(buttonGetTokenEnabled = true) }
            if (_accountManagerState.value.isResealed) {
                _accountManagerState.update { it.copy(spinnerEnabled = false) }
                _accountManagerState.update { it.copy(textNoteEnabled = false) }
            }
            Toast.makeText(
                activity.applicationContext,
                R.string.configurin_env,
                Toast.LENGTH_SHORT
            ).show()
            AccountManager.instance.configureOAuthHostedUI(
                activity,
                oAuthHandler
            )
        } else {
            setIsPolaris(false)
            _accountManagerState.update { it.copy(textEmailEnabled = true) }
            _accountManagerState.update { it.copy(textPasswordEnabled = true) }
            _accountManagerState.update { it.copy(buttonLogoutEnabled = false) }
            _accountManagerState.update { it.copy(buttonGetTokenEnabled = true) }
            _accountManagerState.update { it.copy(textNoteEnabled = false) }
            if (_accountManagerState.value.isResealed) {
                _accountManagerState.update { it.copy(spinnerEnabled = false) }
            }
            AccountManager.instance.selectEnvironment(
                activity.applicationContext,
                prop,
                false
            )
        }
    }

    fun onSpinnerItemSelection(index: Int, list: List<String>, activity: Activity) {

        saveSelectedIndex(index)
        // Update the selected index
        _selectedIndex.value = index

        val selectedItem: String = list[index]
        Toast.makeText(
            activity.applicationContext,
            "$selectedItem Selected", Toast.LENGTH_LONG
        ).show()
        Log.d(TAG, "$selectedItem selected.")
        loadEnvironmentProperties(index, selectedItem, activity)
    }

    fun loadAndVerifyTokenFromFile() {
        setVerifyFromFile(true)
        val credentials = Properties()
        try {
            credentials.load(
                applicationContext.assets
                    .open("com.eloview.sdk.testapp_jwt.prop")
            )
        } catch (e: Exception) {
            e.printStackTrace()
        }
        setAccessToken(credentials.getProperty("jwt"))
        setEmailPass(false)
        setClientSecret(false)
        setTokenFromApi(false)
        AccountManager.instance.verifyEloToken(
            applicationContext,
            _accountManagerState.value.accessToken,
            accessTokenHandler
        )
    }

    // Save the selected index to SharedPreferences
    private fun saveSelectedIndex(index: Int) {
        sharedPreferences.edit().putInt("selected_index", index).apply()
    }

    // Retrieve the saved selected index from SharedPreferences
    private fun getSavedSelectedIndex(): Int {
        return sharedPreferences.getInt("selected_index", 0)  // Default value is 0 if not found
    }
}