/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.systemActivity

import android.app.Activity
import android.app.DownloadManager
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.Service
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.content.res.Configuration
import android.net.Uri
import android.os.Environment
import android.os.IBinder
import android.os.RemoteException
import android.provider.Settings
import android.util.Log
import android.view.KeyEvent
import android.widget.Toast
import androidx.activity.result.ActivityResult
import androidx.activity.result.ActivityResultLauncher
import androidx.core.app.NotificationCompat
import androidx.core.content.ContextCompat
import androidx.lifecycle.viewModelScope
import com.eloview.sdk.testapp.R
import com.eloview.sdk.testapp.common.FileUtils
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.viewmodel.SystemActivityViewModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.io.File
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import androidx.core.net.toUri


class SystemActivityOnClickListeners {
    companion object {
        private var fileByteArray: ByteArray? = null
        private var zipFile: File? = null
        private const val TAG = "SystemActivityOnClickListeners"
        fun onScreenShotClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val fileName = systemActivityViewModel.getScreenShotPath()

                // Perform the screenshot capture on a background thread
                CoroutineScope(Dispatchers.IO).launch {
                    val success =
                        systemActivityViewModel.systemServiceManager?.captureScreenShot(fileName) == true

                    // Switch to the main thread to update the UI after the operation is complete
                    withContext(Dispatchers.Main) {
                        if (success) {
                            systemActivityViewModel.setOutput("Screenshot saved as $fileName")
                        } else {
                            systemActivityViewModel.setOutput("Failed to save screenshot")
                        }
                    }
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetOrientationNonBackpackClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val selection =
                    systemActivityViewModel.getSelectedNonBPOrientation()
                var orientation = "-2"
                when (selection) {
                    "auto-rotate" -> {
                        orientation = "-1"
                    }

                    "landscape" -> {
                        orientation =
                            if (Util.isPerformanceDevice() && Util.is10inch() || Util.isHandHeld()) {
                                "90"
                            } else {
                                "0"
                            }
                    }

                    "portrait" -> {
                        orientation = if (Util.isPerformanceDevice()) {
                            if (Util.is10inch() || Util.isHandHeld()) {
                                "0"
                            } else if (Util.is7inch() || Util.is15inch() || Util.is22inch()) {
                                "90"
                            } else {
                                "270"
                            }
                        } else {
                            "90"
                        }
                    }

                    "landscape-reverse" -> {
                        orientation =
                            if (Util.isPerformanceDevice() && Util.is10inch() || Util.isHandHeld()) {
                                "270"
                            } else {
                                "180"
                            }
                    }

                    "portrait-reverse" -> {
                        orientation = if (Util.isPerformanceDevice()) {
                            if (Util.is10inch() || Util.isHandHeld()) {
                                "180"
                            } else if (Util.is7inch() || Util.is15inch() || Util.is22inch()) {
                                "270"
                            } else {
                                "90"
                            }
                        } else {
                            "270"
                        }
                    }
                }
                if (orientation != "-2") {
                    if (systemActivityViewModel.systemServiceManager?.setOrientation(
                            orientation,
                            systemActivityViewModel.getNoRebootOrientationChecked()
                        ) == true
                    ) {
                        systemActivityViewModel.setOutput("Orientation set to $orientation")
                    } else {
                        systemActivityViewModel.setOutput("Failed to set orientation")
                    }
                } else {
                    Log.e(TAG, "Invalid orientation value")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetOrientationBackpackClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val selection = systemActivityViewModel.getSelectedOrientation()
                var orientation = "-1"
                when (selection) {
                    "landscape (0)" -> {
                        orientation = "0"
                    }

                    "portrait (90)" -> {
                        orientation = "90"
                    }

                    "landscape-reverse (180)" -> {
                        orientation = "180"
                    }

                    "portrait-reverse (270)" -> {
                        orientation = "270"
                    }
                }
                if (orientation != "-1") {
                    if (systemActivityViewModel.systemServiceManager?.setOrientation(
                            orientation,
                            systemActivityViewModel.getNoRebootOrientationChecked()
                        ) == true
                    ) {
                        systemActivityViewModel.setOutput("Orientation set to $orientation")
                    } else {
                        systemActivityViewModel.setOutput("Failed to set orientation")
                    }
                } else {
                    Log.e(TAG, "Invalid orientation value")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetOrientationClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val orientation = systemActivityViewModel.systemServiceManager?.orientation
                systemActivityViewModel.setOutput("Get Orientation: $orientation")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetExternalOrientationClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val selection = systemActivityViewModel.getSelectedExternalOrientation()
                var orientation = "-1"
                when (selection) {
                    "0" -> {
                        orientation = "0"
                    }

                    "90" -> {
                        orientation = "90"
                    }

                    "180" -> {
                        orientation = "180"
                    }

                    "270" -> {
                        orientation = "270"
                    }
                }
                if (orientation != "-1") {
                    if (systemActivityViewModel.systemServiceManager?.setExternalScreenOrientation(
                            orientation
                        ) == true
                    ) {
                        systemActivityViewModel.setOutput("External screen orientation set to $orientation")
                    } else {
                        systemActivityViewModel.setOutput("Failed to set external screen orientation")
                    }
                } else {
                    Log.e(TAG, "Invalid orientation value")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetExternalOrientationClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val orientation =
                    systemActivityViewModel.systemServiceManager?.externalScreenOrientation
                systemActivityViewModel.setOutput("Get External Orientation: $orientation")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onFactoryResetClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.factoryReset(true)
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSilentRebootClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.silentReboot()
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSilentShutdownClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val delayInSeconds = systemActivityViewModel.getSilentShutdownDelay()
                systemActivityViewModel.setOutput("Device will shutdown after $delayInSeconds seconds")
                systemActivityViewModel.systemServiceManager?.silentShutDown(delayInSeconds)
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetReservedLogSpaceClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val reservedLogSpace =
                    systemActivityViewModel.systemServiceManager?.reservedLogSpace
                systemActivityViewModel.setOutput("Reserved Log Space: $reservedLogSpace")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetReservedLogSpaceClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val reservedLogSpaceIndex = systemActivityViewModel.getReservedLogSpaceIndex()
                val list = listOf("144", "256", "512", "756", "1", "2")
                val actualList = listOf(144, 256, 512, 756, 1024, 2048)
                val value = list.getOrElse(reservedLogSpaceIndex) { "144" }
                val actualValue = actualList.getOrElse(reservedLogSpaceIndex) { 144 }
                var currentLogSpace =
                    systemActivityViewModel.systemServiceManager?.readSystemProperty("persist.sys.reserved_log_space")
                if (currentLogSpace.isNullOrEmpty()) {
                    currentLogSpace = "144"
                }
                if (value == currentLogSpace) {
                    systemActivityViewModel.setOutput("Reserved Log Space is already set to $value")
                    return
                }
                if (Util.isSpaceAvailable() > (actualValue + (actualValue / 6))) {
                    if (actualValue < currentLogSpace.toInt()) {
                        systemActivityViewModel.systemServiceManager?.reservedLogSpace = actualValue
                        systemActivityViewModel.setOutput("Reserved Log Space set to lower value $actualValue. Deleted existing log files.")
                    } else {
                        systemActivityViewModel.systemServiceManager?.reservedLogSpace = actualValue
                        systemActivityViewModel.setOutput("Reserved Log Space set to $actualValue")
                    }
                } else {
                    systemActivityViewModel.setOutput("Not enough space available to set reserved log space to $value")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onLocationEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setLocationEnabled(enable) == true) {
                    systemActivityViewModel.setOutput("Location is now $enable")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Location")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onHomeButtonEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setHomeButtonEnabled(enable)
                systemActivityViewModel.setOutput("Home Button is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onRecentButtonEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setRecentButtonEnabled(enable)
                systemActivityViewModel.setOutput("Recent Button is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }
        fun onWifiScanningEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable : Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setWifiScanning(enable)
                systemActivityViewModel.setOutput("Wifi Scanning is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }
        fun onBluetoothScanningEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setBluetoothScanning(enable)
                systemActivityViewModel.setOutput("Bluetooth Scanning is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onTouchSoundEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setTouchSoundEnabled(enable) == true) {
                    systemActivityViewModel.setOutput("Touch Sound is now $enable")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Touch Sound")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onVolumePanelShowMoreEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setVolumePanelShowMoreEnabled(enable)
                systemActivityViewModel.setOutput("Volume Panel Show More is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onTextEditorAssistMenuEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setTextEditorAssistMenuEnabled(enable)
                systemActivityViewModel.setOutput("Text Editor Assist Menu is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onPowerButtonEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setPowerButtonEnabled(enable)
                systemActivityViewModel.setOutput("Power Button is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onNavigationBarEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                var stdAndroid: String? =
                    systemActivityViewModel.systemServiceManager?.readSystemProperty("persist.sys.stdandroid")
                var deviceMode: String? =
                    systemActivityViewModel.systemServiceManager?.readSystemProperty("persist.sys.devicemode")
                if (stdAndroid == null) {
                    stdAndroid = ""
                }
                if (deviceMode == null) {
                    deviceMode = ""
                }
                if (stdAndroid.equals("true", ignoreCase = true) && !deviceMode.equals(
                        "connect",
                        ignoreCase = true
                    )
                ) {
                    val currentState = systemActivityViewModel.systemServiceManager?.navigationBarEnabled
                    if (currentState == enable) {
                        val alreadyState = if (enable) "already enabled" else "already disabled"
                        systemActivityViewModel.setOutput("Navigation bar already set to the selected value")
                        Toast.makeText(
                            systemActivityViewModel.applicationContext,
                            "Navigation Bar is $alreadyState",
                            Toast.LENGTH_SHORT
                        ).show()
                    } else {
                        systemActivityViewModel.systemServiceManager?.navigationBarEnabled = enable
                        systemActivityViewModel.setOutput("Navigation Bar is now $enable")
                        Toast.makeText(
                            systemActivityViewModel.applicationContext,
                            "Navigation Bar is now $enable",
                            Toast.LENGTH_SHORT
                        ).show()
                    }
                } else {
                    systemActivityViewModel.setOutput("Cannot set Navigation bar through the SDK in any of the EloView modes.")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onBCREnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setBarcodeReaderEnabled(enable)
                systemActivityViewModel.setBarcodeReaderEnabled(enable)
                systemActivityViewModel.setOutput("BCR is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onBCRTriggerClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.getBarcodeReaderEnabled()) {
                    systemActivityViewModel.systemServiceManager?.setScanBarcodeReaderEnabled(enable)
                    systemActivityViewModel.setOutput("BCR Scan is now $enable")
                    if (enable) {
                        systemActivityViewModel.setBarcodeReaderEnabled(false)
                    }
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetStatusBarEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                var stdAndroid: String? =
                    systemActivityViewModel.systemServiceManager?.readSystemProperty("persist.sys.stdandroid")
                if (stdAndroid == null) {
                    stdAndroid = ""
                }

                if (stdAndroid.equals("true", ignoreCase = true)) {
                    if (systemActivityViewModel.systemServiceManager?.setStatusBarEnabled(enable) == true) {
                        systemActivityViewModel.setOutput("Status Bar is now $enable")
                        Toast.makeText(
                            systemActivityViewModel.applicationContext,
                            "Status Bar is now $enable",
                            Toast.LENGTH_SHORT
                        ).show()
                    } else {
                        systemActivityViewModel.setOutput("Failed to set Status Bar")
                    }
                } else {
                    systemActivityViewModel.setOutput("Cannot set Status bar through the SDK in any of the EloView modes except connect mode.")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetLockStatusBarEnable(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val connect = Settings.Secure.getString(systemActivityViewModel.applicationContext.contentResolver, "elo_device_mode")
                val stdAndroid = systemActivityViewModel.systemServiceManager?.readSystemProperty("persist.sys.stdandroid").equals("true")
                if (stdAndroid || (connect != null && connect == "connect")) {
                    systemActivityViewModel.systemServiceManager?.setLockStatusBarEnabled(enable)
                    systemActivityViewModel.setOutput("Status Bar with Swipe Lock is now $enable")
                } else {
                    systemActivityViewModel.setOutput("Can only set Status bar with Swipe Lock in Standard Android Home mode or EloView Connect mode.")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetStatusBarVisibilityEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setStatusBarVisibilityEnabled(
                        enable
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("Status Bar Visibility is now $enable")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Status Bar Visibility")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetShowPasswordEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setPasswordEnabled(enable)
                systemActivityViewModel.setOutput("Show Password is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGrantAutoAllAppPermissionEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val isEloviewMode =
                    "false" == systemActivityViewModel.systemServiceManager?.readSystemProperty("persist.sys.stdandroid")
                if (isEloviewMode) {
                    systemActivityViewModel.setOutput("Auto Grant All App Permission is always enabled in EloView")
                } else {
                    systemActivityViewModel.systemServiceManager?.grantAllAppPermissionEnabled =
                        enable
                    systemActivityViewModel.setOutput("Auto Grant All App Permission is now $enable")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGrantAutoUsbPermissionEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.autoAllowUsbPermissionEnabled = enable
                systemActivityViewModel.setOutput("Auto Grant USB Permission is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetShareTextSelectionEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setShowTextEditorShareMenuEnabled(
                    enable
                )
                systemActivityViewModel.setOutput("Share Text Selection is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetWebViewHardwareAccessEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setPeripheralConnectEnabled(enable)
                systemActivityViewModel.setOutput("WebView Hardware Access is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetStarWebViewHardwareAccessEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setStarPeripheralConnectEnabled(enable)
                systemActivityViewModel.setOutput("Star WebView Hardware Access is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetAutoIdleModeEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setIdleModeEnabled(enable)
                systemActivityViewModel.setOutput("Auto Idle Mode is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetIdleModeTileEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setIdleModeTileEnabled(enable) == -1) {
                    systemActivityViewModel.setOutput(
                        systemActivityViewModel.applicationContext.getString(R.string.idle_tile_not_supported)
                    )
                } else {
                    systemActivityViewModel.setOutput("Idle Mode Tile is now $enable")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetMicrophoneEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setMicrophoneMuteEnabled(!enable)
                systemActivityViewModel.setOutput("Microphone is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetInstallPermissionToAllAppsEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setInstallAllAppPermissionEnabled(
                        enable
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("Install Permission to All Apps is now $enable")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Install Permission to All Apps")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetInstallPermissionToSpecificAppsEnableClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val enable =
                    systemActivityViewModel.getSelectedInstallPermissionForSpecificAppEnabled()
                val packageName =
                    systemActivityViewModel.getSelectedInstallPermissionForSpecificAppPackageName()
                if (packageName.isEmpty() || !Util.isPackageInstalled(
                        systemActivityViewModel.applicationContext,
                        packageName
                    )
                ) {
                    systemActivityViewModel.setOutput("$packageName : Invalid package name")
                    return
                }
                if (systemActivityViewModel.systemServiceManager?.setInstallPermissionToSpecificApp(
                        packageName,
                        enable
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("Install Permission to Specific App $packageName is now $enable")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Install Permission to Specific App $packageName")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetAccessibilityServiceEnableClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val enable = systemActivityViewModel.getSetAccessibilityServiceEnabled()
                val componentName = systemActivityViewModel.getAccessibilityServiceComponentName()
                // split "componentName" into package name and class name
                val packageName = componentName.substringBefore("/")
                if (packageName.isEmpty() || !Util.isPackageInstalled(
                        systemActivityViewModel.applicationContext,
                        packageName
                    )
                ) {
                    systemActivityViewModel.setOutput("Package Name is Empty or Invalid")
                    return
                }
                systemActivityViewModel.systemServiceManager?.setAccessibilityService(
                    componentName,
                    enable
                )
                systemActivityViewModel.setOutput("Accessibility Service is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onMultiClientIMEEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean,
            reboot: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val currentValue = systemActivityViewModel.systemServiceManager?.multiClientImeEnabled
                if (currentValue == enable) {
                    systemActivityViewModel.setOutput("Error: Set Multi Client IME enable value same as previous value")
                    return
                }
                val success = systemActivityViewModel.systemServiceManager?.setMultiClientImeEnabled(enable, reboot)
                if (success == true) {
                    systemActivityViewModel.setOutput("Multi Client IME is now $enable")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Multi Client IME")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetNotificationEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setNotificationStatusEnabled(enable)
                Toast.makeText(
                    systemActivityViewModel.applicationContext,
                    "Notification is now $enable",
                    Toast.LENGTH_SHORT
                ).show()
                systemActivityViewModel.setOutput("Notification is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onAppModeEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setMultiAppModeEnabled(
                        systemActivityViewModel.applicationContext,
                        enable
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("App Mode is now $enable")
                } else {
                    systemActivityViewModel.setOutput("Failed to set App Mode. Only supported in EloView Control and Core modes.")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetStickyEloViewModeEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.stickyEloViewModeEnabled = enable
                systemActivityViewModel.setOutput("Sticky EloView Mode is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetADBEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setAdbEnabled(enable)
                systemActivityViewModel.setOutput("ADB is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetLogsEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setEloLoggingEnabled(enable)
                systemActivityViewModel.setOutput("Logs are now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetDarkModeEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val isDarkMode =
                    (systemActivityViewModel.applicationContext.resources.configuration.uiMode and Configuration.UI_MODE_NIGHT_MASK) == Configuration.UI_MODE_NIGHT_YES
                if ((enable && isDarkMode) || (!enable && !isDarkMode)) {
                    systemActivityViewModel.setOutput("Dark Mode is already $enable")
                    return
                }
                systemActivityViewModel.systemServiceManager?.setDarkModeEnabled(!isDarkMode)
                systemActivityViewModel.setOutput("Dark Mode is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetFirmwareLockEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setFirmwareStatusLockEnabled(
                        enable
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("Firmware Lock is now $enable")
                    systemActivityViewModel.setFirmwareLocked(enable)
                } else {
                    systemActivityViewModel.setOutput("Failed to set Firmware Lock")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetCopyPasteEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setCopyPasteModeEnabled(enable)
                systemActivityViewModel.setOutput("Copy Paste is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetMediaProjectionPermissionAllAppsClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.mediaProjectionPermissionAllApps =
                    enable
                systemActivityViewModel.setOutput("Media Projection Permission is now $enable for all apps")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetScreenLockSoundEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.screenLockSoundEnabled = enable
                systemActivityViewModel.setOutput("Screen Lock Sound is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetChargeSoundAndVibrationClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.chargingSoundAndVibrationEnabled =
                    enable
                systemActivityViewModel.setOutput("Charging Sound and Vibration is ${if (enable) "Enabled." else "Disabled."}")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetBluetoothEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setBluetoothEnabled(enable)
                systemActivityViewModel.setOutput("Bluetooth is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetShowTapsEnableClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setShowTapsEnabled(enable)
                systemActivityViewModel.setOutput("Show Taps is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetAdbPortClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setAdbPort(
                        systemActivityViewModel.getAdbPort().toString()
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("ADB Port set to ${systemActivityViewModel.getAdbPort()}")
                } else {
                    systemActivityViewModel.setOutput("Failed to set ADB Port")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetDeviceBrightnessClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setBrightness(systemActivityViewModel.getDeviceBrightness())
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetDeviceBrightnessClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val brightness =
                    systemActivityViewModel.systemServiceManager?.brightness
                systemActivityViewModel.setOutput("Device Brightness: $brightness")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetNetworkConfigurationClick(
            systemActivityViewModel: SystemActivityViewModel,
            activityResult: ActivityResultLauncher<Intent>
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val intent = Intent(Intent.ACTION_OPEN_DOCUMENT)
                intent.addCategory(Intent.CATEGORY_OPENABLE)
                intent.type = "*/*"
                try {
                    activityResult.launch(intent)
                } catch (_: ActivityNotFoundException) {
                    systemActivityViewModel.setOutput("Please install a File Manager.")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetNetworkConfigurationClickActivityResult(
            systemActivityViewModel: SystemActivityViewModel,
            activityResult: ActivityResult
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (activityResult.resultCode == Activity.RESULT_OK) {
                    val selectedFile: Uri? = activityResult.data?.data
                    try {
                        checkNotNull(selectedFile)
                        val type: String? =
                            systemActivityViewModel.applicationContext.contentResolver.getType(
                                selectedFile
                            )
                        Log.d(TAG, "Set Network Config: mimetype: $type")
                        checkNotNull(type)
                        if (type == "text/csv" || type == "text/comma-separated-values") {
                            systemActivityViewModel.setOutput("Selecting file: $selectedFile")
                            val success =
                                systemActivityViewModel.systemServiceManager?.setEloNetworkConf(
                                    systemActivityViewModel.applicationContext,
                                    selectedFile
                                )
                            when (success) {
                                0 -> {
                                    systemActivityViewModel.setOutput("Network Configuration Set")
                                }

                                -1 -> {
                                    systemActivityViewModel.setOutput("Failed to parse Network Configuration File")
                                }

                                -2 -> {
                                    systemActivityViewModel.setOutput("Network Configuration must be CSV file format")
                                }
                            }
                        } else {
                            systemActivityViewModel.setOutput("Only local csv file is allowed")
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                        systemActivityViewModel.setOutput("Unexpected Error has occurred")
                    }
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onClearAppCacheAndUserDataClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val packageList = systemActivityViewModel.getClearAppCacheAndUserDataPackages()
                if (packageList.isEmpty()) {
                    systemActivityViewModel.setOutput("Please enter package name(s)")
                    return
                } else {
                    if (systemActivityViewModel.systemServiceManager?.clearAppCacheUserData(packageList) == true) {
                        systemActivityViewModel.setOutput("Cleared App Cache and Data for $packageList")
                    } else {
                        systemActivityViewModel.setOutput("Failed to clear App Cache and Data")
                    }
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onClearAllAppCacheAndUserDataClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val packageList = systemActivityViewModel.getClearAllAppCacheAndUserDataPackages()
                if (systemActivityViewModel.systemServiceManager?.clearAllAppCacheUserData(
                        packageList
                    ) == true
                ) {
                    if (packageList.isEmpty()) {
                        systemActivityViewModel.setOutput("No package exception set. Clearing all app cache and data")
                    } else {
                        systemActivityViewModel.setOutput("Cleared All App Cache and Data except for $packageList")
                    }
                } else {
                    systemActivityViewModel.setOutput("Failed to clear All App Cache and Data")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetSilentDefaultLauncherClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                var stdAndroid: String? =
                    systemActivityViewModel.systemServiceManager?.readSystemProperty("persist.sys.stdandroid")
                if (stdAndroid == null) {
                    stdAndroid = ""
                }
                if (stdAndroid.equals("true", ignoreCase = true)) {
                    val defaultLauncher = systemActivityViewModel.getSilentDefaultLauncher()
                    if (systemActivityViewModel.systemServiceManager?.setSilentDefaultLauncher(
                            defaultLauncher
                        ) == true
                    ) {
                        systemActivityViewModel.setOutput("Silent Default Launcher is now $defaultLauncher")
                    } else {
                        systemActivityViewModel.setOutput("Failed to set Silent Default Launcher")
                    }
                } else {
                    systemActivityViewModel.setOutput("Not allowed in EloView mode")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetStickyEloViewModeClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val stickyEloViewMode =
                    systemActivityViewModel.systemServiceManager?.stickyEloViewModeEnabled
                systemActivityViewModel.setOutput("Sticky EloView Mode: $stickyEloViewMode")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetNavigationBarClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val navigationBar =
                    systemActivityViewModel.systemServiceManager?.navigationBarEnabled
                systemActivityViewModel.setOutput("Navigation Bar: $navigationBar")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetStatusBarClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val statusBar = systemActivityViewModel.systemServiceManager?.statusBarEnabled
                systemActivityViewModel.setOutput("Status Bar: $statusBar")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetGrantAllAppPermissionClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val grantAllAppPermission =
                    systemActivityViewModel.systemServiceManager?.grantAllAppPermissionEnabled
                systemActivityViewModel.setOutput("Auto Grant All App Permission: $grantAllAppPermission")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetGrantAutoUSBPermissionClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val grantAutoUSBPermission =
                    systemActivityViewModel.systemServiceManager?.autoAllowUsbPermissionEnabled
                systemActivityViewModel.setOutput("Auto Grant Auto USB Permission: $grantAutoUSBPermission")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetMultiClientImeClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val multiClientIme =
                    systemActivityViewModel.systemServiceManager?.multiClientImeEnabled
                systemActivityViewModel.setOutput("Multi Client IME: $multiClientIme")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetInstallPermissionToSpecificAppClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val packageName = systemActivityViewModel.getGetInstallPermissionToSpecificApp()
                val installPermissionToSpecificApps =
                    systemActivityViewModel.systemServiceManager?.getInstallPermissionToSpecificApp(
                        packageName
                    )
                systemActivityViewModel.setOutput("Install Permission for App $packageName: is set to $installPermissionToSpecificApps")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetLockTypeClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val lockType = systemActivityViewModel.systemServiceManager?.lockType
                systemActivityViewModel.setOutput("Lock Type: $lockType")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetBacklightTimeoutClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val backlightTimeoutSelection =
                    systemActivityViewModel.getSelectedBacklightTimeout()
                val backlightTimeout = when (backlightTimeoutSelection.lowercase()) {
                    "never" -> 0
                    "15 seconds" -> 15 * 1000
                    "30 seconds" -> 30 * 1000
                    "1 minute" -> 60 * 1000
                    "2 minutes" -> 120 * 1000
                    "5 minutes" -> 300 * 1000
                    "10 minutes" -> 600 * 1000
                    "30 minutes" -> 1800 * 1000
                    else -> -1
                }
                if (systemActivityViewModel.systemServiceManager?.setBacklightTimeOut(
                        backlightTimeout
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("Backlight Timeout set to $backlightTimeout")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Backlight Timeout")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetDisplaySizeClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val displaySize = systemActivityViewModel.getSelectedDisplaySize()
                if (displaySize == "Select a Display Size") {
                    systemActivityViewModel.setOutput("Invalid Display Size")
                    return
                }
                if (systemActivityViewModel.systemServiceManager?.setDisplaySize(displaySize) == true) {
                    systemActivityViewModel.setOutput("Display Size set to $displaySize")
                    Toast.makeText(
                        systemActivityViewModel.applicationContext,
                        "Display Size set to $displaySize",
                        Toast.LENGTH_SHORT
                    ).show()
                } else {
                    systemActivityViewModel.setOutput("Failed to set Display Size")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetFontSizeClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val fontSize = systemActivityViewModel.getSelectedFontSize()
                if (systemActivityViewModel.systemServiceManager?.setFontSize(fontSize) == true) {
                    systemActivityViewModel.setOutput("Font Size set to $fontSize")
                    Toast.makeText(
                        systemActivityViewModel.applicationContext,
                        "Font Size set to $fontSize",
                        Toast.LENGTH_SHORT
                    ).show()
                } else {
                    systemActivityViewModel.setOutput("Failed to set Font Size")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetWiFiMacAddressClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val macAddress = systemActivityViewModel.getSelectedWiFiMacAddress()
                val randomize = macAddress == "Use randomized MAC"
                systemActivityViewModel.systemServiceManager?.setRandomWifiMacAddressEnabled(
                    randomize
                )
                systemActivityViewModel.setOutput("WiFi MAC Address set to $macAddress")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetVolumeClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val volumeType = systemActivityViewModel.getSelectedVolumeType()
                val volume = systemActivityViewModel.getSetVolumeValue()
                if (volume < 0 || volume > 100) {
                    systemActivityViewModel.setOutput("Invalid Volume: $volume")
                    return
                }
                if (systemActivityViewModel.systemServiceManager?.setVolumeLevel(
                        volumeType.lowercase(),
                        volume
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("$volumeType Volume set to $volume")
                } else {
                    systemActivityViewModel.setOutput("Failed to set $volumeType Volume to $volume")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetDpiClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                try {
                    val dpi = systemActivityViewModel.getSelectedDPI()
                    if (dpi == "Select DPI") {
                        systemActivityViewModel.setOutput("Invalid DPI")
                        return
                    }
                    if (systemActivityViewModel.systemServiceManager?.setDpi(dpi.toInt()) == true) {
                        systemActivityViewModel.setOutput("DPI set to $dpi. Please reboot device for change to take affect")
                    } else {
                        systemActivityViewModel.setOutput("Failed to set DPI")
                    }
                } catch (_: Exception) {
                    systemActivityViewModel.setOutput("Failed to set DPI")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetDpiClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val dpi = systemActivityViewModel.systemServiceManager?.dpi
                systemActivityViewModel.setOutput("DPI: $dpi")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetCFDDpiClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                try {
                    val dpi = systemActivityViewModel.getSelectedCFDDPI()
                    if (dpi == "Select External display DPI") {
                        systemActivityViewModel.setOutput("Invalid External display DPI")
                        return
                    }
                    if (systemActivityViewModel.systemServiceManager?.setCFDDpi(dpi.toInt()) == true) {
                        systemActivityViewModel.setOutput("External display DPI set to $dpi.")
                    } else {
                        systemActivityViewModel.setOutput("Failed to set External display DPI")
                    }
                } catch (_: Exception) {
                    systemActivityViewModel.setOutput("Failed to set External display DPI")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetCFDDpiClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val dpi = systemActivityViewModel.systemServiceManager?.cfdDpi
                systemActivityViewModel.setOutput("CFD DPI: $dpi")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetBacklightTimerClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val backlightTimer = systemActivityViewModel.systemServiceManager?.backlightTimeOut
                if (backlightTimer == Int.MAX_VALUE) {
                    systemActivityViewModel.setOutput("Backlight Timer: Never")
                } else {
                    systemActivityViewModel.setOutput("Backlight Timer: $backlightTimer")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetTimeZoneClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val timeZone = systemActivityViewModel.getSetTimezone()
                if (systemActivityViewModel.systemServiceManager?.setTimeZone(timeZone) == true) {
                    systemActivityViewModel.setOutput("Time Zone set to $timeZone")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Time Zone")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetTimeZoneClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val timeZone = systemActivityViewModel.systemServiceManager?.timeZone
                systemActivityViewModel.setOutput("Time Zone: $timeZone")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetLanguageClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val language = systemActivityViewModel.getSetLanguage()
                if (systemActivityViewModel.systemServiceManager?.setLanguage(language) == true) {
                    systemActivityViewModel.setOutput("Language set to $language")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Language")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetLanguageClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val language = systemActivityViewModel.systemServiceManager?.language
                systemActivityViewModel.setOutput("Language: $language")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetNtpPollIntervalClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            try {
                if (systemActivityViewModel.getServiceConnected()) {
                    val ntpPollInterval = systemActivityViewModel.getNtpPollInterval()
                    val rebootRequired =
                        systemActivityViewModel.getNoRebootPollIntervalChecked().not()
                    if (systemActivityViewModel.systemServiceManager?.setNtpPollInterval(
                            ntpPollInterval.toLong(),
                            rebootRequired
                        ) == true
                    ) {
                        systemActivityViewModel.setOutput("NTP Poll Interval set to $ntpPollInterval")
                    } else {
                        Toast.makeText(
                            systemActivityViewModel.applicationContext,
                            "Enter Valid Interval",
                            Toast.LENGTH_SHORT
                        ).show()
                        systemActivityViewModel.setOutput("Failed to set NTP Poll Interval")
                    }
                } else {
                    systemActivityViewModel.setOutput("Service not connected")
                }
            } catch (e: Exception) {
                systemActivityViewModel.setOutput("Failed to set NTP Poll Interval: $e")
            }
        }

        fun onGetNtpPollIntervalClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val ntpPollInterval = systemActivityViewModel.systemServiceManager?.ntpPollInterval
                systemActivityViewModel.setOutput("NTP Poll Interval: $ntpPollInterval")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetDateTimeClick(
            systemActivityViewModel: SystemActivityViewModel,
            dateTimeEpoch: Long
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                try {
                    if (systemActivityViewModel.getServiceConnected()) {
                        systemActivityViewModel.systemServiceManager?.setDateTime(dateTimeEpoch)
                        val date = Date(dateTimeEpoch)
                        val sdf = SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.getDefault())
                        systemActivityViewModel.setOutput("Date Time set to ${sdf.format(date)}")
                    } else {
                        systemActivityViewModel.setOutput("Failed to set Date Time: Service is not Connected")
                    }
                } catch (e: Exception) {
                    systemActivityViewModel.setOutput("Failed to set Date Time: $e")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetPasswordClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val oldPassword = systemActivityViewModel.getOldPassword()
                val password = systemActivityViewModel.getNewPassword()
                val askPasswordOnBoot = systemActivityViewModel.getAskForPasswordOnBoot()
                CoroutineScope(Dispatchers.IO).launch {
                    // Call the method that returns an error code
                    val responseCode =
                        systemActivityViewModel.systemServiceManager?.changeLockScreenPassword(
                            systemActivityViewModel.applicationContext,
                            oldPassword,
                            password,
                            askPasswordOnBoot
                        )
                    withContext(Dispatchers.Main) {
                        // Handle the response code
                        when (responseCode) {
                            SystemActivityViewModel.SUCCESS_CODE -> {
                                systemActivityViewModel.setOutput("Password successfully set")
                            }

                            SystemActivityViewModel.ERROR_SINGLE_APP_MODE -> {
                                systemActivityViewModel.setOutput("Cannot set password in single app mode")
                            }

                            SystemActivityViewModel.ERROR_OTHER -> {
                                systemActivityViewModel.setOutput("Failed to set Password")
                            }

                            else -> {
                                systemActivityViewModel.setOutput("Unknown error occurred")
                            }
                        }
                    }
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetPinClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val oldPin = systemActivityViewModel.getOldPin()
                val pin = systemActivityViewModel.getNewPin()
                val askPinOnBoot = systemActivityViewModel.getAskForPinOnBoot()
                CoroutineScope(Dispatchers.IO).launch {
                    val responseCode =
                        systemActivityViewModel.systemServiceManager?.changeLockScreenPin(
                            systemActivityViewModel.applicationContext,
                            oldPin,
                            pin,
                            askPinOnBoot
                        )
                    withContext(Dispatchers.Main) {
                        // Handle the response code
                        when (responseCode) {
                            SystemActivityViewModel.SUCCESS_CODE -> {
                                systemActivityViewModel.setOutput("Pin successfully set")
                            }

                            SystemActivityViewModel.ERROR_SINGLE_APP_MODE -> {
                                systemActivityViewModel.setOutput("Cannot set pin in single app mode")
                            }

                            SystemActivityViewModel.ERROR_OTHER -> {
                                systemActivityViewModel.setOutput("Failed to set Pin")
                            }

                            else -> {
                                systemActivityViewModel.setOutput("Unknown error occurred")
                            }
                        }
                    }
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetPlayProtectClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val playProtect = systemActivityViewModel.systemServiceManager?.playProtect
                systemActivityViewModel.setOutput("Play Protect: $playProtect")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onEnablePlayProtectClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.playProtect = enable
                systemActivityViewModel.setOutput("Play Protect is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onEnableBackLightClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setBacklightEnabled(enable) == true) {
                    systemActivityViewModel.setOutput("Back Light is now $enable")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Back Light")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onEnableWiFiClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.wifiEnabled = enable
                systemActivityViewModel.setOutput("WiFi is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetResolutionClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                try {
                    val dpi = systemActivityViewModel.systemServiceManager?.dpi
                    val width = systemActivityViewModel.getResolutionWidth()
                    val height = systemActivityViewModel.getResolutionHeight()

                    if (Util.isM100() || Util.is7inch()) {
                        if (dpi != null && dpi >= 320 && (height < 1280 || width < 720)) {
                            systemActivityViewModel.setOutput("Height must be greater than or equal to 1280. Width must be greater than or equal to 720")
                            return
                        }

                    } else {
                        if (dpi != null && dpi >= 320 && (height < 480 || width < 600)) {
                            systemActivityViewModel.setOutput("Height must be greater than or equal to 480. Width must be greater than or equal to 600")
                            return
                        }

                        if (height < 360 || width < 480) {
                            systemActivityViewModel.setOutput("Height must be greater than or equal to 360. Width must be greater than or equal to 480")
                            return
                        }
                    }

                    if (Util.isValueDevice()) {
                        if (Util.is10inch()) {
                            if (height > 1600) {
                                systemActivityViewModel.setOutput("Height must be less than or equal to 1600")
                                return
                            }
                            if (width > 2560) {
                                systemActivityViewModel.setOutput("Width must be less than or equal to 2560")
                                return
                            }
                        } else {
                            if (height > 2160) {
                                systemActivityViewModel.setOutput("Height must be less than or equal to 2160")
                                return
                            }
                            if (width > 3840) {
                                systemActivityViewModel.setOutput("Width must be less than or equal to 3840")
                                return
                            }
                        }
                    } else if (Util.isPerformanceDevice()) {
                        if (Util.is10inch()) {
                            if (height > 3840) {
                                systemActivityViewModel.setOutput("Height must be less than or equal to 3840")
                                return
                            }
                            if (width > 2400) {
                                systemActivityViewModel.setOutput("Width must be less than or equal to 2400")
                                return
                            }
                        } else if (Util.isM50()) {
                            if (height > 2560) {
                                systemActivityViewModel.setOutput("Height must be less than or equal to 2560")
                                return
                            }
                            if (width > 1440) {
                                systemActivityViewModel.setOutput("Width must be less than or equal to 1440")
                                return
                            }
                        } else if (Util.isM100() || Util.is7inch()) {
                            if (height > 3840) {
                                systemActivityViewModel.setOutput("Height must be less than or equal to 3840")
                                return
                            }
                            if (width > 2160) {
                                systemActivityViewModel.setOutput("Width must be less than or equal to 2160")
                                return
                            }
                        } else if (Util.isM60()) {
                            if (height > 2880) {
                                systemActivityViewModel.setOutput("Height must be less than or equal to 2880")
                                return
                            }
                            if (width > 1440) {
                                systemActivityViewModel.setOutput("Width must be less than or equal to 1440")
                                return
                            }
                        } else {
                            if (height > 2160) {
                                systemActivityViewModel.setOutput("Height must be less than or equal to 2160")
                                return
                            }
                            if (width > 3840) {
                                systemActivityViewModel.setOutput("Width must be less than or equal to 3840")
                                return
                            }
                        }
                    }
                    if (systemActivityViewModel.systemServiceManager?.setScreenSize(
                            width,
                            height
                        ) == true
                    ) {
                        systemActivityViewModel.setOutput("Resolution set to $width x $height")
                    } else {
                        systemActivityViewModel.setOutput("Failed to set Resolution")
                    }
                } catch (e: Exception) {
                    systemActivityViewModel.setOutput("Invalid Resolution")
                    e.printStackTrace()
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetResolutionClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val resolution = systemActivityViewModel.systemServiceManager?.screenSize
                systemActivityViewModel.setOutput("Resolution: $resolution")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetDeviceOwnerStatusClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val deviceOwnerStatus =
                    systemActivityViewModel.systemServiceManager?.deviceOwnerStatus
                systemActivityViewModel.setOutput("Device Owner Status: $deviceOwnerStatus")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetSmallestWidthClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val smallestWidth = systemActivityViewModel.systemServiceManager?.smallestPixelWidth
                systemActivityViewModel.setOutput("Smallest Width: $smallestWidth")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetDeviceOwnerClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val packageName = systemActivityViewModel.getDeviceOwnerPackage()
                val receiverName = systemActivityViewModel.getDeviceOwnerReceiver()
                if (packageName.isEmpty() || receiverName.isEmpty()) {
                    systemActivityViewModel.setOutput("Please enter a package name and receiver name")
                    return
                }
                if (systemActivityViewModel.systemServiceManager?.setDeviceOwner(
                        packageName,
                        receiverName
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("Device Owner set to $packageName")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Device Owner")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onStartActivityFromForegroundServiceClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val packageName = systemActivityViewModel.getForegroundServicePackageName()
                if (packageName.isEmpty()) {
                    systemActivityViewModel.setOutput("Please enter a package name")
                    return
                }
                if (!Util.isPackageInstalled(
                        systemActivityViewModel.applicationContext,
                        packageName
                    )
                ) {
                    Log.d(TAG, "$packageName is not installed")
                    systemActivityViewModel.setOutput("$packageName is not installed")
                    return
                }

                val serviceIntent = Intent(
                    systemActivityViewModel.applicationContext,
                    ForegroundService::class.java
                )
                serviceIntent.putExtra("pkgName", packageName)
                ForegroundService.systemActivityViewModel = systemActivityViewModel
                ContextCompat.startForegroundService(
                    systemActivityViewModel.applicationContext,
                    serviceIntent
                )
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onStopForegroundServiceClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val serviceIntent = Intent(
                    systemActivityViewModel.applicationContext,
                    ForegroundService::class.java
                )
                systemActivityViewModel.applicationContext.stopService(serviceIntent)
                systemActivityViewModel.setOutput("Foreground Service stopped")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetSmallestWidthClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val smallestWidth = systemActivityViewModel.getSmallestPixelWidth()
                if (systemActivityViewModel.systemServiceManager?.setSmallestPixelWidth(
                        smallestWidth
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("Smallest Width set to $smallestWidth")
                } else {
                    systemActivityViewModel.setOutput("Failed to set Smallest Width")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetNtpUrlClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val ntpUrl = systemActivityViewModel.getNtpUrl()
                if (ntpUrl.isEmpty()) {
                    systemActivityViewModel.setOutput("Please enter an NTP URL")
                    return
                }
                val rebootRequired = systemActivityViewModel.getNoRebootNtpUrlChecked().not()
                systemActivityViewModel.systemServiceManager?.setNtpServerUrl(
                    ntpUrl,
                    rebootRequired
                )
                systemActivityViewModel.setOutput("NTP URL set to $ntpUrl")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetNtpUrlClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val ntpUrl = systemActivityViewModel.systemServiceManager?.ntpServerUrl
                systemActivityViewModel.setOutput("NTP URL: $ntpUrl")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetHostNameClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val hostName = systemActivityViewModel.getHostName()
                if (hostName.isEmpty()) {
                    systemActivityViewModel.setOutput("Please enter a valid Hostname")
                    return
                }
                systemActivityViewModel.systemServiceManager?.hostName = hostName
                systemActivityViewModel.setOutput("Host Name set to $hostName")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetHostNameClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val hostName = systemActivityViewModel.systemServiceManager?.hostName
                systemActivityViewModel.setOutput("Host Name: $hostName")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSecureProcessorFirmwareUpgradeClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.spFirmwareUpgrade() == true) {
                    systemActivityViewModel.setOutput("Secure Processor Firmware Upgrade started")
                } else {
                    systemActivityViewModel.setOutput("Failed to start Secure Processor Firmware Upgrade")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetAospKeyboardClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setVirtualKeyBoard("aosp")
                systemActivityViewModel.setOutput("AOSP Keyboard set")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetGboardKeyboardClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setVirtualKeyBoard("gboard")
                systemActivityViewModel.setOutput("GMS Keyboard set")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetAutoIdleModeStatusClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val autoIdleMode = systemActivityViewModel.systemServiceManager?.autoIdleModeEnabled
                systemActivityViewModel.setOutput("Auto Idle Mode: $autoIdleMode")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetTouchSoundStatusClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val touchSound = systemActivityViewModel.systemServiceManager?.touchSoundEnabled
                systemActivityViewModel.setOutput("Touch Sound: $touchSound")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetScreenLockSoundStatusClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val screenLockSound =
                    systemActivityViewModel.systemServiceManager?.screenLockSoundEnabled
                systemActivityViewModel.setOutput("Screen Lock Sound: $screenLockSound")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetHotSpotAccessibilityEnabledStatusClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val hotSpotState =
                    systemActivityViewModel.systemServiceManager?.hotspotAccessibilityEnabled
                systemActivityViewModel.setOutput("HotSpot Accessibility Status: $hotSpotState")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetChargeSoundAndVibrationClick(systemActivityViewModel: SystemActivityViewModel) {
            if (systemActivityViewModel.getServiceConnected()) {
                val chargeSoundsVibration =
                    systemActivityViewModel.systemServiceManager?.chargingSoundAndVibrationEnabled
                systemActivityViewModel.setOutput("Charging Sound and Vibration: ${if (chargeSoundsVibration == true) "Enabled" else "Disabled"}")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetInternalAudioClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setExternalAudioDeviceEnabled(false)
                systemActivityViewModel.setOutput("Internal Audio set")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetExternalAudioClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setExternalAudioDeviceEnabled(true)
                systemActivityViewModel.setOutput("External Audio set")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onVerifyValidFileSignature(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val result = systemActivityViewModel.verifyFileSignature(true)
                systemActivityViewModel.setOutput(result)
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onVerifyInvalidFileSignature(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val result = systemActivityViewModel.verifyFileSignature(false)
                systemActivityViewModel.setOutput(result)
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onScreenWakeUpClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val wakeUpDelay = systemActivityViewModel.getScreenWakeUpDelay()
                if (wakeUpDelay <= 0) {
                    systemActivityViewModel.setOutput("Invalid delay: $wakeUpDelay")
                    return
                }
                systemActivityViewModel.systemServiceManager?.setScreenWakeUp(wakeUpDelay.toLong())
                systemActivityViewModel.setOutput("Screen Wake Up set to: $wakeUpDelay ms")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetScreenLockTypeClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val currentLockType = systemActivityViewModel.systemServiceManager?.lockType
                if (currentLockType?.lowercase() == "pin" || currentLockType?.lowercase() == "password") {
                    val currPassword = systemActivityViewModel.getCurrentLockPassword()
                    if (currPassword?.isNotEmpty() == true) {
                        val lockType = systemActivityViewModel.getSelectedLockType()
                        when (systemActivityViewModel.systemServiceManager.setLockType(
                            currPassword,
                            lockType
                        )) {
                            0 -> systemActivityViewModel.setOutput("Lock Type set to: $lockType")
                            -2 -> systemActivityViewModel.setOutput("Failed to set Lock Type: Incorrect PIN Provided")
                            -3 -> systemActivityViewModel.setOutput("Failed to set Lock Type: Incorrect Password Provided")
                            -4 -> systemActivityViewModel.setOutput("Failed to set Lock Type: System Service is not connected")
                            else -> systemActivityViewModel.setOutput("Failed to set Lock Type: Unknown Error occured")
                        }
                        return
                    } else {
                        systemActivityViewModel.setShowSetLockTypeWithPasswordDialog(true)
                        return
                    }
                }
                val lockType = systemActivityViewModel.getSelectedLockType()
                when (systemActivityViewModel.systemServiceManager?.setLockType(null, lockType)) {
                    0 -> systemActivityViewModel.setOutput("Lock Type set to: $lockType")
                    -2 -> systemActivityViewModel.setOutput("Failed to set Lock Type: Incorrect PIN Provided")
                    -3 -> systemActivityViewModel.setOutput("Failed to set Lock Type: Incorrect Password Provided")
                    -4 -> systemActivityViewModel.setOutput("Failed to set Lock Type: System Service is not connected")
                    else -> systemActivityViewModel.setOutput("Failed to set Lock Type: Unknown Error occured")
                }
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetAutoDateTimeClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setAutoDateTimeEnabled(enable)
                systemActivityViewModel.setOutput("Auto Date Time is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service is not Connected")
            }
        }

        fun onSetUse24HourFormatClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.setTimeFormatEnabled(enable)
                systemActivityViewModel.setOutput("Use 24 Hour Format is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service is not Connected")
            }
        }

        fun onSetShowInsetsClick(
            systemActivityViewModel: SystemActivityViewModel,
            show: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val isEloviewMode =
                    "false" == systemActivityViewModel.systemServiceManager?.readSystemProperty("persist.sys.stdandroid")
                if (!isEloviewMode) {
                    systemActivityViewModel.systemServiceManager?.showInsetsOnSwipe = show
                    systemActivityViewModel.setOutput("Showing insets when swiping: $show")
                } else {
                    systemActivityViewModel.setOutput("API not supported in eloview mode")
                }
            } else {
                systemActivityViewModel.setOutput("Service is not Connected")
            }
        }

        fun onGetShowInsetsClick(
            systemActivityViewModel: SystemActivityViewModel,
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val showInsets = systemActivityViewModel.systemServiceManager?.showInsetsOnSwipe
                systemActivityViewModel.setOutput("show insets on swipe is: $showInsets")
            } else {
                systemActivityViewModel.setOutput("Service is not Connected")
            }
        }

        fun onSetMediaProjectionPermission(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val pkgName = systemActivityViewModel.getMediaProjectionPermissionPackage()
                systemActivityViewModel.systemServiceManager?.mediaProjectionPermission = pkgName
                systemActivityViewModel.setOutput("Package Names Set To: $pkgName")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetMediaProjectionPermissionAllAppsClick(
            systemActivityViewModel: SystemActivityViewModel,
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val allowed = systemActivityViewModel.systemServiceManager?.mediaProjectionPermissionAllApps
                systemActivityViewModel.setOutput("Media Projection Permission Is Allowed For All Apps : $allowed")
            } else {
                systemActivityViewModel.setOutput("Service is not Connected")
            }
        }

        fun onGetMediaProjectionPermissionClick(
            systemActivityViewModel: SystemActivityViewModel,
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val pkgName = systemActivityViewModel.systemServiceManager?.mediaProjectionPermission
                systemActivityViewModel.setOutput("Media Projection Permission Allowed Packages: $pkgName")
            } else {
                systemActivityViewModel.setOutput("Service is not Connected")
            }
        }

        private var keyListenerSet: Boolean = false
        fun onUnsetKeyListenerInterceptorClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (systemActivityViewModel.systemServiceManager?.setKeyListener(
                        "",
                        "",
                        arrayListOf(),
                        arrayListOf()
                    ) == true
                ) {
                    systemActivityViewModel.setOutput("Key Listener Interceptor unset")
                } else {
                    systemActivityViewModel.setOutput("Failed to unset Key Listener Interceptor")
                }
                keyListenerSet = false
                systemActivityViewModel.registerKeyReceiver(false)
            } else {
                systemActivityViewModel.setOutput("Service is not Connected")
            }
        }

        fun onSetKeyListenerClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                if (keyListenerSet) {
                    // to prevent user from accidentally clicking multiple key listener buttons without unregistering
                    if (systemActivityViewModel.systemServiceManager?.setKeyListener(
                            "",
                            "",
                            arrayListOf(),
                            arrayListOf()
                        ) == true
                    ) {
                        systemActivityViewModel.setOutput("Key listener unregistered")
                    } else {
                        systemActivityViewModel.setOutput("Failed to unregister key listener")
                    }
                    keyListenerSet = false
                }
                val keyListenerList = systemActivityViewModel.getSelectedKeyListenerValues()
                val keys = arrayListOf<Int>()
                val interceptKeys = arrayListOf<Boolean>()
                if (keyListenerList.isNotEmpty()) {
                    for (key in keyListenerList) {
                        keys.add(KeyEvent.keyCodeFromString(key))
                        interceptKeys.add(false)
                    }
                    systemActivityViewModel.systemServiceManager?.setKeyListener(
                        systemActivityViewModel.applicationContext.packageName,
                        KeyReceiver::class.java.name,
                        keys,
                        interceptKeys
                    )
                    keyListenerSet = true
                    systemActivityViewModel.registerKeyReceiver(true)
                    val keysString = keys.joinToString(",")
                    val interceptKeysString = interceptKeys.joinToString(",") { it.toString() }
                    systemActivityViewModel.setOutput("Set Key Event Listener Set for $keysString : $interceptKeysString")
                } else {
                    systemActivityViewModel.setOutput("No keys selected")
                }
            } else {
                systemActivityViewModel.setOutput("Service is not Connected")
            }
        }

        fun onSetKeyInterceptorClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {

                if (keyListenerSet) {
                    // to prevent user from accidentally clicking multiple key listener buttons without unregistering
                    if (systemActivityViewModel.systemServiceManager?.setKeyListener(
                            "",
                            "",
                            arrayListOf(),
                            arrayListOf()
                        ) == true
                    ) {
                        systemActivityViewModel.setOutput("Key listener unregistered")
                    } else {
                        systemActivityViewModel.setOutput("Failed to unregister key listener")
                    }
                    keyListenerSet = false
                }
                val keyListenerList = systemActivityViewModel.getSelectedKeyListenerValues()
                val keys = arrayListOf<Int>()
                val interceptKeys = arrayListOf<Boolean>()
                if (keyListenerList.isNotEmpty()) {
                    for (key in keyListenerList) {
                        keys.add(KeyEvent.keyCodeFromString(key))
                        interceptKeys.add(true)
                    }
                    systemActivityViewModel.systemServiceManager?.setKeyListener(
                        systemActivityViewModel.applicationContext.packageName,
                        KeyReceiver::class.java.name,
                        keys,
                        interceptKeys
                    )
                    keyListenerSet = true
                    systemActivityViewModel.registerKeyReceiver(true)
                    val keysString = keys.joinToString(",")
                    val interceptKeysString = interceptKeys.joinToString(",") { it.toString() }
                    systemActivityViewModel.setOutput("Set Key Event Listener Set for $keysString : $interceptKeysString")
                } else {
                    systemActivityViewModel.setOutput("No keys selected")
                }
            } else {
                systemActivityViewModel.setOutput("Service is not Connected")
            }
        }

        fun onEnableHotSpotAccessibilityClick(
            systemActivityViewModel: SystemActivityViewModel,
            enable: Boolean
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                systemActivityViewModel.systemServiceManager?.hotspotAccessibilityEnabled = enable
                systemActivityViewModel.setOutput("HotSpot Accessibility is now $enable")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetSystemPropertyClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val systemProperty =
                    systemActivityViewModel.systemServiceManager?.readSystemProperty(
                        systemActivityViewModel.getGetSystemProperty()
                    )
                systemActivityViewModel.setOutput("System Property: ${systemActivityViewModel.getGetSystemProperty()} : $systemProperty")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetMD5ChecksumClick(
            systemActivityViewModel: SystemActivityViewModel
        ) {
            if (systemActivityViewModel.getServiceConnected()) {
                val md5CheckSum =
                    systemActivityViewModel.systemServiceManager?.getMD5Checksum(
                        systemActivityViewModel.getGetMD5ChecksumFilePath()
                    )
                systemActivityViewModel.setOutput("MD5 Checksum: ${systemActivityViewModel.getGetMD5ChecksumFilePath()} : $md5CheckSum")
            } else {
                systemActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onDownloadAndConvertZipFileToByteArray(
            systemActivityViewModel: SystemActivityViewModel,
            filePath: String,
            selectedUri: Uri
        ) {
            if (Util.is7inch() || Util.isM100()) {
                if (systemActivityViewModel.getServiceConnected()) {
                    if (filePath.startsWith("file:///android_asset/")) {
                        val assetFileName: String = filePath.replace(
                            "file:///android_asset/",
                            ""
                        )
                        zipFile = FileUtils.copyAssetToCache(assetFileName, systemActivityViewModel.applicationContext)
                        if (FileUtils.isFileValid(zipFile)) {
                            fileByteArray = FileUtils.convertZipFileToByteArray(zipFile!!)
                            systemActivityViewModel.setOutput("Successfully Zip File converted to Byte Array")
                        } else {
                            systemActivityViewModel.setOutput("File is not a valid zip")
                        }
                        return
                    } else if (selectedUri != Uri.EMPTY) {
                        if (FileUtils.isFileValid(systemActivityViewModel.applicationContext, selectedUri)) {
                            fileByteArray = FileUtils.convertZipUriToByteArray(systemActivityViewModel.applicationContext, selectedUri)
                            systemActivityViewModel.setOutput("Successfully Zip File converted to Byte Array")
                        } else {
                            systemActivityViewModel.setOutput("File is not a valid zip")
                        }
                        return
                    } else if (filePath.startsWith("https://") || filePath.startsWith("http://")) {
                        systemActivityViewModel.viewModelScope.launch {
                            val bytes = downloadZipFileFromRemoteURL(systemActivityViewModel.applicationContext,
                                filePath)
                            if (bytes != null) {
                                fileByteArray = bytes
                                systemActivityViewModel.setOutput("Downloaded and converted successfully.")
                            } else {
                                systemActivityViewModel.setOutput("Download failed or file invalid.")
                            }
                        }
                        return
                    }
                } else {
                    systemActivityViewModel.setOutput("Service not connected")
                }
            } else {
                systemActivityViewModel.setOutput("Feature not supported")
            }
        }

        fun writeSignedFileToStorage(
            systemActivityViewModel: SystemActivityViewModel,
        ){
            if (Util.is7inch() || Util.isM100()){
                if (systemActivityViewModel.getServiceConnected()) {
                    if (fileByteArray != null && fileByteArray!!.isNotEmpty()) {
                        val result =
                            systemActivityViewModel.systemServiceManager?.writeSignedFileToStorage(
                                fileByteArray
                            )
                        systemActivityViewModel.setOutput("WRITE SIGNED FILE TO STORAGE: \n $result")
                    }else {
                        systemActivityViewModel.setOutput("Select file and convert to byte array before performing the operation")
                    }
                } else {
                    systemActivityViewModel.setOutput("Service not connected")
                }
            } else {
                systemActivityViewModel.setOutput("Feature not supported")
            }
        }

        fun onVerifyFileSignature(
            systemActivityViewModel: SystemActivityViewModel,
        ) {
            if (Util.is7inch() || Util.isM100()){
                if (systemActivityViewModel.getServiceConnected()) {
                    if (fileByteArray != null && fileByteArray!!.isNotEmpty()) {
                        val result =
                            systemActivityViewModel.systemServiceManager?.verifyFileSignature(
                                fileByteArray
                            )
                        systemActivityViewModel.setOutput("VERIFY FILE SIGNATURE: \n $result")
                    }else{
                        systemActivityViewModel.setOutput("Select file and convert to byte array before performing the operation")
                    }
                } else {
                    systemActivityViewModel.setOutput("Service not connected")
                }
            } else {
                systemActivityViewModel.setOutput("Feature not supported")
            }
        }

        fun onReadElopayFileLoaderPartition(
            systemActivityViewModel: SystemActivityViewModel
        ){
            if (Util.is7inch() || Util.isM100()){
                if (systemActivityViewModel.getServiceConnected()) {
                    val result = readElopayFileLoaderPartition()
                    systemActivityViewModel.setOutput(result)
                } else {
                    systemActivityViewModel.setOutput("Service not connected")
                }
            } else {
                systemActivityViewModel.setOutput("Feature not supported")
            }
        }

        @Throws(RemoteException::class)
        fun readElopayFileLoaderPartition(): String {
            var ret = ""
            ret = FileUtils.listFilesInEloPay().ifEmpty {
                "No files found"
            }
            return ret
        }

        fun onEnableSecureWhitelistMode(
            systemActivityViewModel: SystemActivityViewModel
        ){
            if(Util.is7inch() || Util.isM100()){
                if(systemActivityViewModel.getServiceConnected()){
                    val result = systemActivityViewModel.systemServiceManager?.enableSecureWhiteListMode()
                    systemActivityViewModel.setOutput("$result")
                } else {
                    systemActivityViewModel.setOutput("Service not connected")
                }
            } else {
                systemActivityViewModel.setOutput("Feature not supported")
            }
        }

        fun onGetSecureWhitelistModeStatus(
            systemActivityViewModel: SystemActivityViewModel
        ){
            if (Util.is7inch() || Util.isM100()){
                if(systemActivityViewModel.getServiceConnected()){
                    val result = systemActivityViewModel.systemServiceManager?.secureWhitelistModeStatus
                    systemActivityViewModel.setOutput("$result")
                } else {
                    systemActivityViewModel.setOutput("Service not connected")
                }
            } else {
                systemActivityViewModel.setOutput("Feature not supported")
            }
        }

        private suspend fun downloadZipFileFromRemoteURL(
            context: Context,
            zipUrl: String,
        ): ByteArray? {
            if (zipUrl.isBlank() || !(zipUrl.startsWith("http://") || zipUrl.startsWith("https://"))) return null

            if (!FileUtils.isInternetAvailable(context)) {
                Log.d(TAG, "No internet connection.")
                return null
            }

            val downloadsDir = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS)
            val targetFile = File(downloadsDir, "ELO_AUTHORIZED_FILES.zip")

            try {
                if (targetFile.exists() && targetFile.delete()) {
                    Log.d(TAG, "Old downloaded file deleted.")
                }
                val request = buildDownloadRequest(zipUrl, targetFile.name)
                val downloadManager = context.getSystemService(Context.DOWNLOAD_SERVICE) as DownloadManager
                val downloadId = downloadManager.enqueue(request)

                val downloadSuccess = waitForDownload(downloadManager, downloadId)

                return if (downloadSuccess && FileUtils.isFileValid(targetFile)) {
                    val bytes = FileUtils.convertZipFileToByteArray(targetFile)
                    if (FileUtils.isZipFileExists()) FileUtils.deleteExistingFiles()
                    bytes
                } else null
            } catch (e: Exception) {
                Log.e(TAG, "Error downloading file: ${e.message}", e)
                return null
            }
        }

        private fun buildDownloadRequest(url: String, fileName: String): DownloadManager.Request {
            return DownloadManager.Request(url.toUri()).apply {
                setDescription("Downloading file...")
                setNotificationVisibility(DownloadManager.Request.VISIBILITY_VISIBLE_NOTIFY_COMPLETED)
                setDestinationInExternalPublicDir(Environment.DIRECTORY_DOWNLOADS, fileName)
            }
        }

        private suspend fun waitForDownload(downloadManager: DownloadManager, downloadId: Long): Boolean {
            return try {
                while (true) {
                    val query = DownloadManager.Query().setFilterById(downloadId)
                    downloadManager.query(query)?.use { cursor ->
                        if (cursor.moveToFirst()) {
                            when (cursor.getInt(cursor.getColumnIndexOrThrow(DownloadManager.COLUMN_STATUS))) {
                                DownloadManager.STATUS_SUCCESSFUL -> return true
                                DownloadManager.STATUS_FAILED -> return false
                            }
                        }
                    }
                    delay(500)
                }
                false
            } catch (e: Exception) {
                Log.e(TAG, "Error monitoring download: ${e.message}", e)
                false
            }
        }
    }

    class ForegroundService : Service() {
        override fun onStartCommand(intent: Intent, flags: Int, startId: Int): Int {
            Log.d(TAG, "onStartCommand")
            createNotificationChannel()
            val notification = NotificationCompat.Builder(this, packageName) // Use your channel ID
                .setOngoing(true)
                .setContentTitle("EloViewHome SDK Test Foreground Service")
                .setContentText("Running")
                .setPriority(NotificationCompat.PRIORITY_DEFAULT)
                .setCategory(NotificationCompat.CATEGORY_SERVICE)
                .build()
            startForeground(1, notification)

            val pkgName = intent.getStringExtra("pkgName")
            val pm = applicationContext.packageManager
            try {
                checkNotNull(pkgName)
                val activityIntent = pm.getLaunchIntentForPackage(pkgName)
                systemActivityViewModel.systemServiceManager?.startActivityFromForegroundService(
                    applicationContext,
                    activityIntent,
                    this
                )
                systemActivityViewModel.setOutput("Activity started from Foreground Service")
            } catch (e: java.lang.Exception) {
                e.printStackTrace()
            }
            return START_NOT_STICKY
        }

        override fun onBind(intent: Intent): IBinder? {
            return null
        }

        private fun createNotificationChannel() {
            val channelId = packageName
            val channelName = "My Channel Name"
            val channel =
                NotificationChannel(channelId, channelName, NotificationManager.IMPORTANCE_DEFAULT)
            val notificationManager = getSystemService(NotificationManager::class.java)
            notificationManager.createNotificationChannel(channel)
        }

        companion object {
            private const val TAG = "ForegroundService"
            lateinit var systemActivityViewModel: SystemActivityViewModel
        }
    }

}
