/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.systemActivity

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import android.net.Uri
import android.util.Log
import android.view.KeyEvent
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.DateRange
import androidx.compose.material3.Button
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.RadioButton
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TextField
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.material3.rememberTimePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.CommonDatePickerDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.CommonTimePickerDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.DoubleTextFieldSingleCheckBoxTwoButtonDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.DoubleTextFieldTwoButtonDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.MultiSelectDropDownTwoButtonDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.MyAlertDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.SimpleDialogWithAText
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.SingleDropDownSingleCheckBoxTwoButtonDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.SingleDropDownTwoButtonDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.SingleTextFieldSingleCheckBoxTwoButtonDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.SingleTextFieldSingleDropDownTwoButtonDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.SingleTextFieldTwoButtonDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.TwoButtonDialog
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.common.Util.convertUtcMillisToDate
import com.eloview.sdk.testapp.viewmodel.SystemActivityViewModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.util.Calendar

class SystemActivityDialogs {
    @Composable
    @Preview(showBackground = true)
    fun SilentShutdownDialogPreview() {
        SilentShutdownDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun DatePickerDialogPreview() {
        DatePickerDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun FactoryResetDialogPreview() {
        FactoryResetDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun ScreenWakeupDialogPreview() {
        ScreenWakeupDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun CaptureScreenshotDialogPreview() {
        CaptureScreenshotDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun SetInstallPermissionForSpecificAppDialogPreview() {
        SetInstallPermissionForSpecificAppDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun AccessibilityDialogPreview() {
        AccessibilityDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun SetLanguageDialogPreview() {
        SetLanguageDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun SetTimeZoneDialogPreview() {
        SetTimeZoneDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun SetDpiDialogPreview() {
        SetDpiDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun ReservedLogSpaceDialogPreview() {
        ReservedLogSpaceDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun SetPollIntervalDialogPreview() {
        SetPollIntervalDialog()
    }

    companion object {
        private const val TAG = "SystemActivityDialogs"

        @Composable
        fun SilentShutdownDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            ),
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSilentShutdownDialog(true)
            }
            if (systemActivityDetails.showSilentShutdownDialog) {
                MyAlertDialog(
                    title = { Text(text = "Silent Shutdown") },
                    content = {
                        Text(text = "Enter a delay in seconds for shutdown")
                        TextField(
                            value = systemActivityDetails.silentShutdownDelay.toString(),
                            onValueChange = { it ->
                                if (it.isEmpty()) {
                                    systemActivityViewModel.setSilentShutdownDelay(0)
                                } else {
                                    systemActivityViewModel.setSilentShutdownDelay(
                                        try {
                                            (it.filter { it.isDigit() }).toInt()
                                        } catch (_: NumberFormatException) {
                                            0
                                        }
                                    )
                                }
                            })
                    },
                    confirmButton =
                        {
                            Button(
                                onClick = {
                                    systemActivityViewModel.setShowSilentShutdownDialog(false)
                                    SystemActivityOnClickListeners.onSilentShutdownClick(
                                        systemActivityViewModel
                                    )
                                }) {
                                Text("Confirm")
                            }
                        },
                    dismissButton =
                        {
                            Button(
                                onClick = {
                                    systemActivityViewModel.setShowSilentShutdownDialog(false)
                                    systemActivityViewModel.setSilentShutdownDelay(3)
                                }) {
                                Text("Cancel")
                            }
                        },
                    onDismiss = {}
                )
            }
        }

        @OptIn(ExperimentalMaterial3Api::class)
        @Composable
        fun DatePickerDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowDatePickerDialog(true)
            }
            if (systemActivityDetails.showDatePickerDialog) {
                var showDatePicker by remember { mutableStateOf(false) }
                var showTimePicker by remember { mutableStateOf(false) }
                val datePickerState = rememberDatePickerState()
                val selectedDate = datePickerState.selectedDateMillis?.let {
                    convertUtcMillisToDate(it)
                } ?: ""
                val currentTime = Calendar.getInstance()
                val timePickerState = rememberTimePickerState(
                    initialHour = currentTime.get(Calendar.HOUR_OF_DAY),
                    initialMinute = currentTime.get(Calendar.MINUTE),
                    is24Hour = true
                )

                if (showTimePicker) {
                    if (selectedDate.isNotEmpty()) {
                        CommonTimePickerDialog(
                            timePickerState = timePickerState,
                            modifier = Modifier
                                .fillMaxWidth()
                                .fillMaxHeight(),
                            onConfirm = {
                                try {
                                    val cal = Calendar.getInstance()
                                    cal[Calendar.YEAR] = selectedDate.substring(6, 10).toInt()
                                    cal[Calendar.MONTH] = selectedDate.substring(0, 2).toInt() - 1
                                    cal[Calendar.DAY_OF_MONTH] =
                                        selectedDate.substring(3, 5).toInt()
                                    cal[Calendar.HOUR_OF_DAY] = timePickerState.hour
                                    cal[Calendar.MINUTE] = timePickerState.minute
                                    cal[Calendar.SECOND] = 0
                                    cal[Calendar.MILLISECOND] = 0
                                    SystemActivityOnClickListeners.onSetDateTimeClick(
                                        systemActivityViewModel,
                                        cal.time.time
                                    )
                                    systemActivityViewModel.setShowDatePickerDialog(false)
                                } catch (e: Exception) {
                                    Log.e(TAG, "Error setting date and time: $e")
                                    systemActivityViewModel.setOutput("Error setting date and time: $e")
                                }
                            },
                            onDismiss = {
                                systemActivityViewModel.setShowDatePickerDialog(false)
                            }
                        )
                    } else {
                        systemActivityViewModel.setShowDatePickerDialog(false)
                        systemActivityViewModel.setOutput("Please select a date first")
                    }
                }
                MyAlertDialog(
                    title = { Text(text = "Select Date") },
                    content = {
                        OutlinedTextField(
                            value = selectedDate,
                            onValueChange = {},
                            label = { Text("MM/DD/YYYY") },
                            readOnly = true,
                            trailingIcon = {
                                IconButton(onClick = {
                                    showDatePicker = true
                                }) {
                                    Icon(
                                        imageVector = Icons.Default.DateRange,
                                        contentDescription = "Select date"
                                    )
                                }
                            },
                            modifier = Modifier
                                .fillMaxWidth()
                                .height(64.dp)
                        )
                    },
                    confirmButton = {
                        Button(onClick = { showTimePicker = true }) {
                            Text("Confirm")
                        }
                    },
                    dismissButton = {
                        Button(onClick = {
                            systemActivityViewModel.setShowDatePickerDialog(false)
                        }) {
                            Text("Cancel")
                        }
                    },
                    onDismiss = {}
                )
                if (showDatePicker) {
                    CommonDatePickerDialog(
                        datePickerState = datePickerState,
                        onConfirm = {
                            showDatePicker = false
                        },
                        onDismiss = {
                            showDatePicker = false
                        }
                    )
                }
            }
        }

        @Composable
        fun FactoryResetDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowFactoryResetDialog(true)
            }
            if (systemActivityDetails.showFactoryResetDialog) {
                TwoButtonDialog(
                    title = "Factory Reset",
                    content = "Are you sure you want to factory reset the device?",
                    onConfirmClick = {
                        systemActivityViewModel.setShowFactoryResetDialog(false)
                        SystemActivityOnClickListeners.onFactoryResetClick(systemActivityViewModel)
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowFactoryResetDialog(false)
                    }
                )
            }
        }

        @Composable
        fun ScreenWakeupDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowScreenWakeUpDialog(true)
            }
            if (systemActivityDetails.showScreenWakeUpDialog) {
                var delayValue by remember { mutableStateOf("") }
                SingleTextFieldTwoButtonDialog(
                    title = "Enter a delay in seconds for Screen Wake Up",
                    textFieldValueHint = "30000",
                    textFieldValue = delayValue,
                    onValueChange = { delayValue = it },
                    onConfirmClick = {
                        if (delayValue.isBlank()) {
                            systemActivityViewModel.setOutput("Invalid Number")
                            systemActivityViewModel.setScreenWakeUpDelay(30000)
                            systemActivityViewModel.setShowScreenWakeUpDialog(false)
                            return@SingleTextFieldTwoButtonDialog
                        }
                        val delay = delayValue.toIntOrNull()
                        if (delay == null || delay <= 0) {
                            systemActivityViewModel.setOutput("Invalid Number")
                            systemActivityViewModel.setScreenWakeUpDelay(30000)
                            systemActivityViewModel.setShowScreenWakeUpDialog(false)
                            return@SingleTextFieldTwoButtonDialog
                        }
                        systemActivityViewModel.setScreenWakeUpDelay(delay)
                        systemActivityViewModel.setShowScreenWakeUpDialog(false)
                        SystemActivityOnClickListeners.onScreenWakeUpClick(systemActivityViewModel)
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowScreenWakeUpDialog(false)
                    }
                )
            }
        }

        @Composable
        fun CaptureScreenshotDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowCaptureScreenshotDialog(true)
            }
            if (systemActivityDetails.showCaptureScreenshotDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Enter a file name for the screenshot",
                    textFieldValueHint = "ic-screenshot",
                    textFieldValue = systemActivityDetails.screenShotPath,
                    onValueChange = {
                        try {
                            systemActivityViewModel.setScreenShotPath(it)
                        } catch (_: NumberFormatException) {
                            systemActivityViewModel.setScreenShotPath("ic-screenshot")
                        }
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowCaptureScreenshotDialog(false)
                        CoroutineScope(Dispatchers.Main).launch {
                            delay(100) // Delay for 1 seconds
                            SystemActivityOnClickListeners.onScreenShotClick(systemActivityViewModel)
                        }
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowCaptureScreenshotDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetInstallPermissionForSpecificAppDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetInstallPermissionForSpecificAppDialog(true)
            }
            if (systemActivityDetails.showSetInstallPermissionForSpecificAppDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Set Install Permission for Specific App",
                    textFieldValue = systemActivityDetails.selectedInstallPermissionForSpecificAppPackageName,
                    onValueChange = {
                        systemActivityViewModel.setSelectedInstallPermissionForSpecificAppPackageName(
                            it
                        )
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetInstallPermissionForSpecificAppDialog(
                            false
                        )
                        SystemActivityOnClickListeners.onSetInstallPermissionToSpecificAppsEnableClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetInstallPermissionForSpecificAppDialog(
                            false
                        )
                    }
                )
            }
        }

        @Composable
        fun GetInstallPermissionForSpecificAppDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowGetInstallPermissionForSpecificAppDialog(true)
            }
            if (systemActivityDetails.showGetInstallPermissionForSpecificAppDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Get Install Permission for Specific App",
                    textFieldValueHint = "com.example.app",
                    textFieldValue = systemActivityDetails.getInstallPermissionToSpecificApp,
                    onValueChange = {
                        systemActivityViewModel.setGetInstallPermissionToSpecificApp(
                            it
                        )
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowGetInstallPermissionForSpecificAppDialog(
                            false
                        )
                        SystemActivityOnClickListeners.onGetInstallPermissionToSpecificAppClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowGetInstallPermissionForSpecificAppDialog(
                            false
                        )
                    }
                )
            }
        }

        @Composable
        fun AccessibilityDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowAccessibilityDialog(true)
            }
            if (systemActivityDetails.showAccessibilityDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Set Accessibility",
                    textFieldValue = systemActivityDetails.accessibilityServiceComponentName,
                    onValueChange = {
                        systemActivityViewModel.setAccessibilityServiceComponentName(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowAccessibilityDialog(false)
                        SystemActivityOnClickListeners.onSetAccessibilityServiceEnableClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowAccessibilityDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetLanguageDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetLanguageDialog(true)
            }
            if (systemActivityDetails.showSetLanguageDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Enter Language Code",
                    textFieldValueHint = "en_US",
                    textFieldValue = systemActivityDetails.setLanguage,
                    onValueChange = {
                        systemActivityViewModel.setSetLanguage(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetLanguageDialog(false)
                        SystemActivityOnClickListeners.onSetLanguageClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetLanguageDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetTimeZoneDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetTimeZoneDialog(true)
            }
            if (systemActivityDetails.showSetTimeZoneDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Set TimeZone",
                    textFieldValueHint = "America/Los_Angeles",
                    textFieldValue = systemActivityDetails.setTimezone,
                    onValueChange = {
                        systemActivityViewModel.setSetTimezone(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetTimeZoneDialog(false)
                        SystemActivityOnClickListeners.onSetTimeZoneClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetTimeZoneDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetDpiDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetDpiDialog(true)
            }
            if (systemActivityDetails.showSetDpiDialog) {
                val dropDownValues = if (Util.is7inch() || Util.isM100()) {
                    arrayOf("Select DPI", "160", "240", "320", "400", "640")
                } else {
                    arrayOf("Select DPI", "160", "240", "320", "480", "640")
                }

                SingleDropDownTwoButtonDialog(
                    title = "Set DPI",
                    dropDownValues = dropDownValues,
                    onValueSelected = { item ->
                        run {
                            systemActivityViewModel.setSelectedDPI(item)
                        }
                    },
                    selectedValue = systemActivityDetails.selectedDPI,
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetDpiDialog(false)
                        SystemActivityOnClickListeners.onSetDpiClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetDpiDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetCFDDpiDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetCFDDpiDialog(true)
            }
            if (systemActivityDetails.showSetCFDDpiDialog) {
                SingleDropDownTwoButtonDialog(
                    title = "Set External display DPI",
                    dropDownValues = arrayOf("Select DPI", "160", "240", "320", "480", "640"),
                    onValueSelected = { item ->
                        run {
                            systemActivityViewModel.setSelectedCFDDPI(item)
                        }
                    },
                    selectedValue = systemActivityDetails.selectedCFDDPI,
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetCFDDpiDialog(false)
                        SystemActivityOnClickListeners.onSetCFDDpiClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetCFDDpiDialog(false)
                    }
                )
            }
        }

        @Composable
        fun ReservedLogSpaceDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowReservedLogSpaceDialog(true)
            }
            if (systemActivityDetails.showReservedLogSpaceDialog) {
                val logSpaceArray = arrayOf("144MB", "256MB", "512MB", "756MB", "1GB", "2GB")
                SingleDropDownTwoButtonDialog(
                    title = "Set Reserved Log Space",
                    dropDownValues = logSpaceArray,
                    onValueSelected = { item ->
                        run {
                            systemActivityViewModel.setReservedLogSpaceIndex(
                                logSpaceArray.indexOf(
                                    item
                                )
                            )
                        }
                    },
                    selectedValue = logSpaceArray[systemActivityDetails.selectedReservedLogSpaceIndex],
                    onConfirmClick = {
                        systemActivityViewModel.setShowReservedLogSpaceDialog(false)
                        SystemActivityOnClickListeners.onSetReservedLogSpaceClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowReservedLogSpaceDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetLockTypeDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetLockTypeDialog(true)

            }
            if (systemActivityDetails.showSetLockTypeDialog) {
                if (systemActivityViewModel.getServiceConnected()) {
                    val currentLockType = systemActivityViewModel.systemServiceManager?.lockType
                    if (systemActivityDetails.isGetLockeTypePressed == false) {
                        SimpleDialogWithAText("Get Lock Type First",
                            onDismiss = { systemActivityViewModel.setShowSetLockTypeDialog(false) }
                        )
                    } else {
                        if (currentLockType?.lowercase() == "pin" || currentLockType?.lowercase() == "password") {
                            systemActivityViewModel.setShowSetLockTypeDialog(false)
                            systemActivityViewModel.setShowSetLockTypeWithPasswordDialog(true)
                        } else {
                            SingleDropDownTwoButtonDialog(
                                title = "Set Lock Type",
                                dropDownValues = arrayOf("Swipe", "None"),
                                onValueSelected = { item ->
                                    run { systemActivityViewModel.setSelectedLockType(item) }
                                },
                                selectedValue = systemActivityDetails.selectedLockType,
                                onConfirmClick = {
                                    systemActivityViewModel.setShowSetLockTypeDialog(false)
                                    systemActivityViewModel.setIsGetLockTypePressed(false)
                                    SystemActivityOnClickListeners.onSetScreenLockTypeClick(
                                        systemActivityViewModel
                                    )
                                },
                                onCancelClick = {
                                    systemActivityViewModel.setShowSetLockTypeDialog(false)
                                    systemActivityViewModel.setIsGetLockTypePressed(false)
                                }
                            )
                        }

                    }
                }
            }
        }

        @Composable
        fun SetLockTypeWithPasswordDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetLockTypeWithPasswordDialog(true)
            }
            if (systemActivityDetails.showSetLockTypeWithPasswordDialog) {
                SingleTextFieldSingleDropDownTwoButtonDialog(
                    title = "Set Lock Type",
                    textFieldValue = systemActivityDetails.currentLockPassword ?: "",
                    onValueChange = {
                        systemActivityViewModel.setCurrentLockPassword(it)
                    },
                    dropDownValues = arrayOf("Swipe", "None"),
                    onValueSelected = { item ->
                        run { systemActivityViewModel.setSelectedLockType(item) }
                    },
                    selectedValue = systemActivityDetails.selectedLockType,
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetLockTypeWithPasswordDialog(false)
                        SystemActivityOnClickListeners.onSetScreenLockTypeClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetLockTypeWithPasswordDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetAdbPortDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetAdbPortDialog(true)
            }
            if (systemActivityDetails.showSetAdbPortDialog) {
                var adbPortValue by remember { mutableStateOf("") }
                SingleTextFieldTwoButtonDialog(
                    title = "Set ADB Port",
                    textFieldValueHint = "Choose between [1024, 65535]",
                    textFieldValue = adbPortValue,
                    onValueChange = { adbPortValue = it },
                    onConfirmClick = {
                        if (adbPortValue.isBlank()) {
                            systemActivityViewModel.setOutput("Invalid Port Number")
                            systemActivityViewModel.setShowSetAdbPortDialog(false)
                            return@SingleTextFieldTwoButtonDialog
                        }
                        val adbPort = adbPortValue.toIntOrNull()
                        if (adbPort == null || adbPort !in 1024..65535) {
                            systemActivityViewModel.setOutput("Invalid Port Number")
                            systemActivityViewModel.setShowSetAdbPortDialog(false)
                            return@SingleTextFieldTwoButtonDialog
                        }
                        systemActivityViewModel.setAdbPort(adbPort)
                        SystemActivityOnClickListeners.onSetAdbPortClick(systemActivityViewModel)
                        systemActivityViewModel.setShowSetAdbPortDialog(false)
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetAdbPortDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetDeviceBrightnessDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setDeviceBrightnessDialog(true)
            }
            if (systemActivityDetails.showSetDeviceBrightnessDialog) {
                var brightnessValue by remember { mutableStateOf("") }
                SingleTextFieldTwoButtonDialog(
                    title = "Set Device Brightness",
                    textFieldValueHint = "Choose between [0 - 80]",
                    textFieldValue = brightnessValue,
                    onValueChange = { brightnessValue = it },
                    onConfirmClick = {
                        if (brightnessValue.isBlank()) {
                            systemActivityViewModel.setOutput("Invalid Device Brightness")
                            systemActivityViewModel.setDeviceBrightnessDialog(false)
                            return@SingleTextFieldTwoButtonDialog
                        }
                        val brightness = brightnessValue.toIntOrNull()
                        if (brightness == null || brightness !in 0..100) {
                            systemActivityViewModel.setOutput("Invalid Device Brightness")
                            systemActivityViewModel.setDeviceBrightnessDialog(false)
                            return@SingleTextFieldTwoButtonDialog
                        }
                        systemActivityViewModel.setDeviceBrightness(brightness)
                        SystemActivityOnClickListeners.onSetDeviceBrightnessClick(systemActivityViewModel)
                        systemActivityViewModel.setDeviceBrightnessDialog(false)
                    },
                    onCancelClick = {
                        systemActivityViewModel.setDeviceBrightnessDialog(false)
                    }
                )
            }
        }

        @Composable
        fun ClearAppCacheAndUserDataDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowClearAppCacheAndUserDataDialog(true)
            }
            if (systemActivityDetails.showClearAppCacheAndUserDataDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Clear App Cache and User Data",
                    textFieldValueHint = "Comma separated list of package names to clear data",
                    textFieldValue = systemActivityDetails.clearAppCacheAndUserDataPackages,
                    onValueChange = {
                        systemActivityViewModel.setClearAppCacheAndUserDataPackages(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowClearAppCacheAndUserDataDialog(false)
                        SystemActivityOnClickListeners.onClearAppCacheAndUserDataClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowClearAppCacheAndUserDataDialog(false)
                    }
                )
            }
        }

        @Composable
        fun ClearAllAppCacheAndUserDataDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowClearAllAppCacheAndUserDataDialog(true)
            }
            if (systemActivityDetails.showClearAllAppCacheAndUserDataDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Clear All App Cache and User Data",
                    textFieldValueHint = "Comma separated list of package names to not clear data",
                    textFieldValue = systemActivityDetails.clearAllAppCacheAndUserDataPackages,
                    onValueChange = {
                        systemActivityViewModel.setClearAllAppCacheAndUserDataPackages(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowClearAllAppCacheAndUserDataDialog(false)
                        SystemActivityOnClickListeners.onClearAllAppCacheAndUserDataClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowClearAllAppCacheAndUserDataDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetSilentDefaultLauncherDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetSilentDefaultLauncherDialog(true)
            }
            if (systemActivityDetails.showSetSilentDefaultLauncherDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Set Silent Default Launcher",
                    textFieldValueHint = "com.example.launcher",
                    textFieldValue = systemActivityDetails.silentDefaultLauncher,
                    onValueChange = {
                        systemActivityViewModel.setSilentDefaultLauncher(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetSilentDefaultLauncherDialog(false)
                        SystemActivityOnClickListeners.onSetSilentDefaultLauncherClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetSilentDefaultLauncherDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetVolumeDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetVolumeDialog(true)
            }
            if (systemActivityDetails.showSetVolumeDialog) {
                var dropDownArray = arrayOf("All", "Alarm", "Call", "Media", "Notification")
                if (Util.is7inch() || Util.isM100()) {
                    dropDownArray = arrayOf("All", "Alarm", "Media", "Notification")
                }
                var volumeValue by remember { mutableStateOf("") }
                SingleTextFieldSingleDropDownTwoButtonDialog(
                    title = "Set Volume",
                    textFieldValueHint = "80",
                    textFieldValue = volumeValue,
                    onValueChange = { volumeValue = it },
                    dropDownValues = dropDownArray,
                    selectedValue = systemActivityDetails.selectedVolumeType,
                    onValueSelected = { item ->
                        run { systemActivityViewModel.setSelectedVolumeType(item) }
                    },
                    onConfirmClick = {
                        if (volumeValue.isBlank()) {
                            systemActivityViewModel.setOutput("No Input found so default Volume is 0")
                            systemActivityViewModel.setShowSetVolumeDialog(false)
                            return@SingleTextFieldSingleDropDownTwoButtonDialog
                        }
                        val volume = volumeValue.toIntOrNull()
                        if (volume == null || volume !in 0..100) {
                            systemActivityViewModel.setOutput("No Input found so default Volume is 0")
                            systemActivityViewModel.setShowSetVolumeDialog(false)
                            return@SingleTextFieldSingleDropDownTwoButtonDialog
                        }
                        systemActivityViewModel.setSetVolumeValue(volume)
                        systemActivityViewModel.setShowSetVolumeDialog(false)
                        SystemActivityOnClickListeners.onSetVolumeClick(systemActivityViewModel)
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetVolumeDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetWiFiMacAddressDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetWiFiMacAddressDialog(true)
            }
            if (systemActivityDetails.showSetWiFiMacAddressDialog) {
                SingleDropDownTwoButtonDialog(
                    title = "Set WiFi Mac Address",
                    dropDownValues = arrayOf("Use randomized MAC", "Use device MAC"),
                    selectedValue = systemActivityDetails.selectedWiFiMacAddress,
                    onValueSelected = { item ->
                        run { systemActivityViewModel.setSelectedWiFiMacAddress(item) }
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetWiFiMacAddressDialog(false)
                        SystemActivityOnClickListeners.onSetWiFiMacAddressClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetWiFiMacAddressDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetFontSizeDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetFontSizeDialog(true)
            }
            if (systemActivityDetails.showSetFontSizeDialog) {
                SingleDropDownTwoButtonDialog(
                    title = "Set Font Size",
                    dropDownValues = arrayOf("Default", "Small", "Large", "Largest"),
                    selectedValue = systemActivityDetails.selectedFontSize,
                    onValueSelected = { item ->
                        run { systemActivityViewModel.setSelectedFontSize(item) }
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetFontSizeDialog(false)
                        SystemActivityOnClickListeners.onSetFontSizeClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetFontSizeDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetDisplaySizeDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowDisplaySizeDialog(true)
            }
            if (systemActivityDetails.showDisplaySizeDialog) {
                SingleDropDownTwoButtonDialog(
                    title = "Set Display Size",
                    dropDownValues = arrayOf(
                        "Select a Display Size",
                        "Small",
                        "Default",
                        "Large",
                        "Larger",
                        "Largest"
                    ),
                    selectedValue = systemActivityDetails.selectedDisplaySize,
                    onValueSelected = { item ->
                        run { systemActivityViewModel.setSelectedDisplaySize(item) }
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowDisplaySizeDialog(false)
                        SystemActivityOnClickListeners.onSetDisplaySizeClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowDisplaySizeDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetBacklightTimeoutDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetBacklightTimeoutDialog(true)
            }
            if (systemActivityDetails.showSetBacklightTimeoutDialog) {
                SingleDropDownTwoButtonDialog(
                    title = "Set Backlight Timeout",
                    dropDownValues = arrayOf(
                        "Never",
                        "15 Seconds",
                        "30 Seconds",
                        "1 Minute",
                        "2 Minutes",
                        "5 Minutes",
                        "10 Minutes",
                        "30 Minutes"
                    ),
                    selectedValue = systemActivityDetails.selectedBacklightTimeout,
                    onValueSelected = { item ->
                        run { systemActivityViewModel.setSelectedBacklightTimeout(item) }
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetBacklightTimeoutDialog(false)
                        SystemActivityOnClickListeners.onSetBacklightTimeoutClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetBacklightTimeoutDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetPollIntervalDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetPollIntervalDialog(true)
            }
            if (systemActivityDetails.showSetPollIntervalDialog) {
                SingleTextFieldSingleCheckBoxTwoButtonDialog(
                    title = "Set Poll Interval",
                    textFieldValueHint = "900000",
                    textFieldValue = systemActivityDetails.ntpPollInterval,
                    onValueChange = {
                        systemActivityViewModel.setNtpPollInterval(it)
                    },
                    checkBoxValue = systemActivityDetails.noRebootPollIntervalChecked,
                    onCheckBoxChange = {
                        systemActivityViewModel.setNoRebootPollIntervalChecked(it)
                    },
                    checkBoxText = "No Reboot",
                    onConfirmClick = {
                        try {
                            val interval = systemActivityViewModel.getNtpPollInterval()
                            if (interval.isEmpty()) {
                                Toast.makeText(
                                    systemActivityViewModel.applicationContext,
                                    "Please enter the interval value",
                                    Toast.LENGTH_SHORT
                                ).show()
                                systemActivityViewModel.setOutput("Please enter the interval value")
                            } else if (interval.toLong() < 900000) {
                                Toast.makeText(
                                    systemActivityViewModel.applicationContext,
                                    "Enter Valid Interval",
                                    Toast.LENGTH_SHORT
                                ).show()
                                systemActivityViewModel.setOutput("Enter Valid Interval")
                            } else {
                                systemActivityViewModel.setShowSetPollIntervalDialog(false)
                                SystemActivityOnClickListeners.onSetNtpPollIntervalClick(
                                    systemActivityViewModel
                                )
                            }
                        } catch (_: NumberFormatException) {
                            Toast.makeText(
                                systemActivityViewModel.applicationContext,
                                "Invalid interval value",
                                Toast.LENGTH_SHORT
                            ).show()
                            systemActivityViewModel.setOutput("Invalid interval value")
                        }
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetPollIntervalDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetPasswordDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetPasswordDialog(true)
            }
            if (systemActivityDetails.showSetPasswordDialog) {
                DoubleTextFieldSingleCheckBoxTwoButtonDialog(
                    title = "Set Password",
                    textField2KeyboardType = KeyboardType.Text,
                    isPinField = false,
                    textFieldValue1 = systemActivityDetails.oldPassword,
                    textFieldValue2 = systemActivityDetails.newPassword,
                    onValueChange1 = {
                        systemActivityViewModel.setOldPassword(it)
                    },
                    onValueChange2 = {
                        systemActivityViewModel.setNewPassword(it)
                    },
                    checkBoxValue = systemActivityDetails.askForPasswordOnBoot,
                    onCheckBoxChange = {
                        systemActivityViewModel.setAskForPasswordOnBoot(it)
                    },
                    checkBoxText = "Enable Auto Unlock",
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetPasswordDialog(false)
                        SystemActivityOnClickListeners.onSetPasswordClick(
                            systemActivityViewModel
                        )
                        systemActivityViewModel.setOldPassword("")
                        systemActivityViewModel.setNewPassword("")
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetPasswordDialog(false)
                        systemActivityViewModel.setOldPassword("")
                        systemActivityViewModel.setNewPassword("")
                    }
                )
            }
        }

        @Composable
        fun SetPinDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetPinDialog(true)
            }
            if (systemActivityDetails.showSetPinDialog) {
                DoubleTextFieldSingleCheckBoxTwoButtonDialog(
                    title = "Set Pin",
                    textField2KeyboardType = KeyboardType.NumberPassword,
                    isPinField = true,
                    textFieldValue2Hint = "Enter Pin",
                    textFieldValue1 = systemActivityDetails.oldPin,
                    textFieldValue2 = systemActivityDetails.newPin,
                    onValueChange1 = {
                        systemActivityViewModel.setOldPin(it)
                    },
                    onValueChange2 = {
                        systemActivityViewModel.setNewPin(it)
                    },
                    checkBoxValue = systemActivityDetails.askForPinOnBoot,
                    onCheckBoxChange = {
                        systemActivityViewModel.setAskForPinOnBoot(it)
                    },
                    checkBoxText = "Enable Auto Unlock",
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetPinDialog(false)
                        SystemActivityOnClickListeners.onSetPinClick(systemActivityViewModel)
                        systemActivityViewModel.setOldPin("")
                        systemActivityViewModel.setNewPin("")
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetPinDialog(false)
                        systemActivityViewModel.setOldPin("")
                        systemActivityViewModel.setNewPin("")
                    }
                )
            }
        }

        @Composable
        fun SetDeviceOwnerDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetDeviceOwnerDialog(true)
            }
            if (systemActivityDetails.showSetDeviceOwnerDialog) {
                DoubleTextFieldTwoButtonDialog(
                    title = "Set Device Owner",
                    textFieldValue1Hint = "com.example.deviceowner",
                    textFieldValue2Hint = "com.example.deviceowner.DeviceOwnerReceiver",
                    textFieldValue1 = systemActivityDetails.deviceOwnerPackage,
                    onValueChange1 = {
                        systemActivityViewModel.setDeviceOwnerPackage(it)
                    },
                    textFieldValue2 = systemActivityDetails.deviceOwnerReceiver,
                    onValueChange2 = {
                        systemActivityViewModel.setDeviceOwnerReceiver(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetDeviceOwnerDialog(false)
                        SystemActivityOnClickListeners.onSetDeviceOwnerClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetDeviceOwnerDialog(false)
                        systemActivityViewModel.setDeviceOwnerPackage("")
                    }
                )
            }
        }

        @Composable
        fun SetResolutionDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetResolutionDialog(true)
            }
            if (systemActivityDetails.showSetResolutionDialog) {
                var resolutionWidth by remember { mutableStateOf("") }
                var resolutionHeight by remember { mutableStateOf("") }
                DoubleTextFieldTwoButtonDialog(
                    title = "Set Resolution",
                    textFieldValue1Hint = "Width (pixels)",
                    textFieldValue2Hint = "Height (pixels)",
                    textFieldValue1 = resolutionWidth,
                    onValueChange1 = { resolutionWidth = it },
                    textFieldValue2 = resolutionHeight,
                    onValueChange2 = { resolutionHeight = it },
                    onConfirmClick = {
                        if (resolutionWidth.isBlank() || resolutionHeight.isBlank()) {
                            systemActivityViewModel.setOutput("Please enter integer values for width and height")
                            systemActivityViewModel.setShowSetResolutionDialog(false)
                            return@DoubleTextFieldTwoButtonDialog
                        }
                        val width = resolutionWidth.toIntOrNull()
                        val height = resolutionHeight.toIntOrNull()
                        if (width == null || height == null) {
                            systemActivityViewModel.setOutput("Please enter valid integer values")
                            systemActivityViewModel.setShowSetResolutionDialog(false)
                            return@DoubleTextFieldTwoButtonDialog
                        }
                        systemActivityViewModel.setResolutionWidth(width)
                        systemActivityViewModel.setResolutionHeight(height)
                        SystemActivityOnClickListeners.onSetResolutionClick(systemActivityViewModel)
                        systemActivityViewModel.setShowSetResolutionDialog(false)
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetResolutionDialog(false)
                    }
                )
            }
        }

        @Composable
        fun StartActivityFromForegroundServiceDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowStartActivityFromForegroundServiceDialog(true)
            }
            if (systemActivityDetails.showStartActivityFromForegroundServiceDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Start Activity from Foreground Service",
                    textFieldValueHint = "com.example.foregroundservice",
                    textFieldValue = systemActivityDetails.foregroundServicePackageName,
                    onValueChange = {
                        systemActivityViewModel.setForegroundServicePackageName(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowStartActivityFromForegroundServiceDialog(
                            false
                        )
                        SystemActivityOnClickListeners.onStartActivityFromForegroundServiceClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowStartActivityFromForegroundServiceDialog(
                            false
                        )
                    }
                )
            }
        }

        @Composable
        fun SetSmallestWidthDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetSmallestWidthDialog(true)
            }
            if (systemActivityDetails.showSetSmallestWidthDialog) {
                var smallestWidthValue by remember { mutableStateOf("") }
                SingleTextFieldTwoButtonDialog(
                    title = "Set Smallest Width",
                    textFieldValueHint = "800",
                    textFieldValue = smallestWidthValue,
                    onValueChange = { smallestWidthValue = it },
                    onConfirmClick = {
                        if (smallestWidthValue.isBlank()) {
                            systemActivityViewModel.setOutput("No Input found, default width is 800")
                            systemActivityViewModel.setSmallestPixelWidth(800)
                            systemActivityViewModel.setShowSetSmallestWidthDialog(false)
                            return@SingleTextFieldTwoButtonDialog
                        }
                        val width = smallestWidthValue.toIntOrNull()
                        if (width == null || width <= 0) {
                            systemActivityViewModel.setOutput("Invalid input, default width is 800")
                            systemActivityViewModel.setSmallestPixelWidth(800)
                            systemActivityViewModel.setShowSetSmallestWidthDialog(false)
                            return@SingleTextFieldTwoButtonDialog
                        }
                        systemActivityViewModel.setSmallestPixelWidth(width)
                        systemActivityViewModel.setShowSetSmallestWidthDialog(false)
                        SystemActivityOnClickListeners.onSetSmallestWidthClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetSmallestWidthDialog(false)
                        systemActivityViewModel.setSmallestPixelWidth(null)
                    }
                )
            }
        }

        @Composable
        fun SetNtpUrlDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetNtpUrlDialog(true)
            }
            if (systemActivityDetails.showSetNtpUrlDialog) {
                SingleTextFieldSingleCheckBoxTwoButtonDialog(
                    title = "Set NTP URL",
                    textFieldValueHint = "Enter Ntp URL",
                    textFieldValue = systemActivityDetails.ntpUrl,
                    onValueChange = {
                        systemActivityViewModel.setNtpUrl(it)
                    },
                    checkBoxValue = systemActivityDetails.noRebootNtpUrlChecked,
                    onCheckBoxChange = {
                        systemActivityViewModel.setNoRebootNtpUrlChecked(it)
                    },
                    checkBoxText = "No Reboot",
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetNtpUrlDialog(false)
                        SystemActivityOnClickListeners.onSetNtpUrlClick(systemActivityViewModel)
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetNtpUrlDialog(false)
                        systemActivityViewModel.setNtpUrl("")
                    }
                )
            }
        }

        @Composable
        fun SetHostNameDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetHostNameDialog(true)
            }
            if (systemActivityDetails.showSetHostNameDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Set Host Name",
                    textFieldValue = systemActivityDetails.hostName,
                    textFieldValueHint = "Enter HostName",
                    onValueChange = {
                        systemActivityViewModel.setHostName(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetHostNameDialog(false)
                        SystemActivityOnClickListeners.onSetHostNameClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetHostNameDialog(false)
                        systemActivityViewModel.setHostName("")
                    }
                )
            }
        }

        @Composable
        fun SetGetSystemPropertyDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowGetSystemPropertyDialog(true)
            }
            if (systemActivityDetails.showGetSystemPropertyDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Get System Property",
                    textFieldValueHint = "persist.sys.elo.serial",
                    textFieldValue = systemActivityDetails.getSystemProperty,
                    onValueChange = {
                        systemActivityViewModel.setGetSystemProperty(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowGetSystemPropertyDialog(false)
                        SystemActivityOnClickListeners.onGetSystemPropertyClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowGetSystemPropertyDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetGetMD5ChecksumDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowGetMD5ChecksumDialog(true)
            }
            if (systemActivityDetails.showGetMD5ChecksumDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Get MD5 CheckSum",
                    textFieldValueHint = "/sdcard/example/file.txt",
                    textFieldValue = systemActivityDetails.getMD5ChecksumFilePath,
                    onValueChange = {
                        systemActivityViewModel.setGetMD5ChecksumFilePath(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowGetMD5ChecksumDialog(false)
                        SystemActivityOnClickListeners.onGetMD5ChecksumClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowGetMD5ChecksumDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetMediaProjectionPermissionForPackageDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowMediaProjectionPermissionForPackageDialog(true)
            }
            if (systemActivityDetails.showSetMediaProjectionPermissionForPackageDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Set Package Names (Multiple Packages Should Be Separated By Comma)",
                    textFieldValue = systemActivityDetails.mediaProjectionPermissionPackage,
                    textFieldValueHint = "pkgName1,pkgName2",
                    onValueChange = {
                        systemActivityViewModel.setMediaProjectionPermissionPackage(it)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowMediaProjectionPermissionForPackageDialog(
                            false
                        )
                        SystemActivityOnClickListeners.onSetMediaProjectionPermission(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowMediaProjectionPermissionForPackageDialog(
                            false
                        )
                        systemActivityViewModel.setMediaProjectionPermissionPackage("")
                    }
                )
            }
        }

        @Composable
        fun SetKeyListenerDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetKeyListenerDialog(true)
            }
            if (systemActivityDetails.showSetKeyListenerDialog) {
                val keys = arrayOf(
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_A),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_B),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_C),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_D),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_HOME),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_POWER),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_VOLUME_UP),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_VOLUME_DOWN),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_BACK)
                )
                MultiSelectDropDownTwoButtonDialog(
                    title = "Set Key Listener",
                    dropDownValues = keys,
                    selectedValues = systemActivityDetails.selectedKeyListenerValues,
                    onOptionAdded = {
                        val list = systemActivityDetails.selectedKeyListenerValues + it
                        systemActivityViewModel.setSelectedKeyListenerValues(list)
                    },
                    onOptionRemoved = {
                        val list =
                            systemActivityDetails.selectedKeyListenerValues.filter { item -> item != it }
                        systemActivityViewModel.setSelectedKeyListenerValues(list.toTypedArray())
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetKeyListenerDialog(false)
                        SystemActivityOnClickListeners.onSetKeyListenerClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setSelectedKeyListenerValues(emptyArray())
                        systemActivityViewModel.setShowSetKeyListenerDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetKeyInterceptorDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetKeyInterceptionDialog(true)
            }
            if (systemActivityDetails.showSetKeyInterceptionDialog) {
                val keys = arrayOf(
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_A),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_B),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_C),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_D),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_HOME),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_POWER),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_VOLUME_UP),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_VOLUME_DOWN),
                    KeyEvent.keyCodeToString(KeyEvent.KEYCODE_BACK)
                )
                MultiSelectDropDownTwoButtonDialog(
                    title = "Set Key Interceptor",
                    dropDownValues = keys,
                    selectedValues = systemActivityDetails.selectedKeyListenerValues,
                    onOptionAdded = {
                        val list = systemActivityDetails.selectedKeyListenerValues + it
                        systemActivityViewModel.setSelectedKeyListenerValues(list)
                    },
                    onOptionRemoved = {
                        val list =
                            systemActivityDetails.selectedKeyListenerValues.filter { item -> item != it }
                        systemActivityViewModel.setSelectedKeyListenerValues(list.toTypedArray())
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setShowSetKeyInterceptionDialog(false)
                        SystemActivityOnClickListeners.onSetKeyInterceptorClick(
                            systemActivityViewModel
                        )
                    },
                    onCancelClick = {
                        systemActivityViewModel.setShowSetKeyInterceptionDialog(false)
                    }
                )
            }
        }

        @Composable
        fun SetOrientationDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setShowSetOrientationDialog(true)
            }
            if (systemActivityDetails.showSetOrientationDialog) {
                if (!Util.isBackPack()) {
                    SingleDropDownSingleCheckBoxTwoButtonDialog(
                        title = "Set Orientation",
                        dropDownValues = arrayOf(
                            "auto-rotate",
                            "landscape",
                            "portrait",
                            "landscape-reverse",
                            "portrait-reverse"
                        ),
                        selectedValue = systemActivityDetails.selectedNonBPOrientation,
                        onValueSelected = { item ->
                            run { systemActivityViewModel.setSelectedNonBPOrientation(item) }
                        },
                        checkBoxValue = systemActivityDetails.noRebootOrientationChecked,
                        onCheckBoxChange = {
                            systemActivityViewModel.setNoRebootOrientationChecked(it)
                        },
                        checkBoxText = "No Reboot",
                        onConfirmClick = {
                            systemActivityViewModel.setShowSetOrientationDialog(false)
                            SystemActivityOnClickListeners.onSetOrientationNonBackpackClick(
                                systemActivityViewModel
                            )
                        },
                        onCancelClick = {
                            systemActivityViewModel.setShowSetOrientationDialog(false)
                        },
                        checkBoxEnabled = false
                    )
                } else {
                    SingleDropDownSingleCheckBoxTwoButtonDialog(
                        title = "Set Orientation",
                        dropDownValues = arrayOf(
                            "landscape (0)",
                            "portrait (90)",
                            "landscape-reverse (180)",
                            "portrait-reverse (270)"
                        ),
                        selectedValue = systemActivityDetails.selectedOrientation,
                        onValueSelected = { item ->
                            run { systemActivityViewModel.setSelectedOrientation(item) }
                        },
                        checkBoxValue = systemActivityDetails.noRebootOrientationChecked,
                        onCheckBoxChange = {
                            systemActivityViewModel.setNoRebootOrientationChecked(it)
                        },
                        checkBoxText = "No Reboot",
                        onConfirmClick = {
                            systemActivityViewModel.setShowSetOrientationDialog(false)
                            SystemActivityOnClickListeners.onSetOrientationBackpackClick(
                                systemActivityViewModel
                            )
                        },
                        onCancelClick = {
                            systemActivityViewModel.setShowSetOrientationDialog(false)
                        }
                    )
                }
            }
        }

        @Composable
        fun DownloadAndConvertZipFileToByteArrayDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setDownloadAndConvertZipFileToByteArrayDialog(true)
            }
            if (systemActivityDetails.showDownloadAndConvertZipFileToByteArrayDialog) {
                var selectedOption by remember { mutableStateOf<Int?>(null) }
                var filePath by rememberSaveable { mutableStateOf("") }
                val context = LocalContext.current
                var selectedUri : Uri = Uri.EMPTY
                val launcher = rememberLauncherForActivityResult(
                    contract = ActivityResultContracts.StartActivityForResult()
                ) { result ->

                    if (result.resultCode == Activity.RESULT_OK) {
                        val selectedFile: Uri? = result.data?.data
                        try {
                            if (selectedFile != null) {
                                val filePathFromUri = Util.getRealPathFromUri(context, selectedFile)
                                if (filePathFromUri != null && filePathFromUri.endsWith(".zip")) {
                                    filePath = filePathFromUri
                                    selectedUri = selectedFile
                                } else {
                                    Toast.makeText(
                                        systemActivityViewModel.applicationContext,
                                        "Please select a valid zip file",
                                        Toast.LENGTH_SHORT
                                    ).show()
                                }
                            }
                        } catch (e: Exception) {
                            e.printStackTrace()
                            systemActivityViewModel.setOutput("Error selecting file: ${e.message}")
                        }
                    }
                }

                Dialog(
                    onDismissRequest = {
                        systemActivityViewModel.setDownloadAndConvertZipFileToByteArrayDialog(false)
                    },
                    properties = DialogProperties(usePlatformDefaultWidth = false)
                ) {
                    Surface(
                        shape = MaterialTheme.shapes.medium,
                        color = MaterialTheme.colorScheme.background,
                        modifier = Modifier
                            .fillMaxWidth(0.9f)
                            .padding(16.dp)
                    ) {
                        Column(
                            modifier = Modifier
                                .padding(16.dp)
                                .verticalScroll(rememberScrollState())
                        ) {
                            Text(
                                "Select File Options:",
                                style = MaterialTheme.typography.titleMedium
                            )
                            Spacer(modifier = Modifier.height(8.dp))

                            listOf(
                                "Select File From Assets",
                                "Select Remote URL",
                                "Select File URL"
                            ).forEachIndexed { index, label ->
                                Row(verticalAlignment = Alignment.CenterVertically) {
                                    RadioButton(
                                        selected = selectedOption == index,
                                        onClick = {
                                            selectedOption = index
                                            filePath = ""
                                            selectedUri = Uri.EMPTY
                                        }
                                    )
                                    Text(text = label)
                                }
                            }

                            Spacer(modifier = Modifier.height(8.dp))

                            if (selectedOption == 2) {
                                Row(
                                    verticalAlignment = Alignment.CenterVertically,
                                    modifier = Modifier.fillMaxWidth()
                                ) {
                                    OutlinedTextField(
                                        value = filePath,
                                        onValueChange = { /* Read-only */ },
                                        singleLine = true,
                                        label = { Text("Selected file path") },
                                        readOnly = true,
                                        enabled = false,
                                        modifier = Modifier.weight(1f)
                                    )
                                    Spacer(modifier = Modifier.width(8.dp))
                                    Button(
                                        onClick = {
                                            val intent = Intent(Intent.ACTION_OPEN_DOCUMENT).apply {
                                                addCategory(Intent.CATEGORY_OPENABLE)
                                                type = "application/zip"
                                            }
                                            try {
                                                launcher.launch(intent)
                                            } catch (ex: ActivityNotFoundException) {
                                                systemActivityViewModel.setOutput("Please install a File Manager")
                                            }
                                        }
                                    ) {
                                        Text("Browse")
                                    }
                                }
                            } else {
                                OutlinedTextField(
                                    value = filePath,
                                    onValueChange = { filePath = it },
                                    singleLine = true,
                                    label = { Text("Enter file path") },
                                    modifier = Modifier.fillMaxWidth()
                                )
                            }

                            Spacer(modifier = Modifier.height(8.dp))
                            Row(
                                modifier = Modifier.fillMaxWidth(),
                                horizontalArrangement = Arrangement.End
                            ) {
                                TextButton(onClick = {
                                    systemActivityViewModel.setDownloadAndConvertZipFileToByteArrayDialog(
                                        false
                                    )
                                }) {
                                    Text("Cancel")
                                }
                                TextButton(onClick = {
                                    val isValid = when (selectedOption) {
                                        0 -> {
                                            if (filePath.startsWith("file:///android_asset/")) {
                                                true
                                            } else {
                                                Toast.makeText(
                                                    systemActivityViewModel.applicationContext,
                                                    "Please enter File Path",
                                                    Toast.LENGTH_SHORT
                                                ).show()
                                                false
                                            }
                                        }

                                        1 -> {
                                            if (filePath.startsWith("https://") || filePath.startsWith("http://")) {
                                                true
                                            } else {
                                                Toast.makeText(
                                                    systemActivityViewModel.applicationContext,
                                                    "Please enter Remote URL file path",
                                                    Toast.LENGTH_SHORT
                                                ).show()
                                                false
                                            }
                                        }

                                        2 -> {
                                            if (filePath.endsWith(".zip")) {
                                                true
                                            } else {
                                                Toast.makeText(
                                                    systemActivityViewModel.applicationContext,
                                                    "Please select a valid zip file",
                                                    Toast.LENGTH_SHORT
                                                ).show()
                                                false
                                            }
                                        }

                                        else -> {
                                            Toast.makeText(
                                                systemActivityViewModel.applicationContext,
                                                "You have to select at least one option",
                                                Toast.LENGTH_SHORT
                                            ).show()
                                            false
                                        }
                                    }
                                    if (isValid) {
                                        SystemActivityOnClickListeners.onDownloadAndConvertZipFileToByteArray(
                                            systemActivityViewModel,
                                            filePath,
                                            selectedUri
                                        )
                                        systemActivityViewModel.setDownloadAndConvertZipFileToByteArrayDialog(
                                            false
                                        )
                                    }
                                }) {
                                    Text("Confirm")
                                }
                            }
                        }
                    }
                }
            }
        }


        @Composable
        fun SetExternalOrientationDialog(
            systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
                applicationContext = Activity(),
                true
            )
        ) {
            val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()

            if (SystemActivityViewModel.isPreview) {
                systemActivityViewModel.setExternalOrientationDialog(true)
            }

            if (systemActivityDetails.showSetExternalOrientationDialog) {
                SingleDropDownTwoButtonDialog(
                    title = "Set External Screen Orientation",
                    dropDownValues = arrayOf("0", "90", "180", "270"),
                    selectedValue = systemActivityDetails.selectedExternalScreenOrientation,
                    onValueSelected = { item ->
                        systemActivityViewModel.setSelectedExternalOrientation(item)
                    },
                    onConfirmClick = {
                        systemActivityViewModel.setExternalOrientationDialog(false)
                        systemActivityViewModel.setRebootWarningDialog(true)
                    },
                    onCancelClick = {
                        systemActivityViewModel.setExternalOrientationDialog(false)
                    }
                )
            }

            if (systemActivityDetails.showRebootWarningDialog) {
                MyAlertDialog(
                    title = { Text(text = "External Screen Orientation") },
                    content = {
                        Text(text = "This change will only take effect after a reboot. Reboot the device now?")
                    },
                    confirmButton = {
                        Button(onClick = {
                            systemActivityViewModel.setRebootWarningDialog(false)
                            SystemActivityOnClickListeners.onSetExternalOrientationClick(
                                systemActivityViewModel
                            )
                        }) {
                            Text("Confirm")
                        }
                    },
                    dismissButton = {
                        Button(onClick = {
                            systemActivityViewModel.setRebootWarningDialog(false)
                        }) {
                            Text("Cancel")
                        }
                    },
                    onDismiss = {}
                )
            }
        }
    }
}
