/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.systemActivity

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.view.inputmethod.InputMethodManager
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Scaffold
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.common.CommonComposables.Companion.AccountManagerBtn
import com.eloview.sdk.testapp.common.CommonComposables.Companion.BottomButtonBar
import com.eloview.sdk.testapp.common.CommonComposables.Companion.CommonButton
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloOutputText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.common.CommonComposables.Companion.RowTextWithRadio
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.common.Util.is7inch
import com.eloview.sdk.testapp.common.Util.isM100
import com.eloview.sdk.testapp.common.Util.isM51
import com.eloview.sdk.testapp.systemActivity.SystemActivity.Companion.TAG
import com.eloview.sdk.testapp.systemActivity.SystemActivity.Companion.isGMS
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.viewmodel.SystemActivityViewModel

class SystemActivity : ComponentActivity() {
    companion object {
        const val TAG = "SystemActivity"
        var isGMS = true
    }

    private lateinit var mSystemActivityViewModel: SystemActivityViewModel

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        mSystemActivityViewModel = SystemActivityViewModel(
            applicationContext,
            false
        )
        savedInstanceState?.getString("output")?.let { mSystemActivityViewModel.setOutput(it) }
        val activityResult =
            registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
                SystemActivityOnClickListeners.onSetNetworkConfigurationClickActivityResult(
                    mSystemActivityViewModel,
                    result
                )
                Log.d(TAG, "onActivityResult: $result")
            }
        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    SystemActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        systemActivityViewModel = mSystemActivityViewModel,
                        activity = this,
                        activityResult,
                    )
                }
            }
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        outState.putString("output", mSystemActivityViewModel.getOutput())
        super.onSaveInstanceState(outState)
    }
}

//@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
@Composable
fun SystemActivityScreen(
    modifier: Modifier = Modifier,
    systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: SystemActivity = SystemActivity(),
    activityResult: ActivityResultLauncher<Intent>
) {
    val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
    val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()

    SystemDialogs(systemActivityViewModel)

    if (globalDetails.authorized && !systemActivityDetails.serviceConnected) {
        systemActivityViewModel.bindService()
    }
    if (systemActivityDetails.serviceConnected) {
        isGMS =
            systemActivityViewModel.systemServiceManager?.readSystemProperty("ro.vendor.build.oem.prjname")
                .equals("GMS")
    }
    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
            modifier = Modifier
                .padding(
                    top = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                    start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                    end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp
                )
                .fillMaxSize()
                .wrapContentHeight()
        ) {

            EloScreenTitleText("System API")

            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier
                    .padding(top = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {
                TopButtonsComposable(systemActivityViewModel, activity)
                EnableDisableColumn(systemActivityViewModel)
                BottomButtonsComposable(activityResult, systemActivityViewModel, activity)
            }
        }
    }
}

@Composable
@Preview(showBackground = true, widthDp = 640, heightDp = 2000)
fun EnableDisableColumn(
    systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
        applicationContext = Activity(),
        true
    )
) {
    Column(
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxSize()
            .wrapContentHeight()
    ) {
        HorizontalDivider()
        val isServiceConnected = systemActivityViewModel.getServiceConnected()
        var stdAndroid by remember { mutableStateOf<String?>(null) }
        LaunchedEffect(isServiceConnected) {
            if (isServiceConnected) {
                stdAndroid = systemActivityViewModel.systemServiceManager
                    ?.readSystemProperty("persist.sys.stdandroid")
            }
        }
        RowTextWithRadio(
            text1 = "Charging Sound and Vibration",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetChargeSoundAndVibrationClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetChargeSoundAndVibrationClick(
                    systemActivityViewModel,
                    false
                )
            }
        )

        RowTextWithRadio(
            text1 = "Location",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onLocationEnableClick(systemActivityViewModel, true)
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onLocationEnableClick(systemActivityViewModel, false)
            })
        RowTextWithRadio(
            text1 = "Home Button",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onHomeButtonEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onHomeButtonEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Recent Button",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onRecentButtonEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onRecentButtonEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Wifi Scanning",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onWifiScanningEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onWifiScanningEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Bluetooth Scanning",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onBluetoothScanningEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onBluetoothScanningEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Touch Sound",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onTouchSoundEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onTouchSoundEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Volume Panel Show More",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onVolumePanelShowMoreEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onVolumePanelShowMoreEnableClick(
                    systemActivityViewModel,
                    false
                )
            })

        RowTextWithRadio(
            text1 = "Text Editor Assist",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onTextEditorAssistMenuEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onTextEditorAssistMenuEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Power Button",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onPowerButtonEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onPowerButtonEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(text1 = "Accessibility", onClickEnable = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setSetAccessibilityServiceEnabled(true)
            systemActivityViewModel.setShowAccessibilityDialog(true)
        }, onClickDisable = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setSetAccessibilityServiceEnabled(false)
            systemActivityViewModel.setShowAccessibilityDialog(true)
        })
        RowTextWithRadio(
            text1 = "Auto Date/Time",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetAutoDateTimeClick(systemActivityViewModel, true)
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetAutoDateTimeClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Use 24 Hour Format",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetUse24HourFormatClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetUse24HourFormatClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Navigation Bar",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onNavigationBarEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onNavigationBarEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        if (Util.isHandHeld() || isM100() || isM51()) {
            val isScanBCREnabled = remember { mutableStateOf(false) }

            RowTextWithRadio(
                text1 = "Set BCR",
                onClickEnable = {
                    systemActivityViewModel.setOutput("")
                    SystemActivityOnClickListeners.onBCREnableClick(systemActivityViewModel, true)
                    isScanBCREnabled.value = true
                },
                onClickDisable = {
                    systemActivityViewModel.setOutput("")
                    SystemActivityOnClickListeners.onBCREnableClick(systemActivityViewModel, false)
                    isScanBCREnabled.value = false
                })


            if (isScanBCREnabled.value) {
                RowTextWithRadio(
                    text1 = "Scan BCR",
                    onClickEnable = {
                        systemActivityViewModel.setOutput("")
                        SystemActivityOnClickListeners.onBCRTriggerClick(
                            systemActivityViewModel,
                            true
                        )

                    },
                    onClickDisable = {
                        systemActivityViewModel.setOutput("")
                        SystemActivityOnClickListeners.onBCRTriggerClick(
                            systemActivityViewModel,
                            false
                        )
                        isScanBCREnabled.value = false
                    })
            }

        }
        RowTextWithRadio(
            text1 = "Status Bar",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetStatusBarEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetStatusBarEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Status Bar with Swipe Lock",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetLockStatusBarEnable(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetLockStatusBarEnable(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Status Bar Visible",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetStatusBarVisibilityEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetStatusBarVisibilityEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Show Password",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetShowPasswordEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetShowPasswordEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Grant Auto All App Permission",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onGrantAutoAllAppPermissionEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onGrantAutoAllAppPermissionEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Grant Auto USB Permission",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onGrantAutoUsbPermissionEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onGrantAutoUsbPermissionEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Share Text Selection",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetShareTextSelectionEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetShareTextSelectionEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "WebView Hardware Access",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetWebViewHardwareAccessEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetWebViewHardwareAccessEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "StarIO WebView Hardware Access",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetStarWebViewHardwareAccessEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetStarWebViewHardwareAccessEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Auto Idle Mode",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetAutoIdleModeEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetAutoIdleModeEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Idle Mode Tile",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetIdleModeTileEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetIdleModeTileEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Set Microphone",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetMicrophoneEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetMicrophoneEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Install Permission to All Apps",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetInstallPermissionToAllAppsEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetInstallPermissionToAllAppsEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Install Permission to Specific App",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                systemActivityViewModel.setSelectedInstallPermissionForSpecificAppEnabled(true)
                systemActivityViewModel.setShowSetInstallPermissionForSpecificAppDialog(true)
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                systemActivityViewModel.setSelectedInstallPermissionForSpecificAppEnabled(false)
                systemActivityViewModel.setShowSetInstallPermissionForSpecificAppDialog(true)
            })
        RowTextWithRadio(
            text1 = "Multi-Client IME",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onMultiClientIMEEnableClick(
                    systemActivityViewModel,
                    enable = true,
                    reboot = true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onMultiClientIMEEnableClick(
                    systemActivityViewModel,
                    enable = false,
                    reboot = true
                )
            })
        RowTextWithRadio(
            text1 = "Multi-Client IME (no reboot)",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onMultiClientIMEEnableClick(
                    systemActivityViewModel,
                    enable = true,
                    reboot = false
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onMultiClientIMEEnableClick(
                    systemActivityViewModel,
                    enable = false,
                    reboot = false
                )
            })
        RowTextWithRadio(
            text1 = "Notification",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetNotificationEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetNotificationEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "App Mode",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onAppModeEnableClick(systemActivityViewModel, true)
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onAppModeEnableClick(systemActivityViewModel, false)
            })
        RowTextWithRadio(
            text1 = "Sticky EloView Mode",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetStickyEloViewModeEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetStickyEloViewModeEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "ADB",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetADBEnableClick(systemActivityViewModel, true)
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetADBEnableClick(systemActivityViewModel, false)
            })
        RowTextWithRadio(
            text1 = "Logs",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetLogsEnableClick(systemActivityViewModel, true)
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetLogsEnableClick(systemActivityViewModel, false)
            })
        RowTextWithRadio(
            text1 = "Dark Mode",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetDarkModeEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetDarkModeEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        if (Util.isM60()) {
            RowTextWithRadio(
                text1 = "Firmware Lock",
                onClickEnable = {
                    systemActivityViewModel.setOutput("")
                    if (!systemActivityViewModel.getFirmwareLocked()) {
                        SystemActivityOnClickListeners.onSetFirmwareLockEnableClick(
                            systemActivityViewModel,
                            true
                        )
                    } else {
                        Log.d(TAG, "Can not Lock Firmware, as firmware is already locked")
                    }
                },
                onClickDisable = {
                    systemActivityViewModel.setOutput("")
                    SystemActivityOnClickListeners.onSetFirmwareLockEnableClick(
                        systemActivityViewModel,
                        false
                    )
                })
        }
        RowTextWithRadio(
            text1 = "Copy/Paste",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetCopyPasteEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetCopyPasteEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Screen Lock Sound",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetScreenLockSoundEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetScreenLockSoundEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Charge Sounds/Vibration",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetChargeSoundAndVibrationClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetChargeSoundAndVibrationClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Bluetooth",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetBluetoothEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetBluetoothEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Show Taps",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetShowTapsEnableClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetShowTapsEnableClick(
                    systemActivityViewModel,
                    false
                )
            })
        if (isGMS) {
            RowTextWithRadio(
                text1 = "Play Protect",
                onClickEnable = {
                    systemActivityViewModel.setOutput("")
                    SystemActivityOnClickListeners.onEnablePlayProtectClick(
                        systemActivityViewModel,
                        true
                    )
                },
                onClickDisable = {
                    systemActivityViewModel.setOutput("")
                    SystemActivityOnClickListeners.onEnablePlayProtectClick(
                        systemActivityViewModel,
                        false
                    )
                })
        }
        RowTextWithRadio(
            text1 = "BackLight",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onEnableBackLightClick(systemActivityViewModel, true)
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onEnableBackLightClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "WiFi",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onEnableWiFiClick(systemActivityViewModel, true)
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onEnableWiFiClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "HotSpot Accessibility",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onEnableHotSpotAccessibilityClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onEnableHotSpotAccessibilityClick(
                    systemActivityViewModel,
                    false
                )
            })
        RowTextWithRadio(
            text1 = "Show insets on swipe",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetShowInsetsClick(systemActivityViewModel, true)
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetShowInsetsClick(systemActivityViewModel, false)
            })
        RowTextWithRadio(
            text1 = "Media Projection Permission For All Apps",
            onClickEnable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetMediaProjectionPermissionAllAppsClick(
                    systemActivityViewModel,
                    true
                )
            },
            onClickDisable = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetMediaProjectionPermissionAllAppsClick(
                    systemActivityViewModel,
                    false
                )
            })
        HorizontalDivider()
    }
}

@Composable
@Preview(showBackground = true, widthDp = 640)
fun TopButtonsComposable(
    systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: Activity = Activity()
) {
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxSize()
    ) {
        AccountManagerBtn(activity)
        EloWideButton(
            text = "Capture ScreenShot",
            onClick = {
                systemActivityViewModel.setOutput("")
                systemActivityViewModel.setShowCaptureScreenshotDialog(true)
            })
        EloWideButton(
            text = "Set Orientation",
            onClick = {
                systemActivityViewModel.setOutput("")
                systemActivityViewModel.setShowSetOrientationDialog(true)
            })
        EloWideButton(text = "Get Orientation", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetOrientationClick(systemActivityViewModel)
        })


        if (isM100() || is7inch()) {
            CommonButton(
                text = "Set External Orientation",
                onClick = {
                    systemActivityViewModel.setExternalOrientationDialog(true)
                })
            CommonButton(text = "Get External Orientation", onClick = {
                SystemActivityOnClickListeners.onGetExternalOrientationClick(systemActivityViewModel)
            })
        }


        EloWideButton(text = "Factory Reset", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowFactoryResetDialog(true)
        })
        EloWideButton(text = "Silent Reboot", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onSilentRebootClick(systemActivityViewModel)
        })
        EloWideButton(text = "Silent ShutDown", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSilentShutdownDialog(true)
        })
        EloWideButton(text = "Set Date Time", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowDatePickerDialog(true)
        })
        EloWideButton(text = "Set Reserved Log Space", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowReservedLogSpaceDialog(true)
        })
        EloWideButton(text = "Get Reserved Log Space", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetReservedLogSpaceClick(systemActivityViewModel)
        })
    }
}

@Composable
fun BottomButtonsComposable(
    activityResult: ActivityResultLauncher<Intent>,
    systemActivityViewModel: SystemActivityViewModel = SystemActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: Activity = Activity()
) {
    val systemActivityDetails by systemActivityViewModel.systemActivityState.collectAsState()
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxSize()
    ) {
        EloWideButton(text = "Set ADB Port", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetAdbPortDialog(true)
        })
        EloWideButton(text = "Network Configuration", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onSetNetworkConfigurationClick(
                systemActivityViewModel,
                activityResult
            )
        })
        EloWideButton(text = "Clear App Cache and Userdata", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowClearAppCacheAndUserDataDialog(true)
        })
        EloWideButton(text = "Clear All App Cache and Userdata", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowClearAllAppCacheAndUserDataDialog(true)
        })
        EloWideButton(text = "Silent Set Default Launcher", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetSilentDefaultLauncherDialog(true)
        })
        EloWideButton(text = "Get Sticky EloView Mode", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetStickyEloViewModeClick(systemActivityViewModel)
        })
        EloWideButton(text = "Get Navigation Bar", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetNavigationBarClick(systemActivityViewModel)
        })
        EloWideButton(text = "Get Status Bar", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetStatusBarClick(systemActivityViewModel)
        })
        EloWideButton(text = "Get Grant All App Permission", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetGrantAllAppPermissionClick(
                systemActivityViewModel
            )
        })
        EloWideButton(text = "Get Grant Auto USB Permission", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetGrantAutoUSBPermissionClick(
                systemActivityViewModel
            )
        })
        EloWideButton(text = "Get Multi-Client IME", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetMultiClientImeClick(systemActivityViewModel)
        })
        EloWideButton(text = "Get Install Permission to Specific App", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowGetInstallPermissionForSpecificAppDialog(true)
        })
        EloWideButton(text = "Get Lock Type", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetLockTypeClick(systemActivityViewModel)
            systemActivityViewModel.setIsGetLockTypePressed(true)

        })
        EloWideButton(text = "Set Lock Type", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetLockTypeDialog(true)
        })
        EloWideButton(text = "Set Backlight Timeout", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetBacklightTimeoutDialog(true)
        })
        EloWideButton(text = "Set Display Size", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowDisplaySizeDialog(true)
        })
        EloWideButton(text = "Set Font Size", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetFontSizeDialog(true)
        })
        EloWideButton(text = "Set WiFi Mac Address", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetWiFiMacAddressDialog(true)
        })
        EloWideButton(text = "Set Volume", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetVolumeDialog(true)
        })
        EloWideButton(text = "Set DPI", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetDpiDialog(true)
        })
        EloWideButton(text = "Get DPI", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetDpiClick(systemActivityViewModel)
        })
        EloWideButton(text = "Set External display DPI", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetCFDDpiDialog(true)
        })
        EloWideButton(text = "Get External display DPI", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetCFDDpiClick(systemActivityViewModel)
        })
        EloWideButton(text = "Set Screen Wake-Up", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowScreenWakeUpDialog(true)
        })
        EloWideButton(text = "Get Backlight Timer", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetBacklightTimerClick(systemActivityViewModel)
        })
        EloWideButton(text = "Set TimeZone", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetTimeZoneDialog(true)
        })
        EloWideButton(text = "Get TimeZone", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetTimeZoneClick(systemActivityViewModel)
        })
        EloWideButton(text = "Set Language", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetLanguageDialog(true)
        })
        EloWideButton(text = "Get Language", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetLanguageClick(systemActivityViewModel)
        })
        EloWideButton(text = "Set Poll Interval", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetPollIntervalDialog(true)
        })
        EloWideButton(text = "Get Poll Interval", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetNtpPollIntervalClick(systemActivityViewModel)
        })
        EloWideButton(text = "Set Password", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetPasswordDialog(true)
        })
        EloWideButton(text = "Set PIN", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetPinDialog(true)
        })
        if (isGMS) {
            EloWideButton(text = "Get Play Protect", onClick = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onGetPlayProtectClick(systemActivityViewModel)
            })
        }
        EloWideButton(text = "Set Resolution", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetResolutionDialog(true)
        })
        EloWideButton(text = "Get Resolution", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetResolutionClick(systemActivityViewModel)
        })
        EloWideButton(text = "Get Device Owner Status", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetDeviceOwnerStatusClick(systemActivityViewModel)
        })
        EloWideButton(text = "Set Device Owner", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetDeviceOwnerDialog(true)
        })
        EloWideButton(text = "Start Activity From Foreground Service", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowStartActivityFromForegroundServiceDialog(true)
        })
        EloWideButton(text = "Stop Foreground Service", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onStopForegroundServiceClick(
                systemActivityViewModel
            )
        })
        EloWideButton(text = "Set Smallest Width", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetSmallestWidthDialog(true)
        })
        EloWideButton(text = "Get Smallest Width", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetSmallestWidthClick(systemActivityViewModel)
        })


        EloWideButton(text = "Set Key Listener", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetKeyListenerDialog(true)
        })

        EloWideButton(text = "Set Key Interceptor", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetKeyInterceptionDialog(true)
        })

        EloWideButton(text = "Unset Key Listener/Interceptor", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onUnsetKeyListenerInterceptorClick(
                systemActivityViewModel
            )
        })
        EloWideButton(text = "Set NTP URL", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetNtpUrlDialog(true)
        })
        EloWideButton(text = "Get NTP URL", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetNtpUrlClick(systemActivityViewModel)
        })
        EloWideButton(text = "Set HostName", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowSetHostNameDialog(true)
        })
        EloWideButton(text = "Get HostName", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetHostNameClick(systemActivityViewModel)
        })
        if (!Util.isValueDevice()) {
            EloWideButton(text = "Secure Processor Firmware Upgrade", onClick = {
                systemActivityViewModel.setOutput("")
                if (!systemActivityViewModel.getFirmwareLocked()) {
                    SystemActivityOnClickListeners.onSecureProcessorFirmwareUpgradeClick(
                        systemActivityViewModel
                    )
                } else {
                    Log.d(
                        TAG,
                        "Can not use Secure Processor Firmware Upgrade as firmware is locked"
                    )
                }
            })
        }
        var aospImeEnabled = false
        var gboardImeEnabled = false
        try {
            val imm =
                systemActivityViewModel.applicationContext.getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
            for (i in 0 until imm.inputMethodList.size) {
                Log.d(TAG, "Input Method: " + imm.inputMethodList[i])
                if ("com.android.inputmethod.latin" == imm.inputMethodList[i].packageName) {
                    aospImeEnabled = true
                }
                if ("com.google.android.inputmethod.latin" == imm.inputMethodList[i].packageName) {
                    gboardImeEnabled = true
                }
            }
        } catch (e: Exception) {
            Log.d(TAG, "Exception: " + e.message)
        }
        if (aospImeEnabled && gboardImeEnabled) {
            EloWideButton(text = "Set AOSP Keyboard", onClick = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetAospKeyboardClick(
                    systemActivityViewModel
                )
            })
            EloWideButton(text = "Set Google Keyboard", onClick = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetGboardKeyboardClick(
                    systemActivityViewModel
                )
            })

        } else if (aospImeEnabled) {
            EloWideButton(text = "Set AOSP Keyboard", onClick = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetAospKeyboardClick(
                    systemActivityViewModel
                )
            })
        } else if (gboardImeEnabled) {
            EloWideButton(text = "Set Google Keyboard", onClick = {
                systemActivityViewModel.setOutput("")
                SystemActivityOnClickListeners.onSetGboardKeyboardClick(
                    systemActivityViewModel
                )
            })
        }
        EloWideButton(text = "Get Auto Idle Mode Status", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetAutoIdleModeStatusClick(
                systemActivityViewModel
            )
        })
        EloWideButton(text = "Get Touch Sound Status", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetTouchSoundStatusClick(systemActivityViewModel)
        })
        EloWideButton(text = "Get Screen Lock Sound Status", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetScreenLockSoundStatusClick(
                systemActivityViewModel
            )
        })
        EloWideButton(text = "Get HotSpot Accessibility Enabled Status", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetHotSpotAccessibilityEnabledStatusClick(
                systemActivityViewModel
            )
        })
        EloWideButton(text = "Get Charging Sound and Vibration", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetChargeSoundAndVibrationClick(
                systemActivityViewModel
            )
        })
        EloWideButton(text = "Get show insets on swipe", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetShowInsetsClick(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Set Internal Audio", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onSetInternalAudioClick(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Set External Audio", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onSetExternalAudioClick(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Verify Valid File Signature", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onVerifyValidFileSignature(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Verify Invalid File Signature", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onVerifyInvalidFileSignature(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Get System Property", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowGetSystemPropertyDialog(true)
        })

        EloWideButton(text = "Get MD5 Checksum", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowGetMD5ChecksumDialog(true)
        })

        EloWideButton(text = "Set Media Projection Permission For Specific Package", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setShowMediaProjectionPermissionForPackageDialog(true)
        })

        EloWideButton(text = "Get Media Projection Permission For All Apps", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetMediaProjectionPermissionAllAppsClick(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Get Media Projection Permission for allowed packages", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetMediaProjectionPermissionClick(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Download And Convert Zip File to Byte Array", onClick = {
            systemActivityViewModel.setDownloadAndConvertZipFileToByteArrayDialog(true)
        })

        EloWideButton(text = "Write Signed File to Storage", onClick = {
            SystemActivityOnClickListeners.writeSignedFileToStorage(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Verify File Signature", onClick = {
            SystemActivityOnClickListeners.onVerifyFileSignature(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Read Elopay File Loader Partition", onClick = {
            SystemActivityOnClickListeners.onReadElopayFileLoaderPartition(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Enable Secure Whitelist Mode", onClick = {
            SystemActivityOnClickListeners.onEnableSecureWhitelistMode(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Get Secure Whitelist Mode Status", onClick = {
            SystemActivityOnClickListeners.onGetSecureWhitelistModeStatus(
                systemActivityViewModel
            )
        })

        EloWideButton(text = "Set Device Brightness", onClick = {
            systemActivityViewModel.setOutput("")
            systemActivityViewModel.setDeviceBrightnessDialog(true)
        })

        EloWideButton(text = "Get Device Brightness", onClick = {
            systemActivityViewModel.setOutput("")
            SystemActivityOnClickListeners.onGetDeviceBrightnessClick(systemActivityViewModel)
        })
        EloOutputText(systemActivityDetails.output)

        BottomButtonBar(activity) { systemActivityViewModel.setOutput("") }
    }
}

@Composable
fun SystemDialogs(systemActivityViewModel: SystemActivityViewModel) {
    SystemActivityDialogs.SetLockTypeWithPasswordDialog(systemActivityViewModel)
    SystemActivityDialogs.SetInstallPermissionForSpecificAppDialog(systemActivityViewModel)
    SystemActivityDialogs.GetInstallPermissionForSpecificAppDialog(systemActivityViewModel)
    SystemActivityDialogs.AccessibilityDialog(systemActivityViewModel)
    SystemActivityDialogs.DatePickerDialog(systemActivityViewModel)
    SystemActivityDialogs.SilentShutdownDialog(systemActivityViewModel)
    SystemActivityDialogs.FactoryResetDialog(systemActivityViewModel)
    SystemActivityDialogs.ScreenWakeupDialog(systemActivityViewModel)
    SystemActivityDialogs.CaptureScreenshotDialog(systemActivityViewModel)
    SystemActivityDialogs.SetLanguageDialog(systemActivityViewModel)
    SystemActivityDialogs.SetTimeZoneDialog(systemActivityViewModel)
    SystemActivityDialogs.SetDpiDialog(systemActivityViewModel)
    SystemActivityDialogs.SetCFDDpiDialog(systemActivityViewModel)
    SystemActivityDialogs.ReservedLogSpaceDialog(systemActivityViewModel)
    SystemActivityDialogs.SetLockTypeDialog(systemActivityViewModel)
    SystemActivityDialogs.SetAdbPortDialog(systemActivityViewModel)
    SystemActivityDialogs.ClearAppCacheAndUserDataDialog(systemActivityViewModel)
    SystemActivityDialogs.ClearAllAppCacheAndUserDataDialog(systemActivityViewModel)
    SystemActivityDialogs.SetSilentDefaultLauncherDialog(systemActivityViewModel)
    SystemActivityDialogs.SetVolumeDialog(systemActivityViewModel)
    SystemActivityDialogs.SetWiFiMacAddressDialog(systemActivityViewModel)
    SystemActivityDialogs.SetFontSizeDialog(systemActivityViewModel)
    SystemActivityDialogs.SetDisplaySizeDialog(systemActivityViewModel)
    SystemActivityDialogs.SetBacklightTimeoutDialog(systemActivityViewModel)
    SystemActivityDialogs.SetPollIntervalDialog(systemActivityViewModel)
    SystemActivityDialogs.SetPasswordDialog(systemActivityViewModel)
    SystemActivityDialogs.SetPinDialog(systemActivityViewModel)
    SystemActivityDialogs.SetDeviceOwnerDialog(systemActivityViewModel)
    SystemActivityDialogs.SetResolutionDialog(systemActivityViewModel)
    SystemActivityDialogs.StartActivityFromForegroundServiceDialog(systemActivityViewModel)
    SystemActivityDialogs.SetSmallestWidthDialog(systemActivityViewModel)
    SystemActivityDialogs.SetNtpUrlDialog(systemActivityViewModel)
    SystemActivityDialogs.SetHostNameDialog(systemActivityViewModel)
    SystemActivityDialogs.SetKeyListenerDialog(systemActivityViewModel)
    SystemActivityDialogs.SetKeyInterceptorDialog(systemActivityViewModel)
    SystemActivityDialogs.SetOrientationDialog(systemActivityViewModel)
    SystemActivityDialogs.SetExternalOrientationDialog(systemActivityViewModel)
    SystemActivityDialogs.SetGetSystemPropertyDialog(systemActivityViewModel)
    SystemActivityDialogs.SetGetMD5ChecksumDialog(systemActivityViewModel)
    SystemActivityDialogs.SetMediaProjectionPermissionForPackageDialog(systemActivityViewModel)
    SystemActivityDialogs.DownloadAndConvertZipFileToByteArrayDialog(systemActivityViewModel)
    SystemActivityDialogs.SetDeviceBrightnessDialog(systemActivityViewModel)
}
