/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.peripheralActivity

import android.util.Log
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.viewmodel.PeripheralActivityViewModel
import org.json.JSONObject

class PeripheralActivityOnClickListeners {
    companion object {
        const val TAG = "PeripheralActivityOnClickListeners"

        fun onGetDevicePortStatusClick(peripheralActivityViewModel: PeripheralActivityViewModel) {
            if (peripheralActivityViewModel.getServiceConnected()) {
                peripheralActivityViewModel.setOutput("Usb Port Status : ${peripheralActivityViewModel.peripheralServiceManager?.usbPorts}")
            } else {
                peripheralActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetDevicePortsStatusClick(
            peripheralActivityViewModel: PeripheralActivityViewModel,
            maxUsbPorts: Int,
            maxExternalUsbPorts: Int
        ) {
            if (peripheralActivityViewModel.getServiceConnected()) {
                peripheralActivityViewModel.setOutput("Setting Device Ports...")
                try {
                    val json = JSONObject()
                    // All Elo Android devices have at least one USB port.
                    json.put(
                        "port1",
                        createPortJsonObject(peripheralActivityViewModel.getUsb1Enabled())
                    )

                    for (i in 2..maxUsbPorts) {
                        val portEnabled = when (i) {
                            2 -> peripheralActivityViewModel.getUsb2Enabled()
                            3 -> peripheralActivityViewModel.getUsb3Enabled()
                            4 -> peripheralActivityViewModel.getUsb4Enabled()
                            5 -> peripheralActivityViewModel.getUsb5Enabled()
                            6 -> peripheralActivityViewModel.getUsb6Enabled()
                            // No Elo Android device has more than 6 built-in USB Ports
                            else -> throw Exception("Invalid port number")
                        }
                        json.put("port$i", createPortJsonObject(portEnabled))
                    }

                    val extJson = JSONObject()
                    // All Elo External Hub devices have at least one USB port.
                    extJson.put(
                        "port1",
                        createPortJsonObject(peripheralActivityViewModel.getExtUsb1Enabled())
                    )

                    for (i in 2..maxExternalUsbPorts) {
                        val portEnabled = when (i) {
                            2 -> peripheralActivityViewModel.getExtUsb2Enabled()
                            3 -> peripheralActivityViewModel.getExtUsb3Enabled()
                            4 -> peripheralActivityViewModel.getExtUsb4Enabled()
                            5 -> peripheralActivityViewModel.getExtUsb5Enabled()
                            6 -> peripheralActivityViewModel.getExtUsb6Enabled()
                            // No Elo Android External Hub device has more than 6 USB Ports
                            else -> throw Exception("Invalid external port number")
                        }
                        extJson.put("port$i", createPortJsonObject(portEnabled))
                    }
                    if (maxExternalUsbPorts > 0) {
                        json.put("ext", extJson)
                    }

                    if (peripheralActivityViewModel.peripheralServiceManager?.setUSBPorts(json.toString()) == true) {
                        peripheralActivityViewModel.setOutput("Device Ports Set to $json")
                    } else {
                        peripheralActivityViewModel.setOutput("Failed to set Device Ports")
                    }
                } catch (e: Exception) {
                    Log.e(TAG, "onSetDevicePortsClick: ${e.message}")
                    peripheralActivityViewModel.setOutput("Failed to set Device Ports: ${e.message}")
                }
            } else {
                peripheralActivityViewModel.setOutput("Service not connected")
            }
        }

        private fun createPortJsonObject(enabled: Boolean): JSONObject {
            val portJson = JSONObject()
            portJson.put("value", if (enabled) "Enabled" else "Disabled")
            portJson.put("type", "USB")
            return portJson
        }

        fun onStartUsbMonitorClick(peripheralActivityViewModel: PeripheralActivityViewModel) {
            if (peripheralActivityViewModel.getServiceConnected()) {
                peripheralActivityViewModel.setOutput("Starting USB Monitor...")
                peripheralActivityViewModel.peripheralServiceManager?.startUsbMonitoring()
                peripheralActivityViewModel.setOutput("USB Monitor Started...")
            } else {
                peripheralActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onStopUsbMonitorClick(peripheralActivityViewModel: PeripheralActivityViewModel) {
            if (peripheralActivityViewModel.getServiceConnected()) {
                peripheralActivityViewModel.setOutput("Stopping USB Monitor...")
                peripheralActivityViewModel.peripheralServiceManager?.stopUsbMonitoring()
                peripheralActivityViewModel.setOutput("USB Monitor Stopped...")
            } else {
                peripheralActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetUsbInfoClick(peripheralActivityViewModel: PeripheralActivityViewModel) {
            if (peripheralActivityViewModel.getServiceConnected()) {
                val jObject: JSONObject? =
                    peripheralActivityViewModel.peripheralServiceManager?.attachedPeripheralDetails
                if (jObject == null) {
                    peripheralActivityViewModel.setOutput("Failed to get USB Info")
                    return
                }
                peripheralActivityViewModel.setOutput("USB Info: $jObject")
            } else {
                peripheralActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetAllUsbAudioDevicesClick(peripheralActivityViewModel: PeripheralActivityViewModel) {
            if (Util.isBackPack()) {
                peripheralActivityViewModel.setOutput("Backpack devices not supported")
                return
            }
            if (peripheralActivityViewModel.getServiceConnected()) {
                val devices =
                    peripheralActivityViewModel.peripheralServiceManager?.allUsbAudioDevices
                if (devices.isNullOrEmpty()) {
                    peripheralActivityViewModel.setOutput("Failed to get all USB Audio Devices: No devices found")
                    return
                }
                peripheralActivityViewModel.setOutput("All USB Audio Devices: $devices")
            } else {
                peripheralActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetUsbAudioDeviceClick(peripheralActivityViewModel: PeripheralActivityViewModel) {
            if (Util.isBackPack()) {
                peripheralActivityViewModel.setOutput("Backpack devices not supported")
                return
            }
            if (peripheralActivityViewModel.getServiceConnected()) {
                val device = peripheralActivityViewModel.peripheralServiceManager?.usbAudioDevice
                if (device == null || device < 0) {
                    peripheralActivityViewModel.setOutput("Failed to get USB Audio Device")
                    return
                }
                val devices =
                    peripheralActivityViewModel.peripheralServiceManager.allUsbAudioDevices
                for (i in devices.indices) {
                    if (devices[i].isNullOrEmpty()) {
                        continue
                    }
                    val item = JSONObject(devices[i])
                    val deviceName = item.getString("DeviceName")
                    val cardNumber = item.getInt("CardNumber")
                    if (cardNumber == device) {
                        peripheralActivityViewModel.setOutput("USB Audio Device: [$deviceName, $cardNumber]")
                        return
                    }
                }
                peripheralActivityViewModel.setOutput("No Audio Device Found for Card Number: $device")
            } else {
                peripheralActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetUsbAudioDeviceClick(peripheralActivityViewModel: PeripheralActivityViewModel) {
            if (Util.isBackPack()) {
                peripheralActivityViewModel.setOutput("Backpack devices not supported")
                return
            }
            if (peripheralActivityViewModel.getServiceConnected()) {
                val cardNumber = peripheralActivityViewModel.getSelectedUsbAudioDevice()
                if (peripheralActivityViewModel.peripheralServiceManager?.setUsbAudioDevice(
                        cardNumber
                    ) == true
                ) {
                    peripheralActivityViewModel.setOutput("USB Audio Device Set to $cardNumber")
                } else {
                    peripheralActivityViewModel.setOutput("Failed to set USB Audio Device to $cardNumber")
                }
            } else {
                peripheralActivityViewModel.setOutput("Service not connected")
            }
        }
    }
}