/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.peripheralActivity

import android.app.Activity
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.Scaffold
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.common.CommonComposables.Companion.AccountManagerBtn
import com.eloview.sdk.testapp.common.CommonComposables.Companion.BottomButtonBar
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloOutputText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.common.CommonComposables.Companion.RowTextAndSwitchContent
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.viewmodel.PeripheralActivityViewModel

class PeripheralActivity : ComponentActivity() {
    private lateinit var mPeripheralActivityViewModel: PeripheralActivityViewModel

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        mPeripheralActivityViewModel = PeripheralActivityViewModel(
            applicationContext,
            false
        )
        if (savedInstanceState != null) {
            // persist output across UI reconfigurations
            savedInstanceState.getString("output")
                ?.let { mPeripheralActivityViewModel.setOutput(it) }
        }
        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    PeripheralActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        peripheralActivityViewModel = mPeripheralActivityViewModel,
                        activity = this
                    )
                }
            }
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        outState.putString("output", mPeripheralActivityViewModel.getOutput())
        super.onSaveInstanceState(outState)
    }
}

@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
@Composable
fun PeripheralActivityScreen(
    modifier: Modifier = Modifier,
    peripheralActivityViewModel: PeripheralActivityViewModel = PeripheralActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: PeripheralActivity = PeripheralActivity()
) {
    val peripheralActivityDetails by peripheralActivityViewModel.peripheralActivityState.collectAsState()
    val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()

    PeripheralDialogs(peripheralActivityViewModel)

    if (globalDetails.authorized && !peripheralActivityDetails.serviceConnected) {
        peripheralActivityViewModel.bindService()
    }

    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
            modifier = Modifier
                .padding(
                    top = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                    start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                    end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp
                )
                .fillMaxSize()
                .wrapContentHeight()
        ) {

            EloScreenTitleText("Peripheral API")


            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier
                    .padding(top = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {
                TopButtonsComposable(peripheralActivityViewModel, activity)
            }
        }
    }
}

@Composable
//@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
fun TopButtonsComposable(
    peripheralActivityViewModel: PeripheralActivityViewModel = PeripheralActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: Activity = Activity()
) {
    val peripheralActivityDetails by peripheralActivityViewModel.peripheralActivityState.collectAsState()
    val maxUsbPorts =
        if (PeripheralActivityViewModel.isPreview || peripheralActivityViewModel.peripheralServiceManager == null) {
            6
        } else {
            peripheralActivityViewModel.peripheralServiceManager.maxUSBPort
        }
    val maxExternalUsbPorts =
        if (PeripheralActivityViewModel.isPreview || peripheralActivityViewModel.peripheralServiceManager == null) {
            6
        } else {
            peripheralActivityViewModel.peripheralServiceManager.getMaxExtUSBPort(
                peripheralActivityViewModel.applicationContext
            )
        }

    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxSize()
    ) {
        AccountManagerBtn(activity)
        EloWideButton(
            text = "Get Device Port Status",
            onClick = {
                peripheralActivityViewModel.setOutput("")
                PeripheralActivityOnClickListeners.onGetDevicePortStatusClick(
                    peripheralActivityViewModel
                )
            }
        )
        if (maxUsbPorts == 0 && maxExternalUsbPorts == 0) {
            peripheralActivityViewModel.setOutput("This device does not support Enable/Disable USB Ports")
        } else {
            RowTextAndSwitchContent(
                mainLabel = "USB Port 1",
                label = "Disable / Enable",
                state = peripheralActivityDetails.usb1Enabled,
                onStateChange = {
                    peripheralActivityViewModel.setOutput("")
                    peripheralActivityViewModel.setUsb1Enabled(
                        it
                    )
                }, verticalDivider = false,
                center = true
            )

            if (maxUsbPorts > 1) {
                RowTextAndSwitchContent(
                    mainLabel = "USB Port 2",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.usb2Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setUsb2Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
            if (maxUsbPorts > 2) {
                RowTextAndSwitchContent(
                    mainLabel = "USB Port 3",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.usb3Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setUsb3Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
            if (maxUsbPorts > 3) {
                RowTextAndSwitchContent(
                    mainLabel = "USB Port 4",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.usb4Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setUsb4Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
            if (maxUsbPorts > 4) {
                RowTextAndSwitchContent(
                    mainLabel = "USB Port 5",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.usb5Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setUsb5Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
            if (maxUsbPorts > 5) {
                RowTextAndSwitchContent(
                    mainLabel = "USB Port 6",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.usb6Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setUsb6Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
        }
        if (maxExternalUsbPorts > 0) {
            RowTextAndSwitchContent(
                mainLabel = "External USB Port 1",
                label = "Disable / Enable",
                state = peripheralActivityDetails.extUsb1Enabled,
                onStateChange = {
                    peripheralActivityViewModel.setOutput("")
                    peripheralActivityViewModel.setExtUsb1Enabled(
                        it
                    )
                }, verticalDivider = false,
                center = true
            )

            if (maxExternalUsbPorts > 1) {
                RowTextAndSwitchContent(
                    mainLabel = "External USB Port 2",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.extUsb2Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setExtUsb2Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
            if (maxExternalUsbPorts > 2) {
                RowTextAndSwitchContent(
                    mainLabel = "External USB Port 3",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.extUsb3Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setExtUsb3Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
            if (maxExternalUsbPorts > 3) {
                RowTextAndSwitchContent(
                    mainLabel = "External USB Port 4",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.extUsb4Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setExtUsb4Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
            if (maxExternalUsbPorts > 4) {
                RowTextAndSwitchContent(
                    mainLabel = "External USB Port 5",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.extUsb5Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setExtUsb5Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
            if (maxExternalUsbPorts > 5) {
                RowTextAndSwitchContent(
                    mainLabel = "External USB Port 6",
                    label = "Disable / Enable",
                    state = peripheralActivityDetails.extUsb6Enabled,
                    onStateChange = {
                        peripheralActivityViewModel.setOutput("")
                        peripheralActivityViewModel.setExtUsb6Enabled(
                            it
                        )
                    }, verticalDivider = false,
                    center = true
                )
            }
        }
        EloWideButton(
            text = "Set Device Ports Status",
            onClick = {
                peripheralActivityViewModel.setOutput("")
                PeripheralActivityOnClickListeners.onSetDevicePortsStatusClick(
                    peripheralActivityViewModel,
                    maxUsbPorts,
                    maxExternalUsbPorts
                )
            }
        )
        EloWideButton(
            text = "Start Usb Monitor",
            onClick = {
                peripheralActivityViewModel.setOutput("")
                PeripheralActivityOnClickListeners.onStartUsbMonitorClick(
                    peripheralActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Stop Usb Monitor",
            onClick = {
                peripheralActivityViewModel.setOutput("")
                PeripheralActivityOnClickListeners.onStopUsbMonitorClick(
                    peripheralActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Get Maximum USB Port Count",
            onClick = {
                peripheralActivityViewModel.setOutput("")
                val maxUsbPort =
                    if (PeripheralActivityViewModel.isPreview || peripheralActivityViewModel.peripheralServiceManager == null) {
                        6
                    } else {
                        peripheralActivityViewModel.peripheralServiceManager.maxUSBPort
                    }
                peripheralActivityViewModel.setOutput("Maximum USB Port Count: $maxUsbPort")
            }
        )
        EloWideButton(
            text = "Get Maximum External USB Port Count",
            onClick = {
                peripheralActivityViewModel.setOutput("")
                val maxExtUSBPort =
                    if (PeripheralActivityViewModel.isPreview || peripheralActivityViewModel.peripheralServiceManager == null) {
                        6
                    } else {
                        peripheralActivityViewModel.peripheralServiceManager.getMaxExtUSBPort(
                            peripheralActivityViewModel.applicationContext
                        )
                    }
                peripheralActivityViewModel.setOutput(
                    "Maximum External USB Port Count: $maxExtUSBPort"
                )
            }
        )

        EloWideButton(text = "Get Usb Info", onClick = {
            peripheralActivityViewModel.setOutput("")
            PeripheralActivityOnClickListeners.onGetUsbInfoClick(peripheralActivityViewModel)
        })

        if (Util.isValueDevice()) {
            EloWideButton(text = "Get All USB Audio Devices", onClick = {
                peripheralActivityViewModel.setOutput("")
                PeripheralActivityOnClickListeners.onGetAllUsbAudioDevicesClick(
                    peripheralActivityViewModel
                )
            })

            EloWideButton(text = "Get Current USB Audio Device", onClick = {
                peripheralActivityViewModel.setOutput("")
                PeripheralActivityOnClickListeners.onGetUsbAudioDeviceClick(
                    peripheralActivityViewModel
                )
            })

            EloWideButton(text = "Set Current USB Audio Device", onClick = {
                peripheralActivityViewModel.setSetUsbAudioDeviceDialog(true)
            })
        }

        EloOutputText(peripheralActivityDetails.output)

        BottomButtonBar(activity) { peripheralActivityViewModel.setOutput("") }
    }
}

@Composable
fun PeripheralDialogs(peripheralActivityViewModel: PeripheralActivityViewModel) {
    PeripheralActivityDialogs.SetUsbAudioDeviceDialog(peripheralActivityViewModel)
}
