/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.packageActivity

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.packageActivity.PackageActivity.Companion.silentInstallActivityResultLauncher
import com.eloview.sdk.testapp.viewmodel.PackageActivityViewModel
import org.json.JSONArray
import org.json.JSONObject

class PackageActivityOnClickListeners {
    companion object {
        fun onSilentInstallClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                val intent = Intent(Intent.ACTION_OPEN_DOCUMENT)
                intent.addCategory(Intent.CATEGORY_OPENABLE)
                intent.setType("*/*")
                try {
                    silentInstallActivityResultLauncher?.launch(intent)
                } catch (ex: ActivityNotFoundException) {
                    packageActivityViewModel.setOutput("Please install a File Manager.")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSilentUninstallClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                val packageStatus =
                    packageActivityViewModel.packageServiceManager?.getPackageStatus(
                        packageActivityViewModel.getSilentUninstallPackage()
                    )
                if (packageStatus == -1) {
                    packageActivityViewModel.setOutput("Package not installed")
                    return
                } else if (packageStatus == -2) {
                    packageActivityViewModel.setOutput("Empty package name provided")
                    return
                }
                if (packageActivityViewModel.packageServiceManager?.silentPackageUninstall(
                        packageActivityViewModel.applicationContext,
                        GlobalClass.globalViewModel.getAccessToken(),
                        packageActivityViewModel.getSilentUninstallPackage(),
                        packageActivityViewModel.packageHandler
                    ) == true
                ) {
                    packageActivityViewModel.setOutput("Silent uninstall starting")
                } else {
                    packageActivityViewModel.setOutput("Silent uninstall failed")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onStopPackageClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                val packageStatus =
                    packageActivityViewModel.packageServiceManager?.getPackageStatus(
                        packageActivityViewModel.getStopPackageName()
                    )
                if (packageStatus == -1) {
                    packageActivityViewModel.setOutput("Package not installed")
                    return
                } else if (packageStatus == -2) {
                    packageActivityViewModel.setOutput("Empty package name provided")
                    return
                }
                if (packageActivityViewModel.packageServiceManager?.stopPackage(
                        packageActivityViewModel.applicationContext,
                        packageActivityViewModel.getStopPackageName()
                    ) == true
                ) {
                    packageActivityViewModel.setOutput("Package ${packageActivityViewModel.getStopPackageName()} stopped")
                } else {
                    packageActivityViewModel.setOutput("Failed to stop package")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onEnablePackageClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                if (packageActivityViewModel.packageServiceManager?.enablePackage(
                        packageActivityViewModel.applicationContext,
                        packageActivityViewModel.getEnablePackageName(),
                        packageActivityViewModel.packageHandler
                    ) == true
                ) {
                    packageActivityViewModel.setOutput("Enabling ${packageActivityViewModel.getEnablePackageName()}")
                } else {
                    packageActivityViewModel.setOutput("Failed to enable package")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onDisablePackageClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                if (packageActivityViewModel.packageServiceManager?.disablePackage(
                        packageActivityViewModel.applicationContext,
                        packageActivityViewModel.getDisablePackageName(),
                        packageActivityViewModel.packageHandler
                    ) == true
                ) {
                    packageActivityViewModel.setOutput("Disabling ${packageActivityViewModel.getDisablePackageName()}")
                } else {
                    packageActivityViewModel.setOutput("Failed to disable package")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onDisableSettingsPackageClick(
            packageActivityViewModel: PackageActivityViewModel,
            activity: Activity
        ) {
            if (Util.isValueDevice()) {
                packageActivityViewModel.setOutput("Not Supported on Value Device")
                return
            }
            if (packageActivityViewModel.getServiceConnected()) {
                packageActivityViewModel.packageServiceManager?.setSettingsPackageEnabled(
                    activity,
                    false,
                    packageActivityViewModel.packageHandler
                )
                packageActivityViewModel.setOutput("Displaying warning message for disabling settings app")
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onEnableSettingsPackageClick(
            packageActivityViewModel: PackageActivityViewModel,
            activity: Activity
        ) {
            if (Util.isValueDevice()) {
                packageActivityViewModel.setOutput("Not Supported on Value Device")
                return
            }
            if (packageActivityViewModel.getServiceConnected()) {
                packageActivityViewModel.packageServiceManager?.setSettingsPackageEnabled(
                    activity,
                    true,
                    packageActivityViewModel.packageHandler
                )
                packageActivityViewModel.setOutput("Enabling settings app")
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetPackageStatusClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                when (packageActivityViewModel.packageServiceManager?.getPackageStatus(
                    packageActivityViewModel.getPackageStatusPackageName()
                )) {
                    1 -> packageActivityViewModel.setOutput("Package is enabled")
                    0 -> packageActivityViewModel.setOutput("Package is disabled")
                    -1 -> packageActivityViewModel.setOutput("Package is not installed")
                    -2 -> packageActivityViewModel.setOutput("Empty package name provided")
                    -3 -> packageActivityViewModel.setOutput("Package service is not available")
                    else -> packageActivityViewModel.setOutput("Unknown error") // Shouldn't happen
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onAddWhiteListPackagePermissionClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                if (packageActivityViewModel.packageServiceManager?.setWhitelistAppPermission(
                        packageActivityViewModel.getWhiteListPackagePermission()
                    ) == true
                ) {
                    packageActivityViewModel.setOutput("Added ${packageActivityViewModel.getWhiteListPackagePermission()} to whitelist")
                } else {
                    packageActivityViewModel.setOutput("Failed to add package to whitelist")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onSetPackageNameAndPermissionClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                if (packageActivityViewModel.packageServiceManager?.setPackageNameAndPermission(
                        packageActivityViewModel.getPackageName(),
                        packageActivityViewModel.getPackagePermission()
                    ) == true
                ) {
                    packageActivityViewModel.setOutput("Set ${packageActivityViewModel.getPackagePermission()} enabled for package ${packageActivityViewModel.getPackageName()}")
                } else {
                    packageActivityViewModel.setOutput("Failed to set package name and permission")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onClearPackagePermissionClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                if (packageActivityViewModel.packageServiceManager?.clearWhitelistAppPermission(
                        packageActivityViewModel.getClearPackagePermission()
                    ) == true
                ) {
                    packageActivityViewModel.setOutput("Cleared permission for ${packageActivityViewModel.getClearPackagePermission()}")
                } else {
                    packageActivityViewModel.setOutput("Failed to clear package permission")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onClearAllPackagePermissionClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                if (packageActivityViewModel.packageServiceManager?.clearAllWhitelistAppPermission(
                        true
                    ) == true
                ) {
                    packageActivityViewModel.setOutput("Cleared all package permissions")
                } else {
                    packageActivityViewModel.setOutput("Failed to clear all package permissions")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onEnablePackageWhitelistingClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                val pkgName = packageActivityViewModel.getEnablePackageWhitelisting()
                if (pkgName.isNotEmpty()) {
                    val whiteListRule1 = JSONObject()
                    whiteListRule1.put("name", pkgName)
                    whiteListRule1.put("type", "package launching")
                    if (packageActivityViewModel.getEnablePackageWhitelistingBlockPackageCheckBox()) {
                        whiteListRule1.put("action", "block")
                    } else {
                        whiteListRule1.put("action", "allow")
                    }

                    val whiteListRule2 = JSONObject()
                    whiteListRule2.put("name", "com.eloview.sdk.testapp")
                    whiteListRule2.put("type", "package launching")
                    whiteListRule2.put("action", "allow")

                    val jsonArray = JSONArray()
                    jsonArray.put(whiteListRule2)
                    jsonArray.put(whiteListRule1)

                    if (packageActivityViewModel.packageServiceManager?.enableWhitelistPackages(
                            packageActivityViewModel.applicationContext,
                            jsonArray.toString(),
                            packageActivityViewModel.packageHandler
                        ) == false
                    ) {
                        packageActivityViewModel.setOutput("Could not enable package whitelisting. Please check the input and try again.")
                    } else {
                        packageActivityViewModel.setOutput("Enabling package whitelisting...")
                    }
                } else {
                    if (packageActivityViewModel.packageServiceManager?.enableWhitelistPackages(
                            packageActivityViewModel.applicationContext,
                            "",
                            packageActivityViewModel.packageHandler
                        ) == false
                    ) {
                        packageActivityViewModel.setOutput("Could not enable package whitelisting. Please check the input and try again.")
                    } else {
                        packageActivityViewModel.setOutput("Enabling package whitelisting...")
                    }
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onDisablePackageWhitelistingClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                if (packageActivityViewModel.packageServiceManager?.disableWhitelistPackages(
                        packageActivityViewModel.applicationContext,
                        packageActivityViewModel.packageHandler
                    ) == true
                ) {
                    packageActivityViewModel.setOutput("Disabling package whitelisting...")
                } else {
                    packageActivityViewModel.setOutput("Failed to disable package whitelisting")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onAutoStartAppAndLockItClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                if (packageActivityViewModel.getAutoStartAppAndLockItPackageName().trim()
                        .isEmpty()
                ) {
                    packageActivityViewModel.setOutput("Please enter a valid package name")
                } else {
                    if ("true" == packageActivityViewModel.packageServiceManager?.readSystemProperty(
                            "persist.sys.stdandroid"
                        )?.lowercase()
                    ) {
                        if (packageActivityViewModel.packageServiceManager.setAppAutostartPinned(
                                packageActivityViewModel.getAutoStartAppAndLockItNewPassword().trim(),
                                packageActivityViewModel.getAutoStartAppAndLockItOldPassword().trim(),
                                packageActivityViewModel.getAutoStartAppAndLockItPackageName().trim()
                            )
                        ) {
                            packageActivityViewModel.setOutput("Auto starting and locking ${packageActivityViewModel.getAutoStartAppAndLockItPackageName()}")
                        } else {
                            packageActivityViewModel.setOutput("Failed to auto start and lock app")
                        }
                    } else {
                        packageActivityViewModel.setOutput("Auto start and lock is only supported in Android Home mode")
                    }
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onRemoveTaskAutoStartAndLockingClick(packageActivityViewModel: PackageActivityViewModel) {
            val password = packageActivityViewModel.getRemoveTaskAutoStartAndLockingPassword()
            if (password.isEmpty()) {
                packageActivityViewModel.setOutput("Password can't be empty")
                return
            }
            if (packageActivityViewModel.getServiceConnected()) {
                val isStandardAndroidMode = packageActivityViewModel.packageServiceManager
                    ?.readSystemProperty("persist.sys.stdandroid")
                    ?.equals("true", ignoreCase = true) == true

                if (isStandardAndroidMode) {
                    val success = packageActivityViewModel.packageServiceManager?.removeAppAutostartPinned(
                        packageActivityViewModel.getRemoveTaskAutoStartAndLockingPassword(),
                        packageActivityViewModel.getTemporaryUnpinCheckBox() || !packageActivityViewModel.getPermanentUnpinCheckBox()
                    ) == true

                    if (success) {
                        packageActivityViewModel.setOutput("Removed task auto start and locking")
                    } else {
                        packageActivityViewModel.setOutput("Failed to remove task auto start and locking")
                    }
                } else {
                    packageActivityViewModel.setOutput("This operation is permitted only in standard android mode")
                }
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetPinnedAppStatusClick(packageActivityViewModel: PackageActivityViewModel) {
            if (packageActivityViewModel.getServiceConnected()) {
                packageActivityViewModel.setOutput("Pinned App Status: ${packageActivityViewModel.packageServiceManager?.appAutoStartPinnedStatus}")
            } else {
                packageActivityViewModel.setOutput("Service not connected")
            }
        }

    }
}
