/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.packageActivity

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.R
import com.eloview.sdk.testapp.common.CommonComposables.Companion.AccountManagerBtn
import com.eloview.sdk.testapp.common.CommonComposables.Companion.BottomButtonBar
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloOutputText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.networkActivity.ESTCertificateActivity.Companion.TAG
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.ui.theme.getEloLargeBoldTextPrimary
import com.eloview.sdk.testapp.viewmodel.PackageActivityViewModel

class PackageActivity : ComponentActivity() {
    private lateinit var mPackageActivityViewModel: PackageActivityViewModel

    companion object {
        var silentInstallActivityResultLauncher: ActivityResultLauncher<Intent>? = null
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        mPackageActivityViewModel = PackageActivityViewModel(
            applicationContext,
            false
        )
        if (savedInstanceState != null) {
            // persist output across UI reconfigurations
            savedInstanceState.getString("output")
                ?.let { mPackageActivityViewModel.setOutput(it) }
        }
        silentInstallActivityResultLauncher =
            registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
                try {
                    if (result.resultCode == RESULT_OK) {
                        val data: Intent? = result.data
                        Log.d(TAG, "onActivityResult: $result")

                        val selectedFile = data!!.data
                        var selectedFilePath: String
                        try {
                            selectedFilePath =
                                Util.getRealPathFromUri(applicationContext, selectedFile)
                        } catch (e: Exception) {
                            e.printStackTrace()
                            selectedFilePath = ""
                            mPackageActivityViewModel.setOutput("Please select valid apk path for Silent Install")
                        }
                        mPackageActivityViewModel.setSilentInstallPackagePath(selectedFilePath)
                        if (mPackageActivityViewModel.getServiceConnected()) {
                            if (mPackageActivityViewModel.packageServiceManager?.silentPackageInstall(
                                    mPackageActivityViewModel.applicationContext,
                                    GlobalClass.globalViewModel.getAccessToken(),
                                    mPackageActivityViewModel.getSilentInstallPackagePath(),
                                    mPackageActivityViewModel.packageHandler
                                ) == true
                            ) {
                                mPackageActivityViewModel.setOutput("Silent install starting")
                            } else {
                                mPackageActivityViewModel.setOutput("Silent install failed")
                            }
                        } else {
                            mPackageActivityViewModel.setOutput("Service not connected")
                        }
                    } else {
                        Log.e(TAG, "Activity ResultCode not ok")
                        mPackageActivityViewModel.setOutput("Please select valid apk path for Silent Install")
                    }
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    PackageActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        packageActivityViewModel = mPackageActivityViewModel,
                        activity = this
                    )
                }
            }
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        outState.putString("output", mPackageActivityViewModel.getOutput())
        super.onSaveInstanceState(outState)
    }
}

//@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
@Composable
fun PackageActivityScreen(
    modifier: Modifier = Modifier,
    packageActivityViewModel: PackageActivityViewModel = PackageActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: PackageActivity = PackageActivity()
) {
    val packageActivityDetails by packageActivityViewModel.packageActivityState.collectAsState()
    val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()

    PackageDialogs(packageActivityViewModel)

    if (globalDetails.authorized && !packageActivityDetails.serviceConnected) {
        packageActivityViewModel.bindService()
    }
    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,

            modifier = Modifier.padding(top= dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp)
                .fillMaxSize()
                .wrapContentHeight()
        ) {

            EloScreenTitleText("Package API")

            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier. padding(top= dimensionResource(id=com.intuit.sdp.R.dimen._10sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {
                TopButtonsComposable(packageActivityViewModel, activity)
            }
        }
    }
}

@Composable
@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
fun TopButtonsComposable(
    packageActivityViewModel: PackageActivityViewModel = PackageActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: Activity = Activity()
) {
    val packageActivityDetails by packageActivityViewModel.packageActivityState.collectAsState()
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxSize()
    ) {
        AccountManagerBtn(activity)
        EloWideButton(
            text = "Elo Silent Install",
            onClick = {
                packageActivityViewModel.setOutput("")
                PackageActivityOnClickListeners.onSilentInstallClick(
                    packageActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Elo Silent Uninstall",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowSilentUninstallDialog(true)
            }
        )
        EloWideButton(
            text = "Stop Package",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowStopPackageDialog(true)
            }
        )
        EloWideButton(
            text = "Enable Package",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowEnablePackageDialog(true)
            }
        )
        EloWideButton(
            text = "Disable Package",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowDisablePackageDialog(true)
            }
        )
        if (!Util.isValueDevice()) {
            EloWideButton(
                text = "Disable Settings Package",
                onClick = {
                    packageActivityViewModel.setOutput("")
                    PackageActivityOnClickListeners.onDisableSettingsPackageClick(
                        packageActivityViewModel,
                        activity
                    )
                }
            )
            EloWideButton(
                text = "Enable Settings Package",
                onClick = {
                    packageActivityViewModel.setOutput("")
                    PackageActivityOnClickListeners.onEnableSettingsPackageClick(
                        packageActivityViewModel,
                        activity
                )
            }
        )
      }
        EloWideButton(
            text = "Get Package Status",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowGetPackageStatusDialog(true)
            }
        )
        EloWideButton(
            text = "Add Whitelist Package Permission",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowAddWhiteListPackagePermissionDialog(true)
            }
        )
        EloWideButton(
            text = "Set Package Name and Permission",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowSetPackageNameAndPermissionDialog(true)
            }
        )
        EloWideButton(
            text = "Clear Package Permission",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowClearPackagePermissionDialog(true)
            }
        )
        EloWideButton(
            text = "Clear All Package Permission",
            onClick = {
                packageActivityViewModel.setOutput("")
                PackageActivityOnClickListeners.onClearAllPackagePermissionClick(
                    packageActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Enable Package Whitelisting",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowEnablePackageWhitelistingDialog(true)
            }
        )
        EloWideButton(
            text = "Disable Package Whitelisting",
            onClick = {
                packageActivityViewModel.setOutput("")
                PackageActivityOnClickListeners.onDisablePackageWhitelistingClick(
                    packageActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Auto Start App and Lock It",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowAutoStartAppAndLockItDialog(true)
            }
        )
        EloWideButton(
            text = "Remove Task Auto Start and Locking",
            onClick = {
                packageActivityViewModel.setOutput("")
                packageActivityViewModel.setShowRemoveTaskAutoStartAndLockingDialog(true)
            }
        )
        EloWideButton(
            text = "Get Pinned App Status",
            onClick = {
                packageActivityViewModel.setOutput("")
                PackageActivityOnClickListeners.onGetPinnedAppStatusClick(
                    packageActivityViewModel
                )
            }
        )

        EloOutputText(packageActivityDetails.output)

        BottomButtonBar(activity) { packageActivityViewModel.setOutput("") }
    }
}

@Composable
fun PackageDialogs(packageActivityViewModel: PackageActivityViewModel) {
    PackageActivityDialogs.SilentUninstallDialog(packageActivityViewModel)
    PackageActivityDialogs.StopPackageDialog(packageActivityViewModel)
    PackageActivityDialogs.EnablePackageDialog(packageActivityViewModel)
    PackageActivityDialogs.DisablePackageDialog(packageActivityViewModel)
    PackageActivityDialogs.GetPackageStatusDialog(packageActivityViewModel)
    PackageActivityDialogs.AddWhiteListPackagePermissionDialog(packageActivityViewModel)
    PackageActivityDialogs.SetPackageNameAndPermissionDialog(packageActivityViewModel)
    PackageActivityDialogs.ClearPackagePermissionDialog(packageActivityViewModel)
    PackageActivityDialogs.EnablePackageWhitelistingDialog(packageActivityViewModel)
    PackageActivityDialogs.AutoStartAppAndLockItDialog(packageActivityViewModel)
    PackageActivityDialogs.RemoveTaskAutoStartAndLockingDialog(packageActivityViewModel)
}
