/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.otaActivity

import android.app.AlertDialog
import android.content.Context

class OtaProgressDialog(private val context: Context) {

    private var alertDialog: AlertDialog? = null
    private var dismissedByUser = false

    /**
     * Function to show the dialog with progress and status.
     * If the dialog is not already created, it initializes a new one.
     * @param progress - Current progress percentage
     * @param status - Current status message
     * @param onDismiss - Lambda function executed when the dialog is dismissed
     */
    fun show(progress: Int, status: String, onDismiss: () -> Unit) {
        if (alertDialog == null) {

            // Create the dialog
            val builder = AlertDialog.Builder(context)
            builder.setTitle("OTA Progress")
                .setMessage("$status: $progress")
                .setCancelable(false)
                .setPositiveButton("Okay") { dialog, _ ->
                    dismissedByUser = true
                    onDismiss()
                    dialog.dismiss()
                }

            alertDialog = builder.create()
        }

        // Update the dialog's progress and status before showing it
        alertDialog?.setMessage("$status: $progress")

        // Show the dialog
        alertDialog?.show()
    }

    /**
     * Function to dismiss the dialog if it is currently showing.
     * Ensures proper cleanup of the AlertDialog instance.
     */
    fun dismiss() {
        alertDialog?.dismiss() // Dismiss the dialog if it exists
    }

    /**
     * Function to update the progress of an ongoing dialog.
     * Updates the message of the dialog with the provided progress and status.
     * @param progress - Current progress percentage
     * @param status - Current status message
     */
    fun updateProgress(progress: Int, status: String) {
        alertDialog?.setMessage("$status: $progress") // Update the Progress in the dialog.
    }

    /**
     * Function to check if the dialog is currently showing.
     * @return true if the dialog is showing, false otherwise.
     */
    fun isDialogShowing(): Boolean {
        return alertDialog?.isShowing == true
    }

    /**
     * Function to check if the dialog was dismissed by the user.
     * @return true if the user dismissed the dialog, false otherwise.
     */
    fun isDialogDismissed(): Boolean {
        return dismissedByUser
    }
}