/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.otaActivity

import android.content.ActivityNotFoundException
import android.content.Intent
import android.os.Bundle
import android.os.Message
import android.util.Log
import android.util.Patterns
import androidx.core.app.ActivityCompat.startActivityForResult
import com.eloview.sdk.otaManager.AndroidVersion
import com.eloview.sdk.otaManager.OTA
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.common.Util.isAndroidUDevice
import com.eloview.sdk.testapp.viewmodel.OtaActivityViewModel
import org.json.JSONObject
import java.io.File
import java.io.FileInputStream
import java.nio.charset.StandardCharsets

class OtaActivityOnClickListeners {
    companion object {
        private const val TAG = "OtaActivityOnClickListeners"
        fun onCheckEloOtaStatusLocalFileClick(otaActivityViewModel: OtaActivityViewModel) {
            val intent = Intent(Intent.ACTION_OPEN_DOCUMENT)
            intent.addCategory(Intent.CATEGORY_OPENABLE)
            intent.setType("application/json")
            try {
                OtaActivity.checkEloOtaStatusLocalFileActivityResultLauncher?.launch(intent)
            } catch (ex: ActivityNotFoundException) {
                otaActivityViewModel.setOutput("Please install a File Manager.")
                return
            }
        }

        fun onCheckEloOtaStatusClick(otaActivityViewModel: OtaActivityViewModel) {
            val checkEloOtaStatusUrl = otaActivityViewModel.getCheckEloOtaStatusUrl()
            OTA.instance.checkOTAStatus(
                otaActivityViewModel.applicationContext,
                GlobalClass.globalViewModel.getAccessToken(),
                checkEloOtaStatusUrl,
                otaActivityViewModel.otaHandler
            )
            otaActivityViewModel.setOutput("Checking Elo OTA Status URL: $checkEloOtaStatusUrl ...")
        }

        fun onGetAvailableVersionsClick(otaActivityViewModel: OtaActivityViewModel) {
            val selectedAvailableVersion: AndroidVersion? =
                when (otaActivityViewModel.getSelectedAvailableVersion()) {
                    "Q GMS Perf" -> AndroidVersion.ANDROID_Q_GMS_PERF
                    "Q AOSP Perf" -> AndroidVersion.ANDROID_Q_AOSP_PERF
                    "Q GMS Val" -> AndroidVersion.ANDROID_Q_GMS_VAL
                    "Q AOSP Val" -> AndroidVersion.ANDROID_Q_AOSP_VAL
                    "S GMS Perf" -> AndroidVersion.ANDROID_Q1_GMS_PERF
                    "S AOSP Perf" -> AndroidVersion.ANDROID_Q1_AOSP_PERF
                    "S GMS Val" -> AndroidVersion.ANDROID_Q1_GMS_VAL
                    "S AOSP Val" -> AndroidVersion.ANDROID_Q1_AOSP_VAL
                    "S 7 Inch Pay" -> AndroidVersion.ANDROID_7_INCH_PAY
                    "S M100" -> AndroidVersion.ANDROID_M100
                    "U GMS Rigel Perf" -> AndroidVersion.ANDROID_U1_5_RIGEL
                    else -> null
                }

            OTA.instance.getAvailableVersions(
                otaActivityViewModel.applicationContext,
                GlobalClass.globalViewModel.getAccessToken(),
                selectedAvailableVersion,
                otaActivityViewModel.otaHandler
            )
            var outputVersion = selectedAvailableVersion.toString()
            if (selectedAvailableVersion == null) {
                outputVersion = "Device Default"
            }
            otaActivityViewModel.setOutput("Getting available versions for: $outputVersion ...")
        }

        fun onDownloadDefaultEloOtaPackageClick(otaActivityViewModel: OtaActivityViewModel) {
            otaActivityViewModel.setOutput("Downloading default Elo OTA package ...")
            if (Util.isAndroidQDevice() || isAndroidUDevice()) {
                OTA.instance.downloadApplyABOTA(
                    otaActivityViewModel.applicationContext,
                    GlobalClass.globalViewModel.getAccessToken(),
                    "",
                    "",
                    "",
                    otaActivityViewModel.otaHandler
                )
            } else {
                OTA.instance.startDownloadUpgradePackage(
                    otaActivityViewModel.applicationContext,
                    GlobalClass.globalViewModel.getAccessToken(),
                    "",
                    null,
                    "",
                    otaActivityViewModel.otaHandler
                )
            }
        }

        fun onDownloadEloOtaPackageFromUrlClick(otaActivityViewModel: OtaActivityViewModel) {
            val downloadEloOtaPackageFromUrl =
                otaActivityViewModel.getDownloadEloOtaPackageFromUrl()
            otaActivityViewModel.setOutput("Downloading Elo OTA package from URL: $downloadEloOtaPackageFromUrl ...")

            if (downloadEloOtaPackageFromUrl.isNotEmpty()) {
                if (Util.isAndroidQDevice() || isAndroidUDevice()) {
                    if (Patterns.WEB_URL.matcher(downloadEloOtaPackageFromUrl).matches()) {
                        Log.v(TAG, "Using Patterns.WEB_URL for download")
                        OTA.instance.downloadApplyABOTA(
                            otaActivityViewModel.applicationContext,
                            GlobalClass.globalViewModel.getAccessToken(),
                            downloadEloOtaPackageFromUrl,
                            "",
                            "",
                            otaActivityViewModel.otaHandler
                        )
                    } else {
                        Log.v(TAG, "Using user provided URL for download")
                        OTA.instance.downloadApplyABOTA(
                            otaActivityViewModel.applicationContext,
                            GlobalClass.globalViewModel.getAccessToken(),
                            downloadEloOtaPackageFromUrl,
                            "",
                            "",
                            otaActivityViewModel.otaHandler
                        )
                    }
                } else {
                    if (Patterns.WEB_URL.matcher(downloadEloOtaPackageFromUrl).matches()) {
                        Log.v(TAG, "Using Patterns.WEB_URL for download")
                        OTA.instance.startDownloadUpgradePackage(
                            otaActivityViewModel.applicationContext,
                            GlobalClass.globalViewModel.getAccessToken(),
                            downloadEloOtaPackageFromUrl,
                            null,
                            "",
                            otaActivityViewModel.otaHandler
                        )
                    } else {
                        Log.v(TAG, "Using user provided URL for download")
                        OTA.instance.startDownloadUpgradePackage(
                            otaActivityViewModel.applicationContext,
                            GlobalClass.globalViewModel.getAccessToken(),
                            downloadEloOtaPackageFromUrl,
                            null,
                            "",
                            otaActivityViewModel.otaHandler
                        )
                    }
                }
            } else {
                otaActivityViewModel.setOutput("Please enter an OTA download url")
            }
        }

        fun onDownloadEloOtaPackageFromLocalPathClick(otaActivityViewModel: OtaActivityViewModel) {
            val downloadEloOtaPackageFromLocalPath =
                otaActivityViewModel.getDownloadEloOtaPackageFromLocalPath()
            otaActivityViewModel.setOutput("Downloading Elo OTA package from local path: $downloadEloOtaPackageFromLocalPath ...")

            if (Util.isAndroidQDevice() || isAndroidUDevice()) {
                if (downloadEloOtaPackageFromLocalPath.isNotEmpty()) {
                    OTA.instance.downloadApplyABOTA(
                        otaActivityViewModel.applicationContext,
                        GlobalClass.globalViewModel.getAccessToken(),
                        "",
                        downloadEloOtaPackageFromLocalPath,
                        "",
                        otaActivityViewModel.otaHandler
                    )
                } else {
                    otaActivityViewModel.setOutput("Please enter an OTA path")
                }
            } else {
                val intent = Intent(Intent.ACTION_OPEN_DOCUMENT)
                intent.addCategory(Intent.CATEGORY_OPENABLE)
                intent.setType("*/*")
                try {
                    OtaActivity.otaActivityResultLauncher?.launch(intent)
                } catch (ex: ActivityNotFoundException) {
                    otaActivityViewModel.setOutput("Please install a File Manager.")
                }
            }
        }

        fun onDownloadSpecificEloOtaVersionClick(otaActivityViewModel: OtaActivityViewModel) {
            val downloadSpecificEloOtaVersion =
                otaActivityViewModel.getSelectedSpecificEloOtaVersion()
            otaActivityViewModel.setOutput("Downloading specific Elo OTA version: $downloadSpecificEloOtaVersion ...")

            if (downloadSpecificEloOtaVersion.isNotEmpty()) {
                Log.v(TAG, "Using user provided version for download")
                if (Util.isAndroidQDevice() || isAndroidUDevice()) {
                    OTA.instance.downloadApplyABOTA(
                        otaActivityViewModel.applicationContext,
                        GlobalClass.globalViewModel.getAccessToken(),
                        "",
                        "",
                        downloadSpecificEloOtaVersion,
                        otaActivityViewModel.otaHandler
                    )
                } else {
                    OTA.instance.startDownloadUpgradePackage(
                        otaActivityViewModel.applicationContext,
                        GlobalClass.globalViewModel.getAccessToken(),
                        "",
                        null,
                        downloadSpecificEloOtaVersion,
                        otaActivityViewModel.otaHandler
                    )
                }
            } else {
                otaActivityViewModel.setOutput("Please enter a version")
                val message = Message()
                val bundle = Bundle()
                bundle.putString(OTA.OTA_DOWNLOAD_GENERIC_ERROR.toString(), "OTA Version field is empty")
                message.data = bundle
                message.what = OTA.OTA_DOWNLOAD_GENERIC_ERROR
                otaActivityViewModel.otaHandler.sendMessage(message)
            }
        }

        fun onCheckOS360StatusClick(otaActivityViewModel: OtaActivityViewModel) {
            otaActivityViewModel.setOutput("Checking OS360 Status ...")
            OTA.instance.getOS360Details(
                otaActivityViewModel.applicationContext,
                GlobalClass.globalViewModel.getAccessToken(),
                otaActivityViewModel.otaHandler
            )
        }
    }
}