/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.otaActivity

import android.annotation.SuppressLint
import android.app.Activity
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.net.Uri
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.requiredHeight
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.ViewModelProvider
import com.eloview.sdk.accountManager.AccountManager
import com.eloview.sdk.otaManager.OTA
import com.eloview.sdk.otaManager.OTAUtils.ACTION_OTA_PROGRESS
import com.eloview.sdk.testapp.AccountManagerViewModelFactory
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.R
import com.eloview.sdk.testapp.common.CommonComposables.Companion.AccountManagerBtn
import com.eloview.sdk.testapp.common.CommonComposables.Companion.BottomButtonBar
import com.eloview.sdk.testapp.common.CommonComposables.Companion.CommonButton
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloOutputText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.ui.theme.getEloLargeBoldTextPrimary
import com.eloview.sdk.testapp.ui.theme.getEloOutputText
import com.eloview.sdk.testapp.viewmodel.AccountManagerViewModel
import com.eloview.sdk.testapp.viewmodel.OtaActivityViewModel
import org.json.JSONException
import org.json.JSONObject

class OtaActivity : ComponentActivity() {
    companion object {
        const val TAG = "OtaActivity"
        var otaActivityResultLauncher: ActivityResultLauncher<Intent>? = null
        var checkEloOtaStatusLocalFileActivityResultLauncher: ActivityResultLauncher<Intent>? = null
        private var showProgressFlag: Boolean = true
    }

    private lateinit var mOtaActivityViewModel: OtaActivityViewModel

    private lateinit var otaProgressDialog: OtaProgressDialog
    private lateinit var accountManagerViewModel: AccountManagerViewModel
    private var stopShowingDialog = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        // Initialize the ViewModel for account management.
        val factory = AccountManagerViewModelFactory(applicationContext)
        accountManagerViewModel =
            ViewModelProvider(this, factory)[AccountManagerViewModel::class.java]

        mOtaActivityViewModel = OtaActivityViewModel(
            applicationContext,
            false
        )
        if (savedInstanceState != null) {
            // persist output across UI reconfigurations
            savedInstanceState.getString("output")
                ?.let { mOtaActivityViewModel.setOutput(it) }
        }
        otaActivityResultLauncher =
            registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
                if (result.resultCode == RESULT_OK) {
                    val selectedFile: Uri? = result.data?.data

                    try {
                        if (selectedFile != null) {
                            Log.d(TAG, selectedFile.toString())
                            mOtaActivityViewModel.setOutput("Selected File: $selectedFile")
                            OTA.instance.startDownloadUpgradePackage(
                                this@OtaActivity,
                                GlobalClass.globalViewModel.getAccessToken(),
                                "",
                                selectedFile,
                                "",
                                mOtaActivityViewModel.otaHandler
                            )
                        } else {
                            mOtaActivityViewModel.setOutput("No file selected")
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
            }

        checkEloOtaStatusLocalFileActivityResultLauncher =
            registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
                if (result.resultCode == RESULT_OK) {
                    val selectedFile: Uri? = result.data?.data
                    try {
                        if (selectedFile != null) {
                            Log.d(TAG, selectedFile.toString())
                            mOtaActivityViewModel.setOutput("Selected File: $selectedFile")
                            var jsonObject: JSONObject? = null
                            val fileContent = Util.readFileContent(applicationContext, selectedFile)

                            if (!fileContent.isNullOrBlank()) {
                                try {
                                    jsonObject =
                                        JSONObject(Util.readFileContent(applicationContext, selectedFile))
                                    OTA.instance.checkOTAStatus(
                                        mOtaActivityViewModel.applicationContext,
                                        GlobalClass.globalViewModel.getAccessToken(),
                                        jsonObject,
                                        mOtaActivityViewModel.otaHandler
                                    )
                                } catch (e: JSONException) {
                                    e.printStackTrace()
                                    mOtaActivityViewModel.setOutput("Invalid Json File.")
                                } catch (e: Exception) {
                                    e.printStackTrace()
                                }
                            } else {
                                mOtaActivityViewModel.setOutput("File is empty or couldn't be read.")
                            }
                        } else {
                            mOtaActivityViewModel.setOutput("No file selected")
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
            }

        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    OtaActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        otaActivityViewModel = mOtaActivityViewModel,
                        activity = this
                    )
                }
            }
        }
    }

    /**
     * Verifies the access token, initializes the progress dialog, and
     * registers a broadcast receiver to listen for OTA progress updates.
     */
    @SuppressLint("InlinedApi")
    override fun onResume() {
        super.onResume()

        // Retrieve access token from ViewModel's state
        val accountManagerState = accountManagerViewModel.deviceManagerState.value
        val accessToken = accountManagerState.accessToken

        // Verify the access token using AccountManager
        AccountManager.instance.verifyEloToken(
            this,
            accessToken,
            accountManagerViewModel.accessTokenHandler
        )

        // Initialize the progress dialog for OTA updates
        otaProgressDialog = OtaProgressDialog(this)

        val filter = IntentFilter().apply {
            addAction(ACTION_OTA_PROGRESS)
        }
        registerReceiver(otaProgressReceiver, filter, RECEIVER_EXPORTED)

        // Reset stopShowingDialog flag if the user re-enters the activity
        if (!stopShowingDialog && otaProgressDialog.isDialogDismissed()) {
            otaProgressDialog.show(0, "Resuming Progress") {
                stopShowingDialog = true // Update the flag when dialog is dismissed by the user.
            }
        }
    }

    /**
     * BroadcastReceiver to handle OTA progress updates.
     * Updates or shows the progress dialog based on broadcast progress and status.
     */
    private val otaProgressReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            if (showProgressFlag && !stopShowingDialog) { // Show progress dialog only if not stopped by user
                runOnUiThread {
                    // Retrieve progress and status from the broadcast.
                    val status = intent.getStringExtra("status") ?: "Unknown Status"
                    val progress = intent.getIntExtra("progress", 0)

                    // Update the progress dialog with new data.
                    otaProgressDialog.updateProgress(progress, status)

                    // Show the progress dialog if it is not already visible
                    if (!otaProgressDialog.isDialogShowing()) {
                        otaProgressDialog.show(progress, status) {
                            stopShowingDialog =
                                true // Stop showing the dialog if user clicks "Okay"
                        }
                    }
                }
            }
        }
    }

    /**
     * Called when the activity is no longer visible to the user.
     * Unregisters the broadcast receiver to prevent memory leaks.
     */
    override fun onStop() {
        super.onStop()
        unregisterReceiver(otaProgressReceiver)
        otaProgressDialog.dismiss() // Dismiss dialog when activity goes to the background
    }

    override fun onSaveInstanceState(outState: Bundle) {
        outState.putString("output", mOtaActivityViewModel.getOutput())
        super.onSaveInstanceState(outState)
    }
}


@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
@Composable
fun OtaActivityScreen(
    modifier: Modifier = Modifier,
    otaActivityViewModel: OtaActivityViewModel = OtaActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: OtaActivity = OtaActivity()
) {

    OtaDialogs(otaActivityViewModel)

    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,

            modifier = Modifier.padding(top= dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
            start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
            end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp)
                .fillMaxSize()
                .wrapContentHeight()
        ) {
            Text(
                text = stringResource(R.string.str_ota_screen_lable),
                style = getEloLargeBoldTextPrimary(),
            )

            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier. padding(top= dimensionResource(id=com.intuit.sdp.R.dimen._10sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {
                TopButtonsComposable(otaActivityViewModel, activity)
            }
        }
    }
}

@Composable
//@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
fun TopButtonsComposable(
    otaActivityViewModel: OtaActivityViewModel = OtaActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: Activity = Activity(),
) {
    val otaActivityDetails by otaActivityViewModel.otaActivityState.collectAsState()
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxSize()
    ) {
        AccountManagerBtn(activity)
        EloWideButton(
            text = stringResource(R.string.str_check_os360_status),
            onClick = {
                otaActivityViewModel.setOutput("")
                OtaActivityOnClickListeners.onCheckOS360StatusClick(otaActivityViewModel)
            }
        )

        EloWideButton(
            text = stringResource(R.string.str_check_elo_ota_status),
            onClick = {
                otaActivityViewModel.setOutput("")
                otaActivityViewModel.setShowCheckEloOtaStatusDialog(true)
            }
        )

        EloWideButton(
            text = stringResource(R.string.str_check_elo_ota_status_local),
            onClick = {
                otaActivityViewModel.setOutput("")
                OtaActivityOnClickListeners.onCheckEloOtaStatusLocalFileClick(otaActivityViewModel)
            }
        )

        EloWideButton(
            text = stringResource(R.string.str_get_available_version),
            onClick = {
                otaActivityViewModel.setOutput("")
                otaActivityViewModel.setShowGetAvailableVersionsDialog(true)
            }
        )

        EloWideButton(
            text = stringResource(R.string.str_download_default_ota_package),
            onClick = {
                otaActivityViewModel.setOutput("")
                OtaActivityOnClickListeners.onDownloadDefaultEloOtaPackageClick(otaActivityViewModel)
            }
        )

        EloWideButton(
            text = stringResource(R.string.str_download_ota_package_from_url),
            onClick = {
                otaActivityViewModel.setOutput("")
                otaActivityViewModel.setShowDownloadEloOtaPackageFromUrlDialog(true)
            }
        )

        EloWideButton(
            text =stringResource(R.string.str_download_ota_package_from_local_path),
            onClick = {
                otaActivityViewModel.setOutput("")
                otaActivityViewModel.setShowDownloadEloOtaPackageFromLocalPathDialog(true)
            }
        )

        EloWideButton(
            text = stringResource(R.string.str_download_specific_ota_version),
            onClick = {
                otaActivityViewModel.setOutput("")
                otaActivityViewModel.setShowDownloadSpecificEloOtaVersionDialog(true)
            }
        )

        EloOutputText(otaActivityDetails.output)

        BottomButtonBar(activity) { otaActivityViewModel.setOutput("") }
    }
}

@Composable
fun OtaDialogs(otaActivityViewModel: OtaActivityViewModel) {
    OtaActivityDialogs.CheckEloOtaStatusDialog(otaActivityViewModel)
    OtaActivityDialogs.GetAvailableVersionsDialog(otaActivityViewModel)
    OtaActivityDialogs.DownloadEloOtaPackageFromUrlDialog(otaActivityViewModel)
    OtaActivityDialogs.DownloadEloOtaPackageFromLocalPathDialog(otaActivityViewModel)
    OtaActivityDialogs.DownloadSpecificEloOtaVersionDialog(otaActivityViewModel)
}
