/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.networkActivity

import android.text.TextUtils
import android.util.Log
import com.eloview.sdk.networkManager.NetworkConfig
import com.eloview.sdk.networkManager.NetworkSecurityType
import com.eloview.sdk.networkManager.NetworkType
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.viewmodel.NetworkActivityViewModel

class NetworkActivityOnClickListeners {
    companion object {
        private const val TAG = "NetworkActivityOnClickListeners"
        fun onGetWiFiMacClick(networkActivityViewModel: NetworkActivityViewModel) {
            networkActivityViewModel.setOutput("Getting WiFi MAC Address...")
            if (networkActivityViewModel.getServiceConnected()) {
                val macAddress = networkActivityViewModel.networkServiceManager?.wiFiMac
                networkActivityViewModel.setOutput("WiFi MAC Address: $macAddress")
            } else {
                networkActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onGetEthernetMacClick(networkActivityViewModel: NetworkActivityViewModel) {
            networkActivityViewModel.setOutput("Getting Ethernet MAC Address...")
            if (networkActivityViewModel.getServiceConnected()) {
                val macAddress = networkActivityViewModel.networkServiceManager?.ethernetMac
                networkActivityViewModel.setOutput("Ethernet MAC Address: $macAddress")
            } else {
                networkActivityViewModel.setOutput("Service not connected")
            }
        }
        /**
         * @Open network is not connected when device is secured.
         * secured devices(PCIFLAG on) open network are restricted.
         */
        fun onSetNetworkClick(networkActivityViewModel: NetworkActivityViewModel) {
            val networkConfig: NetworkConfig
            if (networkActivityViewModel.getNetworkConfigurationWiFiLanToggle()) {
                networkConfig = NetworkConfig(NetworkType.WiFi)
                // get selected radio button from radioGroup
                when (networkActivityViewModel.getNetworkSecurityType()) {
                    "WPA" -> networkConfig.securityType = NetworkSecurityType.WPA
                    "WPA2" -> networkConfig.securityType = NetworkSecurityType.WPA2
                    "open" -> networkConfig.securityType = NetworkSecurityType.OPEN
                    "EAP" -> networkConfig.securityType = NetworkSecurityType.EAP_TLS
                    else -> {
                        networkActivityViewModel.setOutput("Select Network Type of WiFi")
                    }
                }
            } else {
                networkConfig = NetworkConfig(NetworkType.LAN)
            }

            val name: String = networkActivityViewModel.getSsidName().trim { it <= ' ' }
            if (!TextUtils.isEmpty(name)) {
                networkConfig.name = name
            }
            val alias: String = networkActivityViewModel.getAlias().trim { it <= ' ' }
            if (!TextUtils.isEmpty(alias)) {
                networkConfig.certAlias = alias
            }
            val password: String = networkActivityViewModel.getPassword().trim { it <= ' ' }
            if (!TextUtils.isEmpty(password)) {
                networkConfig.password = password
            }
            val gateway: String = networkActivityViewModel.getGateway().trim { it <= ' ' }
            if (!TextUtils.isEmpty(gateway)) {
                networkConfig.gateway = gateway
            }
            val staticIp: String = networkActivityViewModel.getStaticIp().trim { it <= ' ' }
            if (!TextUtils.isEmpty(staticIp)) {
                networkConfig.staticIp = staticIp
            }
            val networkPrefix: String = networkActivityViewModel.getPrefix().trim { it <= ' ' }
            if (!TextUtils.isEmpty(networkPrefix)) {
                networkConfig.networkPrefix = networkPrefix
            }
            val dns1: String = networkActivityViewModel.getDns1().trim { it <= ' ' }
            if (!TextUtils.isEmpty(dns1)) {
                networkConfig.dns1 = dns1
            }
            val dns2: String = networkActivityViewModel.getDns2().trim { it <= ' ' }
            if (!TextUtils.isEmpty(dns2)) {
                networkConfig.dns2 = dns2
            }
            val host: String = networkActivityViewModel.getProxyHost().trim { it <= ' ' }
            if (!TextUtils.isEmpty(host)) {
                networkConfig.proxyHost = host
            }
            val port: String = networkActivityViewModel.getProxyPort().trim { it <= ' ' }
            if (!TextUtils.isEmpty(port)) {
                try {
                    networkConfig.proxyPort = port.toInt()
                } catch (e: Exception) {
                    e.printStackTrace()
                    networkActivityViewModel.setOutput("Invalid proxy port: $port")
                }
            }

            networkConfig.avoidPoorWifi = networkActivityViewModel.getAvoidPoorConnection()
            networkConfig.defaultExclude = networkActivityViewModel.getDefaultExclude()

            Log.d(TAG, "onClick: $networkConfig")
            networkActivityViewModel.setOutput("Setting Network...")
            networkActivityViewModel.networkServiceManager?.setNetwork(
                networkActivityViewModel.applicationContext,
                GlobalClass.globalViewModel.getAccessToken(),
                networkConfig,
                networkActivityViewModel.networkHandler
            )
        }

        fun onGetNetworkClick(networkActivityViewModel: NetworkActivityViewModel) {
            networkActivityViewModel.setOutput("Getting Network...")
            networkActivityViewModel.networkServiceManager?.getNetwork(
                networkActivityViewModel.applicationContext,
                GlobalClass.globalViewModel.getAccessToken(),
                networkActivityViewModel.networkHandler
            )
        }

        fun onGetDHCPServerHostNameClick(networkActivityViewModel: NetworkActivityViewModel) {
            if (networkActivityViewModel.getServiceConnected()) {
                networkActivityViewModel.networkServiceManager?.dhcpServerHostName?.let {
                    networkActivityViewModel.setOutput(
                        "Server HostName: $it"
                    )
                }
            } else {
                networkActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onRemoveCertificateClick(networkActivityViewModel: NetworkActivityViewModel) {
            if (networkActivityViewModel.getRemoveAlias().isEmpty()) {
                networkActivityViewModel.setOutput("Invalid Data")
                return
            }
            if (networkActivityViewModel.getServiceConnected()) {
                if (networkActivityViewModel.networkServiceManager?.removeCertificate(
                        networkActivityViewModel.getRemoveAlias()
                    ) == true
                ) {
                    networkActivityViewModel.setOutput("Certificate removed")
                } else {
                    networkActivityViewModel.setOutput("Certificate not removed")
                }
            } else {
                networkActivityViewModel.setOutput("Service not connected")
            }
        }

        fun onEthStateClick(networkActivityViewModel: NetworkActivityViewModel) {
            if (networkActivityViewModel.getServiceConnected()) {
                // run on a background thread
                val thread = Thread {
                    networkActivityViewModel.networkServiceManager?.setEthState(
                        networkActivityViewModel.getEthState()
                    )
                    networkActivityViewModel.setOutput("Ethernet state set to ${networkActivityViewModel.getEthState()}")
                }
                thread.start()
            } else {
                networkActivityViewModel.setOutput("Service not connected")
            }
        }
    }
}
