/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.networkActivity

import android.app.Activity
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.layout.wrapContentWidth
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.Button
import androidx.compose.material3.Checkbox
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.RectangleShape
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.eloview.sdk.testapp.common.CommonComposables.Companion.CommonExposedDropdownMenu
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.RowTextAndSwitchContent
import com.eloview.sdk.testapp.common.CommonComposables.Companion.SwitchWithLabel
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.MyAlertDialog
import com.eloview.sdk.testapp.common.CommonDialogComposables.Companion.SingleTextFieldTwoButtonDialog
import com.eloview.sdk.testapp.networkActivity.NetworkActivityDialogs.Companion.RemoveCertificateDialog
import com.eloview.sdk.testapp.networkActivity.NetworkActivityDialogs.Companion.SetNetworkDialog
import com.eloview.sdk.testapp.ui.theme.EloTextTitle1
import com.eloview.sdk.testapp.ui.theme.getEloBoldStdSecondaryText
import com.eloview.sdk.testapp.ui.theme.getEloBoldStdSecondaryTextstart
import com.eloview.sdk.testapp.ui.theme.getEloLargeBoldTextPrimary
import com.eloview.sdk.testapp.ui.theme.getEloLargeBoldTextWhite
import com.eloview.sdk.testapp.ui.theme.getEloNormalStdSecondaryText
import com.eloview.sdk.testapp.ui.theme.getEloSmallSecondaryText
import com.eloview.sdk.testapp.viewmodel.NetworkActivityViewModel

class NetworkActivityDialogs {
    companion object {

        @Composable
        fun SetNetworkDialog(
            networkActivityViewModel: NetworkActivityViewModel = NetworkActivityViewModel(
                Activity(),
                true
            )
        ) {
            val networkActivityDetails by networkActivityViewModel.networkActivityState.collectAsState()
            if (NetworkActivityViewModel.isPreview) {
                networkActivityViewModel.setShowNetworkConfigurationDialog(true)
            }
            if (networkActivityDetails.showNetworkConfigurationDialog) {
                MyAlertDialog(
                    title = {
                       EloScreenTitleText("Network Configuration")
                    },
                    content = {
                        Column(
                            modifier = Modifier
                                .fillMaxHeight(0.80f)
                                .verticalScroll(rememberScrollState())
                                .fillMaxWidth(1f)
                        ) {
                            // Function to generate the DoubleRow with horizontally aligned Text and TextField.
                            val doubleRow: @Composable (String, @Composable () -> Unit) -> Unit =
                                { label, content ->
                                    Row(
                                        modifier = Modifier
                                            .border(BorderStroke(width = 3.dp, color = MaterialTheme.colorScheme.primary))
                                            .wrapContentHeight(align = Alignment.Top)
                                            .fillMaxWidth()
                                            .padding(
                                                start = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp,
                                                end = dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp,
                                                top=dimensionResource(id=com.intuit.sdp.R.dimen._6sdp).value.dp,
                                                bottom = dimensionResource(id=com.intuit.sdp.R.dimen._6sdp).value.dp),

                                                verticalAlignment = Alignment.CenterVertically,
                                        horizontalArrangement = Arrangement.spacedBy(dimensionResource(id=com.intuit.sdp.R.dimen._5sdp).value.dp)

                                    ) {
                                        Box(
                                            modifier = Modifier
                                                .weight(0.46f)
                                                .fillMaxHeight()
                                        ) {
                                            Text(
                                                text = label,
                                                textAlign = TextAlign.Start,
                                                style = getEloBoldStdSecondaryText()
                                            )
                                        }
                                        Box(
                                            modifier = Modifier
                                                .weight(1f),
                                            contentAlignment = Alignment.CenterStart
                                        ) {
                                            content()
                                        }
                                    }

                                    Spacer(modifier = Modifier.fillMaxWidth().height(
                                        dimensionResource(id=com.intuit.sdp.R.dimen._6sdp).value.dp
                                    ))
                                }

                            RowTextAndSwitchContent(mainLabel = "Network Type", label = "LAN / WiFi",
                                state = networkActivityDetails.networkConfigurationWiFiLanToggle,
                                onStateChange = {
                                    networkActivityViewModel.setNetworkConfigurationWiFiLanToggle(
                                        it
                                    )
                                }, verticalDivider = false,
                                center = true)


                            doubleRow("Network Security Type") {
                                CommonExposedDropdownMenu(
                                    values = arrayOf("WPA", "open", "WPA2", "EAP"),
                                    selectedText = networkActivityDetails.networkSecurityTypeSelected,
                                    onValueSelected = {
                                        networkActivityViewModel.setNetworkSecurityType(it)
                                    }
                                )
                            }
                            doubleRow("SSID Name") {
                                TextField(
                                    value = networkActivityDetails.ssidName,
                                    onValueChange = {
                                        networkActivityViewModel.setSsidName(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),

                                    placeholder = { Text("Enter SSID Name", style = getEloSmallSecondaryText()) }
                                )
                            }
                            doubleRow("Alias") {
                                TextField(
                                    value = networkActivityDetails.alias,
                                    onValueChange = {
                                        networkActivityViewModel.setAlias(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),
                                    placeholder = { Text("Enter Alias", style = getEloSmallSecondaryText()) }
                                )
                            }
                            doubleRow("Password") {
                                TextField(
                                    value = networkActivityDetails.password,
                                    onValueChange = {
                                        networkActivityViewModel.setPassword(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),
                                    placeholder = { Text("Enter Password", style = getEloSmallSecondaryText()) },
                                    visualTransformation = PasswordVisualTransformation(),
                                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Password)
                                )
                            }
                            doubleRow("Proxy Host") {
                                TextField(
                                    value = networkActivityDetails.proxyHost,
                                    onValueChange = {
                                        networkActivityViewModel.setProxyHost(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),
                                    placeholder = { Text("Enter Proxy Host", style = getEloSmallSecondaryText()) }
                                )
                            }
                            doubleRow("Proxy Port") {
                                TextField(
                                    value = networkActivityDetails.proxyPort,
                                    onValueChange = {
                                        networkActivityViewModel.setProxyPort(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),
                                    placeholder = { Text("Enter Proxy Port", style = getEloSmallSecondaryText()) }
                                )
                            }
                            doubleRow("Static IP") {
                                TextField(
                                    value = networkActivityDetails.staticIp,
                                    onValueChange = {
                                        networkActivityViewModel.setStaticIp(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),
                                    placeholder = { Text("Enter Static IP", style = getEloSmallSecondaryText()) }
                                )
                            }
                            doubleRow("Gateway") {
                                TextField(
                                    value = networkActivityDetails.gateway,
                                    onValueChange = {
                                        networkActivityViewModel.setGateway(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),
                                    placeholder = { Text("Enter Gateway" , style = getEloSmallSecondaryText()) }
                                )
                            }
                            doubleRow("Network Prefix") {
                                TextField(
                                    value = networkActivityDetails.prefix,
                                    onValueChange = {
                                        networkActivityViewModel.setPrefix(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),
                                    placeholder = { Text("Enter Network Prefix" , style = getEloSmallSecondaryText()) }
                                )
                            }
                            doubleRow("DNS1") {
                                TextField(
                                    value = networkActivityDetails.dns1,
                                    onValueChange = {
                                        networkActivityViewModel.setDns1(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),
                                    placeholder = { Text("Enter DNS1" , style = getEloSmallSecondaryText()) }
                                )
                            }
                            doubleRow("DNS2") {
                                TextField(
                                    value = networkActivityDetails.dns2,
                                    onValueChange = {
                                        networkActivityViewModel.setDns2(it)
                                    },
                                    modifier = Modifier.fillMaxWidth(),
                                    textStyle = getEloBoldStdSecondaryTextstart(),
                                    placeholder = { Text("Enter DNS2", style = getEloSmallSecondaryText()) }
                                )
                            }
                            doubleRow("Avoid Poor Connection") {
                                Checkbox(
                                    checked = networkActivityDetails.avoidPoorConnection,
                                    onCheckedChange = {
                                        networkActivityViewModel.setAvoidPoorConnection(it)
                                    }
                                )
                            }
                            doubleRow("Default Exclude") {
                                Checkbox(
                                    checked = networkActivityDetails.defaultExclude,
                                    onCheckedChange = {
                                        networkActivityViewModel.setDefaultExclude(it)
                                    }
                                )
                            }
                        }
                    },
                    confirmButton = {
                        Button(
                            onClick = {
                                networkActivityViewModel.setShowNetworkConfigurationDialog(false)
                                NetworkActivityOnClickListeners.onSetNetworkClick(
                                    networkActivityViewModel
                                )
                            },
                            modifier = Modifier
                                .fillMaxWidth(1f)
                                .padding(5.dp),
                            shape = RectangleShape
                        ) {
                            Text(
                                text = "Confirm",
                                style = getEloLargeBoldTextWhite()
                            )
                        }
                    },
                    dismissButton = {
                        Button(
                            onClick = {
                                networkActivityViewModel.setShowNetworkConfigurationDialog(false)
                            },
                            modifier = Modifier
                                .fillMaxWidth(0.5f)
                                .padding(5.dp),
                            shape = RectangleShape
                        ) {
                            Text(
                                text = "Cancel",
                                style = getEloLargeBoldTextWhite()

                            )
                        }
                    },
                    onDismiss = {
                        networkActivityViewModel.setShowNetworkConfigurationDialog(false)
                    }
                )
            }
        }

        @Composable
        fun RemoveCertificateDialog(
            networkActivityViewModel: NetworkActivityViewModel = NetworkActivityViewModel(
                Activity(),
                true
            )
        ) {
            val networkActivityDetails by networkActivityViewModel.networkActivityState.collectAsState()
            if (NetworkActivityViewModel.isPreview) {
                networkActivityViewModel.setShowRemoveCertificateDialog(true)
            }
            if (networkActivityDetails.showRemoveCertificateDialog) {
                SingleTextFieldTwoButtonDialog(
                    title = "Remove Certificate",
                    textFieldValueHint = "cert alias",
                    textFieldValue = networkActivityDetails.removeAlias,
                    onValueChange = {
                        networkActivityViewModel.setRemoveAlias(it)
                    },
                    onConfirmClick = {
                        networkActivityViewModel.setShowRemoveCertificateDialog(false)
                        NetworkActivityOnClickListeners.onRemoveCertificateClick(
                            networkActivityViewModel
                        )
                    },
                    onCancelClick = {
                        networkActivityViewModel.setShowRemoveCertificateDialog(false)
                    }
                )
            }
        }
    }
}

@Composable
@Preview(showBackground = true)
fun SetNetworkDialogPreview() {
    SetNetworkDialog()
}

@Composable
@Preview(showBackground = true)
fun RemoveCertificateDialogPreview() {
    RemoveCertificateDialog()
}
