/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.networkActivity

import android.app.Activity
import android.content.Intent
import android.os.Build
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.eloview.sdk.networkManager.CertificateConfig
import com.eloview.sdk.networkManager.ESTConfig
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.common.CommonComposables.Companion.AccountManagerBtn
import com.eloview.sdk.testapp.common.CommonComposables.Companion.BottomButtonBar
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloOutputText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.common.CommonComposables.Companion.RowTextAndSwitchContent
import com.eloview.sdk.testapp.networkActivity.NetworkActivity.Companion.networkActivityCertificateResultLauncher
import com.eloview.sdk.testapp.networkActivity.NetworkActivity.Companion.networkActivityLocalCertificateResultLauncher
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.ui.theme.getEloLargeBoldTextPrimary
import com.eloview.sdk.testapp.viewmodel.NetworkActivityViewModel

class NetworkActivity : ComponentActivity() {
    companion object {
        const val TAG = "NetworkActivity"
        var networkActivityCertificateResultLauncher: ActivityResultLauncher<Intent>? = null
        var networkActivityLocalCertificateResultLauncher: ActivityResultLauncher<Intent>? = null
    }

    private lateinit var mNetworkActivityViewModel: NetworkActivityViewModel

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        mNetworkActivityViewModel = NetworkActivityViewModel(
            applicationContext,
            false
        )
        if (savedInstanceState != null) {
            // persist output across UI reconfigurations
            savedInstanceState.getString("output")
                ?.let { mNetworkActivityViewModel.setOutput(it) }
        }
        networkActivityCertificateResultLauncher =
            registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
                try {
                    if (result.resultCode == RESULT_OK) {
                        val estConfigModel: ESTConfig? =
                            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                                result.data?.getParcelableExtra("est", ESTConfig::class.java)
                            } else {
                                result.data?.getParcelableExtra("est")
                            }
                        mNetworkActivityViewModel.setOutput("Installing Certificate...")
                        mNetworkActivityViewModel.networkServiceManager?.installCertificate(
                            this,
                            estConfigModel,
                            mNetworkActivityViewModel.networkHandler
                        )
                    } else {
                        mNetworkActivityViewModel.setOutput("Activity ResultCode not ok")
                        Log.d(TAG, "Activity ResultCode not ok")
                    }
                } catch (e: Exception) {
                    e.printStackTrace()
                }
                Log.d(TAG, "onActivityResult: $result")
            }
        networkActivityLocalCertificateResultLauncher =
            registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
                try {
                    if (result.resultCode == RESULT_OK) {
                        Log.d(TAG, "LOCAL_CERTIFICATE_CODE")
                        val resultString: String = result.data?.getStringExtra("result").toString()
                        if (resultString == "null") {
                            val certificateConfig: CertificateConfig? = result.data?.getParcelableExtra("certificate")
                            if (certificateConfig != null) {
                                installCertificate(certificateConfig)
                            }
                            mNetworkActivityViewModel.setOutput("Certificate Installed Success")
                        } else {
                            mNetworkActivityViewModel.setOutput(resultString)
                        }
                    } else {
                        mNetworkActivityViewModel.setOutput("Activity ResultCode not ok")
                        Log.d(TAG, "Activity ResultCode not ok")
                    }
                } catch (e: Exception) {
                    e.printStackTrace()
                }
                Log.d(TAG, "onActivityResult: $result")
            }

        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    NetworkActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        networkActivityViewModel = mNetworkActivityViewModel,
                        activity = this
                    )
                }
            }
        }
    }
    private fun installCertificate(certificateConfig: CertificateConfig) {
        if (mNetworkActivityViewModel.networkServiceManager == null) {
            Log.e(LocalCertificateActivity.TAG, "NetworkServiceManager is not initialized.")
            return
        }
        mNetworkActivityViewModel.networkServiceManager?.installLocalCertificate(
            applicationContext,
            certificateConfig
        )
    }

    override fun onSaveInstanceState(outState: Bundle) {
        outState.putString("output", mNetworkActivityViewModel.getOutput())
        super.onSaveInstanceState(outState)
    }
}

@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
@Composable
fun NetworkActivityScreen(
    modifier: Modifier = Modifier,
    networkActivityViewModel: NetworkActivityViewModel = NetworkActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: NetworkActivity = NetworkActivity()
) {
    val networkActivityDetails by networkActivityViewModel.networkActivityState.collectAsState()
    val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()

    NetworkDialogs(networkActivityViewModel)

    if (globalDetails.authorized && !networkActivityDetails.serviceConnected) {
        networkActivityViewModel.bindService()
    }
    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,

            modifier = Modifier.padding(top= dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp)
                .fillMaxSize()
                .wrapContentHeight()
        ) {


            EloScreenTitleText("Network API")

            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier. padding(top= dimensionResource(id=com.intuit.sdp.R.dimen._10sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {
                AccountManagerBtn(activity)

                EloWideButton(
                    text = "Get WiFi Mac",
                    onClick = {
                        networkActivityViewModel.setOutput("")
                        NetworkActivityOnClickListeners.onGetWiFiMacClick(networkActivityViewModel)
                    }
                )


                EloWideButton(
                    text = "Get Ethernet Mac",
                    onClick = {
                        networkActivityViewModel.setOutput("")
                        NetworkActivityOnClickListeners.onGetEthernetMacClick(
                            networkActivityViewModel
                        )
                    }
                )

                EloWideButton(
                    text = "Set Network",
                    onClick = {
                        networkActivityViewModel.setOutput("")
                        networkActivityViewModel.setShowNetworkConfigurationDialog(true)
                    }
                )

                EloWideButton(
                    text = "Get Network",
                    onClick = {
                        networkActivityViewModel.setOutput("")
                        NetworkActivityOnClickListeners.onGetNetworkClick(networkActivityViewModel)
                    }
                )

                EloWideButton(
                    text = "Get DHCP Server HostName",
                    onClick = {
                        networkActivityViewModel.setOutput("")
                        NetworkActivityOnClickListeners.onGetDHCPServerHostNameClick(
                            networkActivityViewModel
                        )
                    }
                )

                EloWideButton(
                    text = "Set Certificate Configuration",
                    onClick = {
                        networkActivityViewModel.setOutput("")
                        networkActivityCertificateResultLauncher?.launch(
                            Intent(
                                activity,
                                ESTCertificateActivity::class.java
                            )
                        )
                    }
                )

                EloWideButton(
                    text = "Remove Certificate Configuration",
                    onClick = {
                        networkActivityViewModel.setOutput("")
                        networkActivityViewModel.setShowRemoveCertificateDialog(true)
                    }
                )

                EloWideButton(
                    text = "Install Local Certificate",
                    onClick = {
                        networkActivityViewModel.setOutput("")
                        networkActivityLocalCertificateResultLauncher?.launch(
                            Intent(
                                activity,
                                LocalCertificateActivity::class.java
                            )
                        )
                    }
                )
                RowTextAndSwitchContent(mainLabel = "Eth State",
                    label = "Off / On",
                    state = networkActivityDetails.ethState,
                    onStateChange = {
                        networkActivityViewModel.setOutput("")
                        networkActivityViewModel.setEthState(it)
                        NetworkActivityOnClickListeners.onEthStateClick(
                            networkActivityViewModel
                        )
                    }, verticalDivider = false,
                    center = true)



                EloOutputText(networkActivityDetails.output)


                BottomButtonBar(activity) { networkActivityViewModel.setOutput("") }
            }
        }
    }
}

@Composable
fun NetworkDialogs(networkActivityViewModel: NetworkActivityViewModel) {
    NetworkActivityDialogs.SetNetworkDialog(networkActivityViewModel)
    NetworkActivityDialogs.RemoveCertificateDialog(networkActivityViewModel)
}
