/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.networkActivity

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.util.Log
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.eloview.sdk.networkManager.CertificateConfig
import com.eloview.sdk.networkManager.CertificateType
import com.eloview.sdk.testapp.common.CommonComposables.Companion.APIButton
import com.eloview.sdk.testapp.common.CommonComposables.Companion.CommonButton
import com.eloview.sdk.testapp.common.CommonComposables.Companion.DoubleRow
import com.eloview.sdk.testapp.common.CommonComposables.Companion.SwitchWithLabel
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.viewmodel.LocalCertificateActivityViewModel

class LocalCertificateActivity : ComponentActivity() {
    companion object {
        const val TAG = "LocalCertificateActivity"
        var networkActivityCertificateResultLauncher: ActivityResultLauncher<Intent>? = null
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val networkActivityViewModel = LocalCertificateActivityViewModel()
        networkActivityCertificateResultLauncher =
            registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
                try {
                    if (result.resultCode == RESULT_OK) {
                        val selectedFile: Uri? = result.data?.data
                        val selectedFilePath = Util.getRealPathFromUri(applicationContext, selectedFile)
                        Log.d(TAG, "Selected file path: $selectedFilePath")
                        networkActivityViewModel.setLocalCertPath(selectedFilePath)
                    } else {
                        Log.e(TAG, "Activity ResultCode not ok")
                    }
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    LocalNetworkActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        networkActivityViewModel = networkActivityViewModel,
                        activity = this
                    )
                }
            }
        }
    }

    @Composable
    @Preview(showBackground = true)
    fun LocalNetworkActivityScreen(
        modifier: Modifier = Modifier,
        networkActivityViewModel: LocalCertificateActivityViewModel = LocalCertificateActivityViewModel(),
        activity: LocalCertificateActivity = LocalCertificateActivity()
    ) {
        val networkActivityDetails by networkActivityViewModel.estCertificateActivityState.collectAsState()

        Box(modifier = modifier, contentAlignment = Alignment.Center) {
            Column(
                verticalArrangement = Arrangement.Top,
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier
                    .fillMaxSize()
                    .padding(32.dp)
                    .verticalScroll(rememberScrollState())
            ) {
                Text(
                    text = "Install Local Certificate",
                    textAlign = TextAlign.Center,
                    fontSize = 32.sp,
                    color = MaterialTheme.colorScheme.primary,
                    fontWeight = FontWeight.Bold,
                    modifier = Modifier.padding(top = 24.dp, bottom = 24.dp)
                )

                var selectedCertificateType by remember { mutableStateOf(CertificateType.CA_CERTIFICATE) }

                DoubleRow(
                    {
                        Text(
                            text = "Certificate Type",
                            fontSize = 18.sp,
                            modifier = Modifier.fillMaxWidth(0.5f)
                                .padding(bottom = 14.dp),
                            style = TextStyle(textAlign = TextAlign.Center, fontWeight = FontWeight.Bold)
                        )
                    },
                    {
                        Row(
                            verticalAlignment = Alignment.CenterVertically,
                            horizontalArrangement = Arrangement.Start,
                            modifier = Modifier.fillMaxWidth()
                        ) {
                            Row(
                                verticalAlignment = Alignment.CenterVertically,
                                modifier = Modifier.weight(1f)
                            ) {
                                RadioButton(
                                    selected = selectedCertificateType == CertificateType.CA_CERTIFICATE,
                                    onClick = { selectedCertificateType = CertificateType.CA_CERTIFICATE }
                                )
                                Text(text = "CA", modifier = Modifier.padding(start = 8.dp))
                            }

                            Row(
                                verticalAlignment = Alignment.CenterVertically,
                                modifier = Modifier.weight(1f)
                            ) {
                                RadioButton(
                                    selected = selectedCertificateType == CertificateType.USER_CERTIFICATE,
                                    onClick = { selectedCertificateType = CertificateType.USER_CERTIFICATE }
                                )
                                Text(text = "User", modifier = Modifier.padding(start = 8.dp))
                            }
                        }
                    },
                    center = true,
                    verticalDivider = false
                )

                DoubleRow(
                    {
                        Text(
                            text = "Certificate Name",
                            fontSize = 18.sp,
                            modifier = Modifier.fillMaxWidth(0.5f),
                            style = TextStyle(textAlign = TextAlign.Center, fontWeight = FontWeight.Bold)
                        )
                    },
                    {
                        TextField(
                            value = networkActivityDetails.localCertName,
                            onValueChange = {
                                networkActivityViewModel.setLocalCertName(it)
                            },
                            modifier = Modifier.fillMaxWidth()
                                .padding(bottom = 11.dp),
                            placeholder = { Text("Enter certificate name") }
                        )
                    },
                    center = true,
                    verticalDivider = false
                )

                DoubleRow(
                    {
                        Text(
                            text = "Password \n (Required for User Certificate only)",
                            fontSize = 18.sp,
                            modifier = Modifier.fillMaxWidth(0.5f),
                            style = TextStyle(textAlign = TextAlign.Center, fontWeight = FontWeight.Bold)
                        )
                    },
                    {
                        TextField(
                            value = networkActivityDetails.localCertPassword,
                            onValueChange = {
                                networkActivityViewModel.setLocalCertPassword(it)
                            },
                            modifier = Modifier.fillMaxWidth()
                                .padding(bottom = 11.dp),
                            placeholder = { Text("Enter password") },
                            visualTransformation = PasswordVisualTransformation()
                        )
                    },
                    center = true,
                    verticalDivider = false
                )

                DoubleRow(
                    {
                        Text(
                            text = "Certificate File Path",
                            fontSize = 18.sp,
                            modifier = Modifier.fillMaxWidth(0.5f),
                            style = TextStyle(textAlign = TextAlign.Center, fontWeight = FontWeight.Bold)
                        )
                    },
                    {
                        TextField(
                            value = networkActivityDetails.localCertPath,
                            onValueChange = {
                                networkActivityViewModel.setLocalCertPath(it)
                            },
                            modifier = Modifier.fillMaxWidth()
                                .padding(bottom = 24.dp),
                            placeholder = { Text("/sdcard/cacert.pem") }
                        )
                    },
                    center = true,
                    verticalDivider = false
                )

                APIButton(
                    text = "Select File",
                    onClick = {
                        val intent = Intent(Intent.ACTION_OPEN_DOCUMENT)
                        intent.addCategory(Intent.CATEGORY_OPENABLE)
                        intent.setType("*/*")
                        try {
                            networkActivityCertificateResultLauncher?.launch(intent)
                        } catch (ex: ActivityNotFoundException) {
                            Toast.makeText(
                                activity,
                                "Please install a File Manager.",
                                Toast.LENGTH_SHORT
                            ).show()
                        }
                    },
                    modifier = Modifier.fillMaxWidth(0.25f)
                        .padding(bottom = 24.dp),
                )

                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.SpaceBetween
                ) {
                    APIButton(
                        text = "Cancel",
                        onClick = {
                            activity.setResult(Activity.RESULT_CANCELED)
                            activity.finish()
                        },
                        modifier = Modifier.weight(1f).padding(end = 8.dp)
                    )

                    APIButton(
                        text = "Apply Certificate",
                        onClick = {
                            val certName: String = networkActivityDetails.localCertName
                            val password: String = networkActivityDetails.localCertPassword

                            // if user added path manually
                            val selectedFilePath = networkActivityDetails.localCertPath
                            if (!(selectedFilePath.contains(".pem") || selectedFilePath.contains(
                                    ".cer"
                                ) || selectedFilePath.contains(".crt") || selectedFilePath.contains(
                                    ".key"
                                ) || selectedFilePath.contains(".p12"))
                            ) {
                                val intent = Intent()
                                intent.putExtra("result", "Invalid certificate format")
                                activity.setResult(Activity.RESULT_OK, intent)
                                activity.finish()
                            } else {
                                val certificateType = selectedCertificateType

                                val certificateConfig =
                                    CertificateConfig(
                                        selectedFilePath,
                                        certName,
                                        certificateType
                                    )

                                if (password.isNotEmpty()) {
                                    certificateConfig.userCertPwd = password
                                }

                                val intent = Intent()
                                intent.putExtra("certificate", certificateConfig)
                                activity.setResult(Activity.RESULT_OK, intent)
                                activity.finish()
                            }
                        },
                        modifier = Modifier.weight(1f).padding(start = 8.dp)
                    )
                }
            }
        }
    }
}