/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.networkActivity

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.eloview.sdk.networkManager.ESTConfig
import com.eloview.sdk.testapp.common.CommonComposables.Companion.CommonButton
import com.eloview.sdk.testapp.common.CommonComposables.Companion.CommonExposedDropdownMenu
import com.eloview.sdk.testapp.common.CommonComposables.Companion.DoubleRow
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.networkActivity.ESTCertificateActivity.Companion.networkActivityCertificateResultLauncher
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.ui.theme.getEloBoldStdSecondaryText
import com.eloview.sdk.testapp.ui.theme.getEloBoldStdSecondaryTextstart
import com.eloview.sdk.testapp.ui.theme.getEloSmallSecondaryText
import com.eloview.sdk.testapp.viewmodel.ESTCertificateActivityViewModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class ESTCertificateActivity : ComponentActivity() {
    companion object {
        const val TAG = "ESTCertificateActivity"
        var networkActivityCertificateResultLauncher: ActivityResultLauncher<Intent>? = null
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val networkActivityViewModel = ESTCertificateActivityViewModel()
        networkActivityCertificateResultLauncher =
            registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
                try {
                    if (result.resultCode == RESULT_OK) {
                        val data: Intent? = result.data
                        Log.d(TAG, "onActivityResult: $result")

                        val selectedFile = data!!.data
                        var selectedFilePath: String
                        try {
                            selectedFilePath =
                                Util.getRealPathFromUri(applicationContext, selectedFile)
                            if (!selectedFilePath.endsWith(".pem")) {
                                selectedFilePath = ""
                                Toast.makeText(
                                    applicationContext,
                                    "Please select valid file (ex. abc.pem)",
                                    Toast.LENGTH_SHORT
                                ).show()
                            }
                        } catch (e: Exception) {
                            e.printStackTrace()
                            selectedFilePath = ""
                            Toast.makeText(
                                applicationContext,
                                "Can not select the file",
                                Toast.LENGTH_SHORT
                            ).show()
                        }
                        networkActivityViewModel.setEstCertificatePath(selectedFilePath)
                    } else {
                        Log.e(TAG, "Activity ResultCode not ok")
                    }
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    ESTNetworkActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        networkActivityViewModel = networkActivityViewModel,
                        activity = this
                    )
                }
            }
        }

    }
}

@Composable
@Preview(showBackground = true)
fun ESTNetworkActivityScreen(
    modifier: Modifier = Modifier,
    networkActivityViewModel: ESTCertificateActivityViewModel = ESTCertificateActivityViewModel(),
    activity: ESTCertificateActivity = ESTCertificateActivity()
) {
    val networkActivityDetails by networkActivityViewModel.estCertificateActivityState.collectAsState()

    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Top,
            horizontalAlignment = Alignment.CenterHorizontally,
            modifier = Modifier.padding(top= dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp)
                .fillMaxSize()
                .wrapContentHeight()
        ) {

            EloScreenTitleText("EST Certificate Configuration")


            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier.padding(top= dimensionResource(id=com.intuit.sdp.R.dimen._10sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {


                val doubleRow: @Composable (String, @Composable () -> Unit) -> Unit =
                    { label, content ->
                        Row(
                            modifier = Modifier
                                .border(BorderStroke(width = 3.dp, color = MaterialTheme.colorScheme.primary))
                                .wrapContentHeight(align = Alignment.Top)
                                .fillMaxWidth()
                                .padding(
                                    start = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp,
                                    end = dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp,
                                    top= dimensionResource(id=com.intuit.sdp.R.dimen._6sdp).value.dp,
                                    bottom = dimensionResource(id=com.intuit.sdp.R.dimen._6sdp).value.dp),

                            verticalAlignment = Alignment.CenterVertically,
                            horizontalArrangement = Arrangement.spacedBy(dimensionResource(id=com.intuit.sdp.R.dimen._5sdp).value.dp)

                        ) {
                            Box(
                                modifier = Modifier
                                    .weight(0.46f)
                                    .fillMaxHeight()
                            ) {
                                Text(
                                    text = label,
                                    textAlign = TextAlign.Start,
                                    style = getEloBoldStdSecondaryText()
                                )
                            }
                            Box(
                                modifier = Modifier
                                    .weight(1f),
                                contentAlignment = Alignment.CenterStart
                            ) {
                                content()
                            }
                        }

                        Spacer(modifier = Modifier.fillMaxWidth().height(
                            dimensionResource(id=com.intuit.sdp.R.dimen._6sdp).value.dp
                        ))
                    }

                doubleRow("Alias") {

                    TextField(
                        value = networkActivityDetails.estAlias,
                        onValueChange = {
                            networkActivityViewModel.setEstAlias(it)
                        },
                        modifier = Modifier.fillMaxWidth(),
                        textStyle = getEloBoldStdSecondaryTextstart(),
                        placeholder = { Text("test", style = getEloSmallSecondaryText()) }
                    )

                }

                doubleRow("EST URL") {

                    TextField(
                        value = networkActivityDetails.estUrl, onValueChange = {
                            networkActivityViewModel.setEstUrl(it)
                        },
                        modifier = Modifier.fillMaxWidth(),
                        textStyle = getEloBoldStdSecondaryTextstart(),
                        placeholder = { Text("192.168.1.2", style = getEloSmallSecondaryText()) }
                    )

                }

                doubleRow("Port") {
                    var estPortInput by remember { mutableStateOf(networkActivityDetails.estPort?.toString() ?: "") }

                    TextField(
                        value = estPortInput,
                        onValueChange = {
                            estPortInput = it
                            val port = it.toIntOrNull()
                            if (port != null) {
                                networkActivityViewModel.setEstPort(port)
                            } else {
                                networkActivityViewModel.setEstPort(8085) // fallback only on invalid input
                            }
                        },
                        modifier = Modifier.fillMaxWidth(),
                        textStyle = getEloBoldStdSecondaryTextstart(),
                        placeholder = { Text("8085", style = getEloSmallSecondaryText()) },
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number)
                    )

                }

                doubleRow("Username") {

                    TextField(
                        value = networkActivityDetails.estUserName, onValueChange = {
                            networkActivityViewModel.setEstUserName(it)
                        },
                        modifier = Modifier.fillMaxWidth(),
                        textStyle = getEloBoldStdSecondaryTextstart(),
                        placeholder = { Text("estuser", style = getEloSmallSecondaryText()) }
                    )

                }

                doubleRow("Password") {

                    TextField(
                        value = networkActivityDetails.estPassword, onValueChange = {
                            networkActivityViewModel.setEstPassword(it)
                        },
                        modifier = Modifier.fillMaxWidth(),
                        textStyle = getEloBoldStdSecondaryTextstart(),
                        placeholder = { Text("estpassword", style = getEloSmallSecondaryText()) }
                    )

                }


                doubleRow("OpenSSL Certificate Path") {

                    TextField(
                        value = networkActivityDetails.estCertificatePath,
                        onValueChange = {
                            networkActivityViewModel.setEstCertificatePath(it)
                        },
                        modifier = Modifier.fillMaxWidth(),
                        textStyle = getEloBoldStdSecondaryTextstart(),
                        placeholder = { Text("/sdcard/cacert.pem", style = getEloSmallSecondaryText()) }
                    )

                }


                EloWideButton(
                    text = "Select File",
                    onClick = {
                        val intent = Intent(Intent.ACTION_OPEN_DOCUMENT)
                        intent.addCategory(Intent.CATEGORY_OPENABLE)
                        intent.setType("*/*")
                        try {
                            networkActivityCertificateResultLauncher?.launch(intent)
                        } catch (ex: ActivityNotFoundException) {
                            Toast.makeText(
                                activity,
                                "Please install a File Manager.",
                                Toast.LENGTH_SHORT
                            ).show()
                        }
                    }
                )

                Row(
                    modifier = Modifier
                        .wrapContentHeight(align = Alignment.Top)
                        .fillMaxWidth()
                        .padding(
                            top= dimensionResource(id=com.intuit.sdp.R.dimen._3sdp).value.dp,
                            bottom = dimensionResource(id=com.intuit.sdp.R.dimen._3sdp).value.dp),

                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.spacedBy(dimensionResource(id=com.intuit.sdp.R.dimen._5sdp).value.dp)

                )
                    {
                        Box(modifier = Modifier.weight(1f)){
                            EloWideButton(
                                text = "Cancel",
                                onClick = {
                                    activity.setResult(Activity.RESULT_CANCELED)
                                    activity.finish()
                                }
                            )
                        }

                        Box(modifier = Modifier.weight(1f)) {

                            EloWideButton(
                                text = "Apply Certificate",
                                onClick = {
                                    CoroutineScope(Dispatchers.IO).launch {
                                        val alias: String = networkActivityDetails.estAlias
                                        val estUrl: String = networkActivityDetails.estUrl
                                        val port: String = networkActivityDetails.estPort.toString()
                                        val username: String = networkActivityDetails.estUserName
                                        val password: String = networkActivityDetails.estPassword

                                        // if user added path manually
                                        val selectedFilePath = networkActivityDetails.estCertificatePath

                                        // ESTConfig estConfig = new ESTConfig("192.168.48.28", "8085", "estuser", "estpwd", "test", "/sdcard/cacert.pem", "");
                                        val estConfig = ESTConfig().apply {
                                            this.alias = alias
                                            this.url = estUrl
                                            this.port = port
                                            this.estUsername = username
                                            this.estPassword = password
                                            this.certificateFilePath = selectedFilePath
                                        }
                                        withContext(Dispatchers.Main) {
                                            val intent = Intent()
                                            intent.putExtra("est", estConfig)
                                            activity.setResult(Activity.RESULT_OK, intent)
                                            activity.finish()
                                        }
                                    }
                                }
                            )
                        }
                    }

            }
        }
    }
}