/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.memoryActivity

import android.app.Activity
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.common.CommonComposables.Companion.AccountManagerBtn
import com.eloview.sdk.testapp.common.CommonComposables.Companion.BottomButtonBar
import com.eloview.sdk.testapp.common.CommonComposables.Companion.CommonButton
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloOutputText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.eloActivity.EloActivityScreen
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.ui.theme.getEloLargeBoldTextPrimary
import com.eloview.sdk.testapp.viewmodel.MemoryActivityViewModel

class MemoryActivity : ComponentActivity() {
    private lateinit var mMemoryActivityViewModel: MemoryActivityViewModel
    companion object {
        const val TAG = "MemoryActivity"
    }
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        mMemoryActivityViewModel = MemoryActivityViewModel(
            applicationContext,
            false
        )
        if (savedInstanceState != null) {
            // persist output across UI reconfigurations
            savedInstanceState.getString("output")
                ?.let { mMemoryActivityViewModel.setOutput(it) }
        }
        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    MemoryActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        memoryActivityViewModel = mMemoryActivityViewModel,
                        activity = this
                    )
                }
            }
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        outState.putString("output", mMemoryActivityViewModel.getOutput())
        super.onSaveInstanceState(outState)
    }
}

@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
@Composable
fun MemoryActivityScreen(
    modifier: Modifier = Modifier,
    memoryActivityViewModel: MemoryActivityViewModel = MemoryActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: MemoryActivity = MemoryActivity()
) {
    val memoryActivityDetails by memoryActivityViewModel.memoryActivityState.collectAsState()
    val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()

    if (globalDetails.authorized && !memoryActivityDetails.serviceConnected) {
        memoryActivityViewModel.bindService()
    }

    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,

            modifier = Modifier.padding(top= dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp)
                .fillMaxSize()
                .wrapContentHeight()
        ) {

            EloScreenTitleText("Memory API")

            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {
                TopButtonsComposable(memoryActivityViewModel, activity)
            }
        }
    }
}

@Composable
//@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
fun TopButtonsComposable(
    memoryActivityViewModel: MemoryActivityViewModel = MemoryActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: Activity = Activity()
) {
    val memoryActivityDetails by memoryActivityViewModel.memoryActivityState.collectAsState()
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxSize()
    ) {
        AccountManagerBtn(activity)
        EloWideButton(
            text = "Get Memory Info",
            onClick = {
                memoryActivityViewModel.setOutput("")
                MemoryActivityOnClickListeners.onGetMemoryInfoClick(memoryActivityViewModel)
            }
        )
        EloOutputText(memoryActivityDetails.output)



        BottomButtonBar(activity) { memoryActivityViewModel.setOutput("") }
    }
}
