/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.infoActivity

import android.app.Activity
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.common.CommonComposables.Companion.AccountManagerBtn
import com.eloview.sdk.testapp.common.CommonComposables.Companion.BottomButtonBar
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloOutputText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.viewmodel.InfoActivityViewModel

class InfoActivity : ComponentActivity() {
    private lateinit var mInfoActivityViewModel: InfoActivityViewModel
    companion object {
        const val TAG = "InfoActivity"
    }
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        mInfoActivityViewModel = InfoActivityViewModel(
            applicationContext,
            false
        )
        if (savedInstanceState != null) {
            // persist output across UI reconfigurations
            savedInstanceState.getString("output")
                ?.let { mInfoActivityViewModel.setOutput(it) }
        }
        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    InfoActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        infoActivityViewModel = mInfoActivityViewModel,
                        activity = this
                    )
                }
            }
        }
    }

    override fun onStop() {
        super.onStop()
        mInfoActivityViewModel.setOutput("")
    }

    override fun onSaveInstanceState(outState: Bundle) {
        outState.putString("output", mInfoActivityViewModel.getOutput())
        super.onSaveInstanceState(outState)
    }
}

@Composable
@Preview(showBackground = true)
fun InfoActivityScreen(
    modifier: Modifier = Modifier,
    infoActivityViewModel: InfoActivityViewModel = InfoActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: InfoActivity = InfoActivity()
) {
    val infoActivityDetails by infoActivityViewModel.infoActivityState.collectAsState()
    val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()

    if (globalDetails.authorized && !infoActivityDetails.serviceConnected) {
        infoActivityViewModel.bindService()
    }

    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
            modifier = Modifier.padding(top= dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
            start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
            end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp)
                .fillMaxSize()
                .wrapContentHeight()
        ) {

            EloScreenTitleText("Info API")
            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier =  Modifier. padding(top= dimensionResource(id=com.intuit.sdp.R.dimen._10sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {
                AccountManagerBtn(activity)
                EloWideButton(
                    text = "Get Device Info",
                    onClick = {
                        infoActivityViewModel.setOutput("")
                        InfoActivityOnClickListeners.onGetDeviceInfoClick(infoActivityViewModel)
                    }
                )
                EloWideButton(
                    text = "Is SDK Supported",
                    onClick = {
                        infoActivityViewModel.setOutput("")
                        InfoActivityOnClickListeners.onIsSdkSupportedClick(infoActivityViewModel)
                    }
                )
                EloWideButton(
                    text = "Get SDK Version",
                    onClick = {
                        infoActivityViewModel.setOutput("")
                        InfoActivityOnClickListeners.onGetSdkVersionClick(infoActivityViewModel)
                    }
                )
                EloWideButton(
                    text = "Get System CPU Usage",
                    onClick = {
                        infoActivityViewModel.setOutput("")
                        InfoActivityOnClickListeners.onGetSystemCpuUsageClick(infoActivityViewModel)
                    }
                )
                EloWideButton(
                    text = "Register Device Info Receiver",
                    onClick = {
                        infoActivityViewModel.setOutput("")
                        InfoActivityOnClickListeners.onRegisterDeviceInfoReceiverClick(
                            infoActivityViewModel
                        )
                    }
                )
                EloWideButton(
                    text = "Unregister Device Info Receiver",
                    onClick = {
                        infoActivityViewModel.setOutput("")
                        InfoActivityOnClickListeners.onUnregisterDeviceInfoReceiverClick(
                            infoActivityViewModel
                        )
                    }
                )

                EloOutputText(infoActivityDetails.output)


                BottomButtonBar(activity) { infoActivityViewModel.setOutput("") }
            }
        }
    }

}