/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.eloActivity

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.eloview.sdk.testapp.AccountManagerActivity
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.common.CommonComposables.Companion.AccountManagerBtn
import com.eloview.sdk.testapp.common.CommonComposables.Companion.BottomButtonBar
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloOutputText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.systemActivity.SystemActivity.Companion.TAG
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.viewmodel.EloActivityViewModel

class EloActivity : ComponentActivity() {
    private lateinit var mEloActivityViewModel: EloActivityViewModel
    companion object {
        const val TAG = "EloActivity"
    }
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        mEloActivityViewModel = EloActivityViewModel(
            applicationContext,
            false
        )
        if (savedInstanceState != null) {
            // persist output across UI reconfigurations
            savedInstanceState.getString("output")
                ?.let { mEloActivityViewModel.setOutput(it) }
        }

        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    EloActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        eloActivityViewModel = mEloActivityViewModel,
                        activity = this
                    )
                }
            }
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        outState.putString("output", mEloActivityViewModel.getOutput())
        super.onSaveInstanceState(outState)
    }
}

@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
@Composable
fun EloActivityScreen(
    modifier: Modifier = Modifier,
    eloActivityViewModel: EloActivityViewModel = EloActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: EloActivity = EloActivity()
) {
    val eloActivityDetails by eloActivityViewModel.eloActivityState.collectAsState()
    val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()

    EloDialogs(eloActivityViewModel)
    if (globalDetails.authorized && !eloActivityDetails.serviceConnected) {
        eloActivityViewModel.bindService()  // Required for "Switch to Android Home Mode" button
    }

    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
            modifier = Modifier.padding(top= dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp)
                .fillMaxSize()
                .wrapContentHeight()
        ) {

            EloScreenTitleText("EloView API")

            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier =  Modifier. padding(top= dimensionResource(id=com.intuit.sdp.R.dimen._10sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {
                TopButtonsComposable(eloActivityViewModel, activity)
            }
        }
    }
}

@Composable
//@Preview(showBackground = true, widthDp = 640, heightDp = 1920)
fun TopButtonsComposable(
    eloActivityViewModel: EloActivityViewModel = EloActivityViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: Activity = Activity()
) {
    val eloActivityDetails by eloActivityViewModel.eloActivityState.collectAsState()
    val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxSize()
    ) {
        AccountManagerBtn(activity)
        EloWideButton(
            text = "Set Offline Play Content",
            onClick = {
                eloActivityViewModel.setOutput("")
                eloActivityViewModel.setShowOfflinePlayContentDialog(true)
            }
        )
        EloWideButton(
            text = "Get Play Content",
            onClick = {
                eloActivityViewModel.setOutput("")
                EloActivityOnClickListeners.onGetOfflinePlayContentClick(
                    eloActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Get Play Content Status",
            onClick = {
                eloActivityViewModel.setOutput("")
                EloActivityOnClickListeners.onGetOfflinePlayContentStatusClick(
                    eloActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Get Device Online Status",
            onClick = {
                eloActivityViewModel.setOutput("")
                EloActivityOnClickListeners.onGetDeviceOnlineStatusClick(
                    eloActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Enable WiFi Option in Quick Settings Without Admin Password",
            onClick = {
                eloActivityViewModel.setOutput("")
                EloActivityOnClickListeners.onEnableWifiOptionInQuickSettingsWithoutAdminPasswordClick(
                    eloActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Disable WiFi Option in Quick Settings Without Admin Password",
            onClick = {
                eloActivityViewModel.setOutput("")
                EloActivityOnClickListeners.onDisableWifiOptionInQuickSettingsWithoutAdminPasswordClick(
                    eloActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Enable Bluetooth Option in Quick Settings Without Admin Password",
            onClick = {
                eloActivityViewModel.setOutput("")
                EloActivityOnClickListeners.onEnableBluetoothOptionInQuickSettingsWithoutAdminPasswordClick(
                    eloActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Disable Bluetooth Option in Quick Settings Without Admin Password",
            onClick = {
                eloActivityViewModel.setOutput("")
                EloActivityOnClickListeners.onDisableBluetoothOptionInQuickSettingsWithoutAdminPasswordClick(
                    eloActivityViewModel
                )
            }
        )
        EloWideButton(
            text = "Switch to Android Home Mode",
            onClick = {
                eloActivityViewModel.setOutput("")
                EloActivityOnClickListeners.onSwitchToAndroidHomeModeClick(
                    eloActivityViewModel
                )
            }
        )

        EloOutputText(eloActivityDetails.output)
        BottomButtonBar(activity) { eloActivityViewModel.setOutput("") }
    }
}



@Composable
fun EloDialogs(eloActivityViewModel: EloActivityViewModel) {
    EloActivityDialogs.OfflinePlayContentDialog(eloActivityViewModel)
}
