/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.common;

import android.app.Activity;
import android.content.ContentUris;
import android.content.Context;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.provider.DocumentsContract;
import android.provider.MediaStore;
import android.util.Log;
import android.view.View;
import android.view.inputmethod.InputMethodManager;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

public class Util {

    public final static String TAG = "Util";

    public static void hideKeyboard(Activity activity) {
        InputMethodManager imm = (InputMethodManager) activity.getSystemService(Activity.INPUT_METHOD_SERVICE);
        //Find the currently focused view, so we can grab the correct window token from it.
        View view = activity.getCurrentFocus();
        //If no view currently has focus, create a new one, just so we can grab a window token from it
        if (view == null) {
            view = new View(activity);
        }
        assert imm != null;
        imm.hideSoftInputFromWindow(view.getWindowToken(), 0);
    }

    public static boolean isAndroidQDevice() {
        Log.v(TAG, "Model " + Build.MODEL);
        switch (Build.MODEL) {
            case "10in-I-Series-4":
            case "15in-I-Series-4":
            case "22in-I-Series-4":
            case "10in-I-Series-4-Value":
            case "15in-I-Series-4-Value":
            case "22in-I-Series-4-Value":
            case "15in-I-Series-4-USB-C":
            case "Backpack-4":
            case "Backpack-4-Value":
            case "15in-I-Series-4-LS":
            case "15in-I-Series-4-USBC-V":
            case "M50":
            case "M50C":
            case "M60-Pay":
            case "M60C-Pay":
            case "7in-I-Series-Pay":
            case "M100":
            case "M100C": {
                return true;
            }
            default:
                return false;
        }
    }

    public static boolean isAndroidUDevice() {
        Log.v(TAG, "Model " + Build.MODEL);
        switch (Build.MODEL) {
            case "10in-I-Series-5":
            case "15in-I-Series-5":
            case "22in-I-Series-5":
            case "EMC-M51":
            case "EMC-M51C": {
                return true;
            }
            default:
                return false;
        }
    }

    public static boolean isValueDevice() {
        return Build.DISPLAY.toLowerCase().contains("5.") && Build.DISPLAY.toLowerCase().contains("+v") ||
                (Build.DISPLAY.toLowerCase().contains("5.") && Build.DISPLAY.toLowerCase().contains("+av")) ||
                (Build.DISPLAY.toLowerCase().contains("6.") && Build.DISPLAY.toLowerCase().contains("+v")) ||
                (Build.DISPLAY.toLowerCase().contains("6.") && Build.DISPLAY.toLowerCase().contains("+av")) ||
                (Build.DISPLAY.toLowerCase().contains("7.") && Build.DISPLAY.toLowerCase().contains("+v")) ||
                (Build.DISPLAY.toLowerCase().contains("7.") && Build.DISPLAY.toLowerCase().contains("+av"));
    }

    public static boolean isPerformanceDevice() {
        return Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("5.") && Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("+p")
                || (Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("5.") && Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("+ap"))
                || (Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("6.") && Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("+p"))
                || (Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("6.") && Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("+ap"))
                || (Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("7.") && Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("+p"))
                || (Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("7.") && Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("+ap"))
                || (Build.DISPLAY.toLowerCase(Locale.getDefault()).contains("release-keys"));
    }

    public static boolean isHandHeld() {
        return Build.MODEL.contains("M50") || Build.MODEL.contains("M60")
                || Build.MODEL.contains("m50") || Build.MODEL.contains("m60");
    }

    public static boolean isM60() {
        return Build.MODEL.contains("M60") || Build.MODEL.contains("m60");
    }

    public static boolean isM50() {
        return Build.MODEL.contains("M50") || Build.MODEL.contains("m50");
    }

    public static boolean isM100() {
        return Build.MODEL.contains("M100") || Build.MODEL.contains("m100");
    }

    public static boolean isM51() {
        return Build.MODEL.contains("M51") || Build.MODEL.contains("m51");
    }

    public static boolean is10inch() {
        return Build.MODEL.toLowerCase(Locale.getDefault()).contains("10in");
    }
    public static boolean is15inch() {
        return Build.MODEL.toLowerCase(Locale.getDefault()).contains("15in");
    }
    public static boolean is22inch() {
        return Build.MODEL.toLowerCase(Locale.getDefault()).contains("22in");
    }

    public static boolean is7inch() {
        return Build.MODEL.toLowerCase(Locale.getDefault()).contains("7in");
    }

    public static boolean isBackPack() {
        return Build.MODEL.toLowerCase(Locale.ENGLISH).contains("bpack") ||
                Build.MODEL.toLowerCase(Locale.ENGLISH).contains("backpack");
    }

    public static long isSpaceAvailable() {
        int SIZE_MB = 1024;
        StatFs stat = new StatFs(Environment.getExternalStorageDirectory().getPath());
        long availableSpace = stat.getBlockSizeLong() * stat.getAvailableBlocksLong();
        Log.d(TAG, "Available Space in MB : " + availableSpace);
        return availableSpace / (SIZE_MB * SIZE_MB);
    }

    public static String convertUtcMillisToDate(long millis) {
        try {
            SimpleDateFormat sourceFormat = new SimpleDateFormat("MM/dd/yyyy", Locale.getDefault());
            sourceFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            return sourceFormat.format(new Date(millis));
        } catch (Exception e) {
            Log.e(TAG, "convertUtcMillisToDate: " + e);
            return "";
        }
    }

    public static String getDataColumn(Context context, Uri uri, String selection,
                                       String[] selectionArgs) {

        Cursor cursor = null;
        final String column = "_data";
        final String[] projection = {
                column
        };

        try {
            cursor = context.getContentResolver().query(uri, projection, selection, selectionArgs,
                    null);
            if (cursor != null && cursor.moveToFirst()) {
                final int index = cursor.getColumnIndexOrThrow(column);
                return cursor.getString(index);
            }
        } finally {
            if (cursor != null)
                cursor.close();
        }
        return null;
    }


    public static String getFilePath(Context context, Uri uri) {

        String filePath = uri.getPath();
        Log.d("TAG", "getFilePath: uri filePath >> " + filePath);

        final String[] projection = {
                MediaStore.MediaColumns.DISPLAY_NAME
        };
        try (Cursor cursor = context.getContentResolver().query(uri, projection, null, null,
                null)) {
            if (cursor != null && cursor.moveToFirst()) {
                final int index = cursor.getColumnIndexOrThrow(MediaStore.MediaColumns.DISPLAY_NAME);
                String fileName = cursor.getString(index);
                Log.e("TAG", "getFilePath: fileName >>" + fileName);
                if (filePath != null && filePath.contains("Download")) {
                    filePath = filePath.substring(filePath.indexOf("Download") + 9);
                    Log.e("TAG", "getFilePath: trimmed path >> " + filePath);
                    if (fileName.equals(filePath)) {
                        return fileName;
                    } else {
                        return filePath;
                    }
                } else {
                    return fileName;
                }
            }
        }
        return null;
    }

    public static String getRealPathFromUri(final Context context, final Uri uri) {
        // DocumentProvider
        if (DocumentsContract.isDocumentUri(context, uri)) {
            // ExternalStorageProvider
            if (isExternalStorageDocument(uri)) {
                final String docId = DocumentsContract.getDocumentId(uri);
                final String[] split = docId.split(":");
                final String type = split[0];

                // This is for checking Main Memory
                if ("primary".equalsIgnoreCase(type)) {
                    if (split.length > 1) {
                        return Environment.getExternalStorageDirectory() + "/" + split[1];
                    } else {
                        return Environment.getExternalStorageDirectory() + "/";
                    }
                    // This is for checking SD Card
                } else {
                    return "storage" + "/" + docId.replace(":", "/");
                }

            }
            // DownloadsProvider
            else if (isDownloadsDocument(uri)) {
                String fileName = getFilePath(context, uri);
                if (fileName != null) {
                    return Environment.getExternalStorageDirectory().toString() + "/Download/" + fileName;
                }

                String id = DocumentsContract.getDocumentId(uri);
                if (id.startsWith("raw:")) {
                    id = id.replaceFirst("raw:", "");
                    File file = new File(id);
                    if (file.exists())
                        return id;
                }

                final Uri contentUri = ContentUris.withAppendedId(Uri.parse("content://downloads/public_downloads"), Long.parseLong(id));
                return getDataColumn(context, contentUri, null, null);
            }
            // MediaProvider
            else if (isMediaDocument(uri)) {
                final String docId = DocumentsContract.getDocumentId(uri);
                final String[] split = docId.split(":");
                final String type = split[0];

                Uri contentUri = null;
                if ("image".equals(type)) {
                    contentUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI;
                } else if ("video".equals(type)) {
                    contentUri = MediaStore.Video.Media.EXTERNAL_CONTENT_URI;
                } else if ("audio".equals(type)) {
                    contentUri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI;
                }

                final String selection = "_id=?";
                final String[] selectionArgs = new String[]{
                        split[1]
                };

                return getDataColumn(context, contentUri, selection, selectionArgs);
            }
        }
        // MediaStore (and general)
        else if ("content".equalsIgnoreCase(uri.getScheme())) {

            // Return the remote address
            if (isGooglePhotosUri(uri))
                return uri.getLastPathSegment();

            return getDataColumn(context, uri, null, null);
        }
        // File
        else if ("file".equalsIgnoreCase(uri.getScheme())) {
            return uri.getPath();
        }

        return null;
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is ExternalStorageProvider.
     */
    public static boolean isExternalStorageDocument(Uri uri) {
        return "com.android.externalstorage.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is DownloadsProvider.
     */
    public static boolean isDownloadsDocument(Uri uri) {
        return "com.android.providers.downloads.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is MediaProvider.
     */
    public static boolean isMediaDocument(Uri uri) {
        return "com.android.providers.media.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is Google Photos.
     */
    public static boolean isGooglePhotosUri(Uri uri) {
        return "com.google.android.apps.photos.content".equals(uri.getAuthority());
    }

    public static String readFileContent(Context context, Uri fileUri) {
        try (InputStream inputStream = context.getContentResolver().openInputStream(fileUri)) {
            if (inputStream != null) {
                // Read the content from the inputStream
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream))) {
                    StringBuilder stringBuilder = new StringBuilder();
                    String line;
                    while ((line = reader.readLine()) != null) {
                        stringBuilder.append(line);
                    }
                    return stringBuilder.toString();
                } catch (Exception e) {
                    Log.e(TAG, "readFileContent: " + e);
                }
            }
        } catch (IOException e) {
            Log.e(TAG, "readFileContent: " + e);
        }
        return "";
    }

    public static boolean isPackageInstalled(Context context, String pkgName) {
        boolean status = false;
        PackageManager pm = context.getPackageManager();
        try {
            status = pm.getApplicationInfo(pkgName, 0).enabled;
        } catch (PackageManager.NameNotFoundException e) {
            Log.e(TAG, "isPackageInstalled: " + e);
        }
        return status;
    }
}
