package com.eloview.sdk.testapp.common

import android.content.Context
import android.net.ConnectivityManager
import android.net.Uri
import android.os.Environment
import android.util.Log
import java.io.ByteArrayOutputStream
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException

object FileUtils {
    private const val TAG = "FileUtils"
    private const val MAX_FILE_SIZE_MB = 1
    private const val BUFFER_SIZE = 1024
    private const val ZIP_HEADER_1 = 0x50
    private const val ZIP_HEADER_2 = 0x4B
    private const val TARGET_ZIP_NAME = "ELO_AUTHORIZED_FILES.zip"

    fun isZipFile(file: File): Boolean {
        if (!file.exists() || file.isDirectory || file.length() < 4) return false

        return try {
            FileInputStream(file).use { fis ->
                val header = ByteArray(2)
                fis.read(header)
                header[0].toInt() == ZIP_HEADER_1 && header[1].toInt() == ZIP_HEADER_2
            }
        } catch (e: IOException) {
            Log.d(TAG, "isZipFile: $e", e)
            false
        }
    }

    fun isZipFile(context: Context, uri: Uri): Boolean {
        return try {
            context.contentResolver.openInputStream(uri)?.use { inputStream ->
                val header = ByteArray(2)
                if (inputStream.read(header) == 2) {
                    header[0].toInt() == ZIP_HEADER_1 && header[1].toInt() == ZIP_HEADER_2
                } else {
                    false
                }
            } ?: false
        } catch (e: IOException) {
            Log.d(TAG, "isZipFile: $e", e)
            false
        }
    }

    fun convertZipFileToByteArray(zipFile: File): ByteArray {
        return try {
            FileInputStream(zipFile).use { fis ->
                ByteArrayOutputStream().use { baos ->
                    val buffer = ByteArray(BUFFER_SIZE)
                    var bytesRead: Int
                    while (fis.read(buffer).also { bytesRead = it } != -1) {
                        baos.write(buffer, 0, bytesRead)
                    }
                    baos.toByteArray()
                }
            }
        } catch (e: Exception) {
            Log.e(TAG, "convertZipFileToByteArray: $e", e)
            ByteArray(0)
        }
    }

    fun convertZipUriToByteArray(context: Context, uri: Uri): ByteArray {
        return try {
            context.contentResolver.openInputStream(uri)?.use { inputStream ->
                ByteArrayOutputStream().use { baos ->
                    val buffer = ByteArray(8192)
                    var bytesRead: Int
                    while (inputStream.read(buffer).also { bytesRead = it } != -1) {
                        baos.write(buffer, 0, bytesRead)
                    }
                    baos.toByteArray()
                }
            } ?: ByteArray(0) // Return empty if stream is null
        } catch (e: Exception) {
            Log.e("TAG", "convertZipUriToByteArray: $e", e)
            ByteArray(0)
        }
    }

    fun isFileValid(file: File?): Boolean {
        if (file == null) return false

        val fileSizeMB = file.length() / (1024 * 1024)
        if (fileSizeMB > MAX_FILE_SIZE_MB) {
            Log.d(TAG, "File too large: ${file.length()} bytes")
            return false
        }

        if (!isZipFile(file)) {
            Log.d(TAG, "File is not a valid zip")
            return false
        }

        return true
    }

    fun listFilesInEloPay(): String {
        Log.d(TAG, "listFilesInDirectory: ")
        try {
            val folder = File("/data/CastlesFile/protected/")
            val listOfFiles = folder.listFiles()
            val fileNames = StringBuilder()

            if (listOfFiles != null) {
                Log.d(TAG, "listFilesInDirectory: listOfFiles " + listOfFiles.size)
                for (listOfFile in listOfFiles) {
                    if (listOfFile.isFile) {
                        Log.d(TAG, "isFile : " + listOfFile.name)
                        fileNames.append(listOfFile.name).append("\n")
                    } else if (listOfFile.isDirectory) {
                        Log.d(TAG, "isDirectory : " + listOfFile.name)
                    }
                }
            } else {
                Log.d(TAG, "listFilesInDirectory: No files found")
            }
            return fileNames.toString().trim { it <= ' ' }
        } catch (e: java.lang.Exception) {
            Log.d(TAG, "Exception: e")
        }
        return ""
    }

    fun isFileValid(context: Context, uri: Uri): Boolean {
        // Check file size
        try {
            context.contentResolver.openFileDescriptor(uri, "r")?.use { pfd ->
                val fileSizeBytes = pfd.statSize
                val fileSizeMB = fileSizeBytes / (1024 * 1024)
                if (fileSizeMB > MAX_FILE_SIZE_MB) {
                    Log.d(TAG, "File too large: $fileSizeBytes bytes")
                    return false
                }
            }
        } catch (e: Exception) {
            Log.e("TAG", "Error checking file size: $e", e)
            return false
        }

        // Check if it's a valid zip
        if (!isZipFile(context, uri)) {
            Log.d(TAG, "File is not a valid zip")
            return false
        }

        return true
    }

    fun isInternetAvailable(context: Context): Boolean {
        val cm = context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
        val activeNetwork = cm?.activeNetworkInfo
        return activeNetwork?.isConnectedOrConnecting == true
    }

    fun isZipFileExists(): Boolean {
        val targetFile = File(
            Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS),
            TARGET_ZIP_NAME
        )
        val exists = targetFile.exists() && targetFile.isFile
        Log.d(TAG, "Zip file exists: $exists, path: $targetFile")
        return exists
    }

    fun deleteExistingFiles() {
        val downloadsDir =
            Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS)
        val files = downloadsDir.listFiles()

        if (files.isNullOrEmpty()) {
            Log.e(TAG, "No files found in the Downloads directory.")
            return
        }

        files.filter { it.name.startsWith("ELO_AUTHORIZED_FILES") }
            .forEach {
                if (it.delete()) {
                    Log.d(TAG, "Deleted file: ${it.name}")
                } else {
                    Log.e(TAG, "Failed to delete file: ${it.name}")
                }
            }
    }

    fun copyAssetToCache(assetFileName: String, applicationContext: Context): File {
        // Specify the file path in the cache directory
        val outFile = File(applicationContext.cacheDir, assetFileName)

        // Copy the asset to the cache directory
        try {
            applicationContext.assets.open(assetFileName).use { inputStream ->
                FileOutputStream(outFile).use { outputStream ->
                    val buffer = ByteArray(1024)
                    var length: Int
                    while ((inputStream.read(buffer).also { length = it }) > 0) {
                        outputStream.write(buffer, 0, length)
                    }
                }
            }
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
        }
        return outFile // Return the file object
    }
}
