/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.common

import android.content.res.Configuration
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.ScrollState
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.widthIn
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.Checkbox
import androidx.compose.material3.CheckboxDefaults
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DatePickerDialog
import androidx.compose.material3.DatePickerState
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TimePicker
import androidx.compose.material3.TimePickerLayoutType
import androidx.compose.material3.TimePickerState
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.material3.rememberTimePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.key
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.drawBehind
import androidx.compose.ui.geometry.CornerRadius
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalConfiguration
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.LocalSoftwareKeyboardController
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.text.input.ImeAction
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import com.eloview.sdk.testapp.common.CommonComposables.Companion.CommonExposedDropdownMenu
import com.eloview.sdk.testapp.common.CommonComposables.Companion.DoubleRow
import com.eloview.sdk.testapp.common.CommonComposables.Companion.MultiComboBox
import com.eloview.sdk.testapp.ui.theme.getEloBoldStdText
import com.eloview.sdk.testapp.ui.theme.getEloNormalStdPrimaryText
import com.eloview.sdk.testapp.ui.theme.getEloNormalStdSecondaryText
import com.eloview.sdk.testapp.ui.theme.getEloSmallBoldSecondaryText
import com.eloview.sdk.testapp.ui.theme.getEloSmallSecondaryText

class CommonDialogComposables {
    @Composable
    @Preview(showBackground = true)
    fun TwoButtonDialogPreview() {
        TwoButtonDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun SingleTextFieldTwoButtonDialogPreview() {
        SingleTextFieldTwoButtonDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun DoubleTextFieldTwoButtonDialogPreview() {
        DoubleTextFieldTwoButtonDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun TripleTextFieldTwoButtonDialogPreview() {
        TripleTextFieldTwoButtonDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun MyAlertDialogPreview() {
        MyAlertDialog()
    }

    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    @Preview(showBackground = true)
    fun CommonTimePickerDialogPreview() {
        CommonTimePickerDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun TimePickerDialogPreview() {
        TimePickerDialog(
            orientation = Configuration.ORIENTATION_PORTRAIT
        ) {
            Text("Time Picker Content", style = getEloNormalStdPrimaryText())
        }
    }

    @Composable
    @Preview(showBackground = true)
    fun SingleDropDownTwoButtonDialogPreview() {
        SingleDropDownTwoButtonDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun SingleTextFieldSingleDropDownTwoButtonDialogPreview() {
        SingleTextFieldSingleDropDownTwoButtonDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun SingleTextFieldSingleCheckBoxTwoButtonDialogPreview() {
        SingleTextFieldSingleCheckBoxTwoButtonDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun DoubleTextFieldSingleCheckBoxTwoButtonDialogPreview() {
        DoubleTextFieldSingleCheckBoxTwoButtonDialog()
    }

    @Composable
    @Preview(showBackground = true)
    fun SimpleDialogWithATextPreview() {
        SimpleDialogWithAText()
    }

    companion object {
        @Composable
        fun TwoButtonDialog(
            title: String = "Dialog Title",
            content: String = "Dialog Content",
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {}
        ) {
            MyAlertDialog(
                title = { Text(text = title, style = getEloNormalStdPrimaryText()) },
                content = {
                    Text(text = content, style = getEloSmallBoldSecondaryText())
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }


        @Composable
        fun SingleTextFieldTwoButtonDialog(
            title: String = "Dialog Title",
            textFieldValue: String = "Dialog TextField",
            textFieldKeyboardType: KeyboardType = KeyboardType.Text,
            isPinField: Boolean = false,
            textFieldValueHint: String = "",
            onValueChange: (String) -> Unit = {},
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {}
        ) {
            MyAlertDialog(
                title = { Text(text = title, style = getEloNormalStdPrimaryText()) },
                content = {
                    OutlinedTextField(
                        keyboardOptions = KeyboardOptions(
                            keyboardType = textFieldKeyboardType
                        ),
                        placeholder = {
                            Text(textFieldValueHint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue,
                        onValueChange = { input ->
                            if (isPinField) {
                                onValueChange(input.filter { it.isDigit() })
                            } else {
                                onValueChange(input)
                            }
                        })
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun DoubleTextFieldTwoButtonDialog(
            title: String = "Dialog Title",
            textField1KeyboardType: KeyboardType = KeyboardType.Text,
            textField2KeyboardType: KeyboardType = KeyboardType.Text,
            textFieldValue1Hint: String = "",
            textFieldValue2Hint: String = "",
            textFieldValue1: String = "Dialog TextField 1",
            textFieldValue2: String = "Dialog TextField 2",
            onValueChange1: (String) -> Unit = {},
            onValueChange2: (String) -> Unit = {},
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {}
        ) {
            MyAlertDialog(
                title = { Text(text = title, style = getEloNormalStdPrimaryText()) },
                content = {
                    OutlinedTextField(
                        keyboardOptions = KeyboardOptions(
                            keyboardType = textField1KeyboardType
                        ),
                        placeholder = {
                            Text(textFieldValue1Hint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue1,
                        onValueChange = {
                            onValueChange1(it)
                        },
                        modifier = Modifier.padding(bottom = 8.dp)
                    )
                    OutlinedTextField(
                        keyboardOptions = KeyboardOptions(
                            keyboardType = textField2KeyboardType
                        ),
                        placeholder = {
                            Text(text = textFieldValue2Hint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue2,
                        onValueChange = {
                            onValueChange2(it)
                        })
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun TripleTextFieldTwoButtonDialog(
            title: String = "Dialog Title",
            textFieldValue1Hint: String = "",
            textFieldValue2Hint: String = "",
            textFieldValue3Hint: String = "",
            textFieldValue1: String = "",
            textFieldValue2: String = "",
            textFieldValue3: String = "",
            onValueChange1: (String) -> Unit = {},
            onValueChange2: (String) -> Unit = {},
            onValueChange3: (String) -> Unit = {},
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {}
        ) {
            MyAlertDialog(
                title = { Text(text = title, style = getEloNormalStdPrimaryText()) },
                content = {
                    OutlinedTextField(
                        placeholder = {
                            Text(textFieldValue1Hint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue1,
                        onValueChange = {
                            onValueChange1(it)
                        },
                        modifier = Modifier.padding(bottom = 8.dp)
                    )
                    OutlinedTextField(
                        placeholder = {
                            Text(textFieldValue2Hint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue2,
                        onValueChange = {
                            onValueChange2(it)
                        },
                        modifier = Modifier.padding(bottom = 8.dp)
                    )
                    OutlinedTextField(
                        placeholder = {
                            Text(textFieldValue3Hint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue3,
                        onValueChange = {
                            onValueChange3(it)
                        })
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun SingleTextFieldSingleCheckBoxTwoButtonDialog(
            title: String = "Dialog Title",
            textFieldValue: String = "Dialog TextField",
            textFieldValueHint: String = "",
            onValueChange: (String) -> Unit = {},
            checkBoxValue: Boolean = false,
            onCheckBoxChange: (Boolean) -> Unit = {},
            checkBoxText: String = "Dialog CheckBox",
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {}
        ) {
            MyAlertDialog(
                title = { Text(text = title, style = getEloNormalStdPrimaryText()) },
                content = {

                    Row(
                        modifier = Modifier
                            .border(
                                BorderStroke(
                                    width = 2.dp,
                                    color = MaterialTheme.colorScheme.primary
                                ), shape = RoundedCornerShape(3.dp)
                            )
                            .wrapContentHeight(align = Alignment.Top)
                            .fillMaxWidth()
                            .padding(
                                start = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp),
                                end = dimensionResource(id = com.intuit.sdp.R.dimen._5sdp)
                            )
                            .height(IntrinsicSize.Min),
                        verticalAlignment = Alignment.CenterVertically,
                    ) {
                        Checkbox(
                            checkBoxValue,
                            onCheckBoxChange,
                            modifier = Modifier.padding(
                                top = 8.dp,
                                bottom = 8.dp
                            )
                        )


                        Text(text = checkBoxText, style = getEloSmallBoldSecondaryText())

                    }
                    Spacer(
                        modifier = Modifier
                            .fillMaxWidth()
                            .height(dimensionResource(id = com.intuit.sdp.R.dimen._6sdp))
                    )

                    OutlinedTextField(
                        placeholder = {
                            Text(textFieldValueHint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue,
                        onValueChange = {
                            onValueChange(it)
                        })
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun SingleTextFieldDoubleCheckBoxTwoButtonDialog(
            title: String = "Dialog Title",
            textFieldValue: String = "Dialog TextField",
            textFieldHint: String = "",
            onValueChange: (String) -> Unit = {},
            checkBoxValue1: Boolean = false,
            onCheckBoxChange1: (Boolean) -> Unit = {},
            checkBoxText1: String = "Dialog CheckBox 1",
            checkBoxValue2: Boolean = false,
            onCheckBoxChange2: (Boolean) -> Unit = {},
            checkBoxText2: String = "Dialog CheckBox 2",
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {}
        ) {
            MyAlertDialog(
                title = { Text(text = title) },
                content = {
                    DoubleRow(
                        {
                            Checkbox(
                                checkBoxValue1,
                                onCheckBoxChange1,
                                modifier = Modifier.padding(
                                    start = 10.dp,
                                )
                            )
                        },
                        {
                            Text(text = checkBoxText1)
                        },
                        verticalDivider = false,
                        center = true
                    )
                    DoubleRow(
                        {
                            Checkbox(
                                checkBoxValue2,
                                onCheckBoxChange2,
                                modifier = Modifier.padding(
                                    start = 10.dp,
                                )
                            )
                        },
                        {
                            Text(text = checkBoxText2)
                        },
                        verticalDivider = false,
                        center = true
                    )
                    OutlinedTextField(
                        value = textFieldValue,
                        onValueChange = {
                            onValueChange(it)
                        },
                        placeholder = { Text(text = textFieldHint) },
                        modifier = Modifier.padding(top = 8.dp)
                    )
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun DoubleTextFieldSingleCheckBoxTwoButtonDialog(
            title: String = "Dialog Title",
            textField1KeyboardType: KeyboardType = KeyboardType.Text,
            textField2KeyboardType: KeyboardType = KeyboardType.Text,
            textFieldValue1Hint: String = "Enter Old Password/Pin",
            textFieldValue2Hint: String = "Enter New Password",
            textFieldValue1: String = "Dialog TextField 1",
            textFieldValue2: String = "Dialog TextField 2",
            onValueChange1: (String) -> Unit = {},
            onValueChange2: (String) -> Unit = {},
            checkBoxValue: Boolean = false,
            onCheckBoxChange: (Boolean) -> Unit = {},
            checkBoxText: String = "Dialog CheckBox",
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {},
            isPinField: Boolean = false
        ) {
            MyAlertDialog(
                title = { Text(text = title, style = getEloNormalStdPrimaryText()) },
                content = {
                    Row(
                        modifier = Modifier
                            .border(
                                BorderStroke(
                                    width = 2.dp,
                                    color = MaterialTheme.colorScheme.primary
                                ), shape = RoundedCornerShape(3.dp)
                            )
                            .wrapContentHeight(align = Alignment.Top)
                            .fillMaxWidth()
                            .padding(
                                start = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp),
                                end = dimensionResource(id = com.intuit.sdp.R.dimen._5sdp)
                            )
                            .height(IntrinsicSize.Min),
                        verticalAlignment = Alignment.CenterVertically,
                    ) {
                        Checkbox(
                            checkBoxValue,
                            onCheckBoxChange,
                            modifier = Modifier.padding(
                                top = 8.dp,
                                bottom = 8.dp
                            )
                        )


                        Text(text = checkBoxText, style = getEloSmallBoldSecondaryText())

                    }
                    Spacer(
                        modifier = Modifier
                            .fillMaxWidth()
                            .height(dimensionResource(id = com.intuit.sdp.R.dimen._6sdp))
                    )

                    val keyboardController = LocalSoftwareKeyboardController.current
                    OutlinedTextField(
                        keyboardOptions = KeyboardOptions(
                            keyboardType = textField1KeyboardType,
                            imeAction = ImeAction.Done
                        ),
                        keyboardActions = KeyboardActions(
                            onDone = {
                                keyboardController?.hide()
                            }),
                        visualTransformation = PasswordVisualTransformation(),
                        placeholder = {
                            Text(textFieldValue1Hint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue1,
                        onValueChange = {
                            onValueChange1(it)
                        })
                    Spacer(
                        modifier = Modifier
                            .fillMaxWidth()
                            .height(dimensionResource(id = com.intuit.sdp.R.dimen._6sdp))
                    )
                    OutlinedTextField(
                        keyboardOptions = KeyboardOptions(
                            keyboardType = textField2KeyboardType,
                            imeAction = ImeAction.Done
                        ),
                        keyboardActions = KeyboardActions(
                            onDone = {
                                keyboardController?.hide()
                            }),
                        visualTransformation = PasswordVisualTransformation(),
                        placeholder = {
                            Text(textFieldValue2Hint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue2,
                        onValueChange = { input ->
                            if (isPinField) {
                                onValueChange2(input.filter { it.isDigit() })
                            } else {
                                onValueChange2(input)
                            }
                        })
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun SingleDropDownTwoButtonDialog(
            title: String = "Dialog Title",
            dropDownValues: Array<String> = arrayOf("Value 1", "Value 2", "Value 3"),
            selectedValue: String = try {
                dropDownValues[0]
            } catch (e: Exception) {
                "Value 1"
            },
            onValueSelected: (item: String) -> Unit = {},
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {}
        ) {
            MyAlertDialog(
                title = { Text(text = title, style = getEloNormalStdPrimaryText()) },
                content = {
                    CommonExposedDropdownMenu(dropDownValues, onValueSelected, selectedValue)
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun SingleTextFieldSingleDropDownTwoButtonDialog(
            title: String = "Dialog Title",
            textFieldValue: String = "Dialog TextField",
            textFieldValueHint: String = "",
            onValueChange: (String) -> Unit = {},
            dropDownValues: Array<String> = arrayOf("Value 1", "Value 2", "Value 3"),
            selectedValue: String = try {
                dropDownValues[0]
            } catch (e: Exception) {
                "Value 1"
            },
            onValueSelected: (item: String) -> Unit = {},
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {}
        ) {
            MyAlertDialog(
                title = { Text(text = title, style = getEloNormalStdPrimaryText()) },
                content = {
                    OutlinedTextField(
                        placeholder = {
                            Text(textFieldValueHint, style = getEloSmallSecondaryText())
                        },
                        textStyle = getEloSmallBoldSecondaryText(),
                        value = textFieldValue,
                        onValueChange = {
                            onValueChange(it)
                        })
                    CommonExposedDropdownMenu(
                        dropDownValues,
                        onValueSelected,
                        selectedValue
                    )
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun MultiSelectDropDownTwoButtonDialog(
            title: String = "Dialog Title",
            dropDownValues: Array<String> = arrayOf("Value 1", "Value 2", "Value 3"),
            selectedValues: Array<String> = arrayOf("Value 1"),
            onOptionAdded: (value: String) -> Unit = {},
            onOptionRemoved: (value: String) -> Unit = {},
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {}
        ) {
            MyAlertDialog(
                title = { Text(text = title) },
                content = {
                    MultiComboBox(
                        labelText = "Select Values",
                        options = dropDownValues,
                        onOptionAdded = { onOptionAdded(it) },
                        onOptionRemoved = { onOptionRemoved(it) },
                        selectedIds = selectedValues
                    )
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun SingleDropDownSingleCheckBoxTwoButtonDialog(
            title: String = "Dialog Title",
            dropDownValues: Array<String> = arrayOf("Value 1", "Value 2", "Value 3"),
            selectedValue: String = try {
                dropDownValues[0]
            } catch (e: Exception) {
                "Value 1"
            },
            onValueSelected: (item: String) -> Unit = {},
            checkBoxValue: Boolean = false,
            onCheckBoxChange: (Boolean) -> Unit = {},
            checkBoxText: String = "Dialog CheckBox",
            onConfirmClick: () -> Unit = {},
            onCancelClick: () -> Unit = {},
            checkBoxEnabled: Boolean = true
        ) {
            MyAlertDialog(
                title = { Text(text = title) },
                content = {
                    DoubleRow(
                        {
                            Checkbox(
                                checkBoxValue,
                                onCheckBoxChange,
                                enabled = checkBoxEnabled,
                                colors = CheckboxDefaults.colors(
                                    disabledUncheckedColor = Color.Gray,
                                    disabledCheckedColor = Color.Gray
                                ),
                                modifier = Modifier.padding(
                                    start = 10.dp,
                                    top = 10.dp,
                                    bottom = 10.dp
                                )
                            )
                        },
                        {
                            Text(
                                text = checkBoxText,
                                color = Color.Gray
                            )
                        },
                        verticalDivider = false,
                        center = true
                    )
                    CommonExposedDropdownMenu(dropDownValues, onValueSelected, selectedValue)
                },
                confirmButton =
                    {
                        Button(
                            onClick = {
                                onConfirmClick()
                            }) {
                            Text("Confirm", style = getEloBoldStdText())
                        }
                    },
                dismissButton =
                    {
                        Button(
                            onClick = {
                                onCancelClick()
                            }) {
                            Text("Cancel", style = getEloBoldStdText())
                        }
                    },
                onDismiss = {}
            )
        }

        @Composable
        fun MyAlertDialog(
            title: @Composable () -> Unit = {
                Text(
                    text = "Title",
                    style = getEloNormalStdPrimaryText()
                )
            },
            content: @Composable () -> Unit = {
                OutlinedTextField(
                    value = "Content",
                    onValueChange = {},
                    textStyle = getEloNormalStdSecondaryText()
                )
            },
            dismissButton: @Composable () -> Unit = {
                Button(
                    onClick = {}) {
                    Text("Cancel", style = getEloBoldStdText())
                }
            },
            confirmButton: @Composable () -> Unit = {
                Button(
                    onClick = {}) {
                    Text("Confirm", style = getEloBoldStdText())
                }
            },
            onDismiss: () -> Unit = {},
        ) {
            Dialog(onDismiss) {
                Surface(shape = MaterialTheme.shapes.medium) {
                    Column {
                        Column(Modifier.padding(24.dp)) {
                            title.invoke()
                            Spacer(Modifier.size(16.dp))
                            content.invoke()
                        }
                        Spacer(Modifier.size(4.dp))
                        Row(
                            Modifier
                                .padding(8.dp)
                                .fillMaxWidth(),
                            Arrangement.spacedBy(8.dp, Alignment.End),
                        ) {
                            dismissButton.invoke()
                            confirmButton.invoke()
                        }
                    }
                }
            }
        }

        @OptIn(ExperimentalMaterial3Api::class)
        @Composable
        fun CommonTimePickerDialog(
            timePickerState: TimePickerState = rememberTimePickerState(),
            onConfirm: (TimePickerState) -> Unit = {},
            onDismiss: () -> Unit = {},
            modifier: Modifier = Modifier
        ) {
            val configuration = LocalConfiguration.current
            val orientation = configuration.orientation
            val layoutType = if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
                TimePickerLayoutType.Horizontal
            } else {
                TimePickerLayoutType.Vertical
            }
            val orientationKey = remember(orientation) { orientation }
            TimePickerDialog(
                onDismiss = onDismiss,
                onConfirm = { onConfirm(timePickerState) },
                orientation = orientation
            ) {
                key(orientationKey) {
                    TimePicker(
                        state = timePickerState,
                        layoutType = layoutType,
                        modifier = modifier
                            .fillMaxWidth()
                            .fillMaxHeight()
                    )
                }
            }
        }

        @Composable
        fun TimePickerDialog(
            onDismiss: () -> Unit = {},
            onConfirm: () -> Unit = {},
            orientation: Int,
            content: @Composable () -> Unit = {}
        ) {
            Dialog(
                onDismissRequest = onDismiss,
                properties = DialogProperties(usePlatformDefaultWidth = false)
            ) {
                val dialogWidth =
                    if (orientation == Configuration.ORIENTATION_LANDSCAPE) 0.7f else 0.70f
                val dialogHeight =
                    if (orientation == Configuration.ORIENTATION_LANDSCAPE) 0.95f else 0.65f
                Surface(
                    shape = RoundedCornerShape(16.dp),
                    tonalElevation = 8.dp,
                    modifier = Modifier
                        .fillMaxWidth(dialogWidth)
                        .fillMaxHeight(dialogHeight)
                ) {
                    Column(
                        modifier = Modifier
                            .padding(16.dp)
                            .fillMaxSize()
                    ) {
                        Box(
                            modifier = Modifier
                                .weight(1f)
                                .fillMaxWidth()
                        ) {
                            content()
                        }
                        Spacer(modifier = Modifier.height(16.dp))
                        Row(
                            modifier = Modifier.fillMaxWidth(),
                            horizontalArrangement = Arrangement.End
                        ) {
                            TextButton(onClick = onDismiss) {
                                Text("Dismiss")
                            }
                            Spacer(modifier = Modifier.width(8.dp))
                            TextButton(onClick = onConfirm) {
                                Text("OK")
                            }
                        }
                    }
                }
            }
        }

        @OptIn(ExperimentalMaterial3Api::class)
        @Composable
        fun CommonDatePickerDialog(
            datePickerState: DatePickerState = rememberDatePickerState(),
            onConfirm: (DatePickerState) -> Unit = {},
            onDismiss: () -> Unit = {}
        ) {
            val configuration = LocalConfiguration.current
            val screenHeight = configuration.screenHeightDp.dp
            val screenWidth = configuration.screenWidthDp.dp
            val isLandscape = configuration.orientation == Configuration.ORIENTATION_LANDSCAPE
            val scrollState = rememberScrollState()
            DatePickerDialog(
                onDismissRequest = { onDismiss() },
                confirmButton = {},
                dismissButton = {},
                modifier = Modifier
                    .then(
                        if (isLandscape) {
                            Modifier
                                .widthIn(max = screenWidth)
                                .heightIn(max = screenHeight * 0.95f)
                        } else {
                            Modifier.wrapContentSize()
                        }
                    )
            ) {
                Box(
                    modifier = Modifier
                        .fillMaxWidth()
                        .heightIn(min = 100.dp, max = screenHeight)
                ) {
                    Row(modifier = Modifier.fillMaxSize()) {
                        Column(
                            modifier = Modifier
                                .weight(1f)
                                .verticalScroll(scrollState)
                                .padding(end = 5.dp)
                        ) {
                            DatePicker(
                                state = datePickerState,
                                showModeToggle = false,
                                modifier = Modifier.fillMaxWidth()
                            )

                            Spacer(modifier = Modifier.height(16.dp))
                        }
                        VerticalScrollbar(
                            scrollState = scrollState,
                            modifier = Modifier
                                .fillMaxHeight()
                                .width(4.dp)
                        )
                    }
                    Surface(
                        color = MaterialTheme.colorScheme.surface,
                        tonalElevation = 4.dp,
                        modifier = Modifier
                            .fillMaxWidth()
                            .align(Alignment.BottomEnd)
                    ) {
                        Row(
                            modifier = Modifier.padding(5.dp),
                            horizontalArrangement = Arrangement.End
                        ) {
                            TextButton(onClick = onDismiss) {
                                Text("Cancel")
                            }
                            TextButton(onClick = { onConfirm(datePickerState) }) {
                                Text("OK")
                            }
                        }
                    }
                }
            }
        }

        @Composable
        fun VerticalScrollbar(
            scrollState: ScrollState,
            modifier: Modifier = Modifier,
            thumbColor: Color = Color.Gray,
            trackColor: Color = Color.LightGray.copy(alpha = 0.3f)
        ) {
            val density = LocalDensity.current
            val thumbHeight = with(density) {
                (100 * (scrollState.maxValue.toFloat() / (scrollState.maxValue + scrollState.maxValue).coerceAtLeast(
                    1
                ))).dp
            }
            Box(
                modifier = modifier
                    .drawBehind {
                        drawRoundRect(
                            color = trackColor,
                            cornerRadius = CornerRadius(x = size.width / 2, y = size.width / 2)
                        )
                        val proportion =
                            scrollState.value.toFloat() / scrollState.maxValue.coerceAtLeast(1)
                        val y = proportion * (size.height - thumbHeight.toPx())
                        drawRoundRect(
                            color = thumbColor,
                            topLeft = Offset(x = 0f, y = y),
                            size = Size(width = size.width, height = thumbHeight.toPx()),
                            cornerRadius = CornerRadius(x = size.width / 2, y = size.width / 2)
                        )
                    }
            )
        }

        @Composable
        fun SimpleDialogWithAText(
            message: String = "message",
            onDismiss: () -> Unit = {}
        ) {
            AlertDialog(
                onDismissRequest = onDismiss,
                title = {
                    Box(
                        modifier = Modifier
                            .fillMaxWidth(),
                        contentAlignment = Alignment.Center
                    ) {
                        Text(
                            text = message,
                            style = getEloNormalStdSecondaryText(),
                            textAlign = TextAlign.Center
                        )
                    }
                },
                confirmButton = {
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        horizontalArrangement = Arrangement.Center
                    ) {
                        Button(onClick = onDismiss) {
                            Text("OK")
                        }
                    }
                }
            )
        }
    }
}