/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp.common

import android.app.Activity
import android.content.Intent
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.layout.wrapContentWidth
import androidx.compose.foundation.shape.CutCornerShape
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.Checkbox
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.RadioButton
import androidx.compose.material3.RadioButtonDefaults
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.VerticalDivider
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberUpdatedState
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.eloview.sdk.testapp.AccountManagerActivity
import com.eloview.sdk.testapp.GlobalClass
import com.eloview.sdk.testapp.MainActivity
import com.eloview.sdk.testapp.ui.theme.getEloBoldStdSecondaryText
import com.eloview.sdk.testapp.ui.theme.getEloBoldStdSecondaryTextstart
import com.eloview.sdk.testapp.ui.theme.getEloBoldStdText
import com.eloview.sdk.testapp.ui.theme.getEloLargeBoldTextPrimary
import com.eloview.sdk.testapp.ui.theme.getEloNormalRadioLabel
import com.eloview.sdk.testapp.ui.theme.getEloNormalStdSecondaryText
import com.eloview.sdk.testapp.ui.theme.getEloNormalStdText
import com.eloview.sdk.testapp.ui.theme.getEloOutputText
import com.eloview.sdk.testapp.ui.theme.getEloSmallSecondaryText
import com.eloview.sdk.testapp.ui.theme.getEloSmallWhiteText

class CommonComposables {
    @Composable
    @Preview(showBackground = true)
    fun ButtonRowPreview() {
        ButtonRow()
    }

    @Composable
    @Preview(showBackground = true)
    fun DoubleRowPreview() {
        DoubleRow(center = true, verticalDivider = false)
    }

    @Composable
    @Preview(showBackground = true)
    fun CommonButtonPreview() {
        CommonButton()
    }

    @Composable
    @Preview(showBackground = true)
    fun CommonExposedDropdownMenuPreview() {
        CommonExposedDropdownMenu()
    }

    @Composable
    @Preview(showBackground = true)
    fun DropDownDoublePreview() {
        DropDownDouble()
    }

    @Composable
    @Preview(showBackground = true)
    fun MultiComboBoxPreview() {
        MultiComboBox()
    }

    @Composable
    @Preview(showBackground = true)
    fun SwitchWithLabelPreview() {
        SwitchWithLabel()
    }

    @Composable
    @Preview(showBackground = true)
    fun EloWideButtonPreview() {
        EloWideButton()
    }

    @Composable
    @Preview(showBackground = true, widthDp = 640)
    fun BottomButtonBarPreview() {
        BottomButtonBar()
    }


    @Composable
    @Preview(showBackground = true, widthDp = 640)
    fun EloOutputTextPreview() {
        EloOutputText()
    }


    @Composable
    @Preview(showBackground = true)
    fun AccountManagerButtonView() {
        AccountManagerBtn()
    }

    @Composable
    @Preview(showBackground = true)
    fun ButtonRowNewView() {
        RowTextWithRadio()
    }

    companion object {
        @Composable
        fun ButtonRow(
            text1: String = "Button 1",
            text2: String = "Button 2",
            onClick1: () -> Unit = {},
            onClick2: () -> Unit = {},
            enabled: Boolean = true,
            verticalDivider: Boolean = true,
            center: Boolean = false

        ) {
            DoubleRow(
                { CommonButton(text = text1, onClick = onClick1, enabled = enabled) },
                { CommonButton(text = text2, onClick = onClick2, enabled = enabled) },
                verticalDivider = verticalDivider,
                center = center
            )
        }

        @Composable
        fun RowTextWithRadio(
            text1: String = "Text 1",
            onClickEnable: () -> Unit = {},
            onClickDisable: () -> Unit = {},
        ) {
            val selectedOption = remember { mutableStateOf("") }
            RowRadioButtonWithLabel(
                { Text(text = text1,style = getEloNormalRadioLabel()) },
                {  val options = listOf("Enable", "Disable")
                    Row (modifier = Modifier.wrapContentWidth()) {
                        options.forEach { option ->
                            Row {
                                RadioButton(
                                    onClick = { selectedOption.value = option
                                        when (option) {
                                            "Enable" -> onClickEnable()
                                            "Disable" -> onClickDisable()
                                        }
                                    },
                                    selected = selectedOption.value == option,
                                    colors = RadioButtonDefaults.colors(
                                        selectedColor = Color.White,
                                        unselectedColor = Color.White
                                    )
                                )
                                Text(option, modifier = Modifier.align(Alignment.CenterVertically), style = getEloSmallWhiteText())
                            }
                        }
                    } },
                verticalDivider = false,
                center = false
            )
        }

        @Composable
        fun RowRadioButtonWithLabel(
            composable1: @Composable () -> Unit = { Text(text = "Text 1") },
            composable2: @Composable () -> Unit = { Text(text = "Text 2") },
            verticalDivider: Boolean = true,
            center: Boolean = false
        ) {
            Row(
                modifier = Modifier
                    .border(BorderStroke(width = 3.dp, color = MaterialTheme.colorScheme.primary))
                    .wrapContentHeight(align = Alignment.Top)
                    .fillMaxWidth()
                    .background(MaterialTheme.colorScheme.primary)
                    .padding(
                        start = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp),
                        end = dimensionResource(id = com.intuit.sdp.R.dimen._5sdp)
                    )
                    .height(IntrinsicSize.Min),
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = if (center) {
                    Arrangement.Center
                } else {
                    Arrangement.SpaceAround
                },
            ) {
                Column(Modifier.weight(1.5f)) { // Wrap composable1 in a Column to apply weight directly
                    composable1()
                }
                if (verticalDivider) {
                    VerticalDivider(modifier = Modifier.fillMaxHeight())
                }
                composable2()
            }

            Spacer(modifier = Modifier
                .fillMaxWidth()
                .height(dimensionResource(id = com.intuit.sdp.R.dimen._6sdp)))
        }




        @Composable
        fun RowTextAndSwitchContent(
            mainLabel:String="Text 1",
            label: String = "Switch Lable",
            state: Boolean = false,
            onStateChange: (Boolean) -> Unit = {},
            verticalDivider: Boolean = true,
            center: Boolean = false
        ) {
            Row(
                modifier = Modifier
                    .border(BorderStroke(width = 3.dp, color = MaterialTheme.colorScheme.primary))
                    .wrapContentHeight(align = Alignment.Top)
                    .fillMaxWidth()
                    .padding(
                        start = dimensionResource(id = com.intuit.sdp.R.dimen._5sdp),
                        end = dimensionResource(id = com.intuit.sdp.R.dimen._5sdp)
                    )
                    .height(IntrinsicSize.Min),
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = if (center) {
                    Arrangement.Center
                } else {
                    Arrangement.SpaceAround
                },
            ) {
                Box(modifier = Modifier.weight(1f).padding(start = dimensionResource(id=com.intuit.sdp.R.dimen._5sdp).value.dp),
                    contentAlignment=Alignment.CenterStart) {
                    Text(text = mainLabel, style = getEloBoldStdSecondaryText())
                }

                if (verticalDivider) {
                    VerticalDivider(modifier = Modifier
                        .fillMaxHeight()
                        .width(1.dp))
                }
                Box(
                    contentAlignment=Alignment.Center) {
                    SwitchWithLabel(label,state, onStateChange)
                }
            }
            Spacer(modifier = Modifier
                .fillMaxWidth()
                .height(dimensionResource(id = com.intuit.sdp.R.dimen._6sdp)))
        }


        @Composable
        fun DoubleRow(
            composable1: @Composable () -> Unit = { Text(text = "Text 1") },
            composable2: @Composable () -> Unit = { Text(text = "Text 2") },
            verticalDivider: Boolean = true,
            center: Boolean = false
        ) {
            Row(
                modifier = Modifier
                    .border(BorderStroke(width = 3.dp, color = MaterialTheme.colorScheme.primary))
                    .wrapContentHeight(align = Alignment.Top)
                    .fillMaxWidth()
                    .padding(
                        start = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp),
                        end = dimensionResource(id = com.intuit.sdp.R.dimen._5sdp)
                    )
                    .height(IntrinsicSize.Min),
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = if (center) {
                    Arrangement.Center
                } else {
                    Arrangement.SpaceAround
                },
            ) {

                Box(modifier = Modifier.weight(1f),
                    contentAlignment=Alignment.Center) {
                    composable1()
                }

                if (verticalDivider) {
                    VerticalDivider(modifier = Modifier
                        .fillMaxHeight()
                        .width(1.dp))
                }

                Box(modifier = Modifier.weight(1f),
                    contentAlignment=Alignment.Center) {
                    composable2()
                }
            }

            Spacer(modifier = Modifier
                .fillMaxWidth()
                .height(dimensionResource(id = com.intuit.sdp.R.dimen._6sdp)))
        }


        @Composable
        fun EloWideButton(text: String = "Button Text", onClick: () -> Unit = {}) {
            Button(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(
                        bottom = dimensionResource(com.intuit.sdp.R.dimen._3sdp).value.dp,
                        top = dimensionResource(com.intuit.sdp.R.dimen._2sdp).value.dp
                    ),
                shape = CutCornerShape(
                    topStart = 0f,
                    topEnd = 0f,
                    bottomEnd = 0f,
                    bottomStart = 0f
                ),
                onClick = onClick
            ) {
                Text(text = text, style = getEloNormalStdText(), modifier = Modifier.fillMaxWidth())
            }
        }

        @Composable
        fun EloScreenTitleText(text: String = "Screen  Title"){
            Text(
                text = text,
                style = getEloLargeBoldTextPrimary(),
                modifier = Modifier.padding(bottom = dimensionResource(id=com.intuit.sdp.R.dimen._5sdp))
            )
        }


        @Composable
        fun EloOutputText(text: String = "Button Text") {

            Spacer(modifier = Modifier
                .fillMaxWidth()
                .padding(bottom = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp)))


            AnimatedVisibility(
                visible = text.isNotEmpty(),
                enter = fadeIn(),
                exit = fadeOut()
            ) {
                Row(
                    modifier = Modifier
                        .border(
                            BorderStroke(
                                width = 3.dp,
                                color = MaterialTheme.colorScheme.primary
                            )
                        )
                        .padding(12.dp)
                ) {
                    Text(
                        text = text,
                        style = getEloOutputText(),
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(start = 10.dp, top = 10.dp, bottom = 10.dp)

                    )
                }
            }

            Spacer(modifier = Modifier
                .fillMaxWidth()
                .padding(bottom = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp)))

        }


        @Composable
        fun CommonButton(text: String = "Button Text", onClick: () -> Unit = {}) {
            Button(
                shape = CutCornerShape(
                    topStart = 0f,
                    topEnd = 0f,
                    bottomEnd = 0f,
                    bottomStart = 0f
                ),
                onClick = onClick
            ) {
                Text(text = text, fontSize = 16.sp)
            }
        }

        @Composable
        fun CommonButton( text: String = "Button Text", onClick: () -> Unit = {},
                          enabled: Boolean = true) {
            Button(
                enabled = enabled,
                contentPadding= PaddingValues(horizontal = 24.dp, vertical = 0.dp),
                shape = CutCornerShape(
                    topStart = 0f,
                    topEnd = 0f,
                    bottomEnd = 0f,
                    bottomStart = 0f
                ),
                onClick = onClick
            ) {
                Text(text = text, fontSize = 16.sp)
            }
        }

        @Composable
        fun APIButton(
            modifier: Modifier = Modifier,
            text: String = "Button Text",
            onClick: () -> Unit = {},
        ) {
            Button(
                modifier = modifier,
                shape = CutCornerShape(
                    topStart = 0f,
                    topEnd = 0f,
                    bottomEnd = 0f,
                    bottomStart = 0f
                ),
                onClick = onClick
            ) {
                Text(text = text, style = getEloBoldStdText())
            }
        }


        @OptIn(ExperimentalMaterial3Api::class)
        @Composable
        fun CommonExposedDropdownMenu(
            values: Array<String> = arrayOf("Value 1", "Value 2", "Value 3"),
            onValueSelected: (item: String) -> Unit = {},
            selectedText: String = "Value 1"
        ) {
            var expanded by remember {
                mutableStateOf(false)
            }
            Box(
                modifier = Modifier
                    .wrapContentWidth()
                    .padding(10.dp)
            ) {
                ExposedDropdownMenuBox(expanded = expanded, onExpandedChange = {
                    expanded = !expanded
                }) {
                    TextField(
                        value = selectedText,
                        onValueChange = {},
                        readOnly = true,
                        trailingIcon = {
                            ExposedDropdownMenuDefaults.TrailingIcon(
                                expanded = expanded
                            )
                        },
                        modifier = Modifier.menuAnchor(),
                        textStyle = getEloSmallSecondaryText()
                    )

                    ExposedDropdownMenu(
                        expanded = expanded,
                        onDismissRequest = { expanded = false }
                    ) {
                        values.forEach { item ->
                            DropdownMenuItem(text = { Text(text = item, style = getEloSmallSecondaryText()) }, onClick = {
                                onValueSelected(item)
                                expanded = false
                            })
                        }
                    }
                }
            }
        }

        @Composable
        fun DropDownDouble(
            buttonText: String = "Button Text",
            onClick: () -> Unit = {},
            values: Array<String> = arrayOf("Value 1", "Value 2", "Value 3"),
            selectedValue: String = "Value 1",
            onValueSelected: (item: String) -> Unit = { }
        ) {
            Box(contentAlignment = Alignment.Center) {
                DoubleRow({ CommonButton(text = buttonText, onClick = onClick) }, {
                    CommonExposedDropdownMenu(values, onValueSelected, selectedValue)
                }, verticalDivider = false, center = true)
            }
        }

        @OptIn(ExperimentalMaterial3Api::class)
        @Composable
        fun MultiComboBox(
            modifier: Modifier = Modifier,
            labelText: String = "Select options",
            options: Array<String> = arrayOf("Option 1", "Option 2", "Option 3"),
            onOptionAdded: (String) -> Unit = {},
            onOptionRemoved: (String) -> Unit = {},
            selectedIds: Array<String> = emptyArray<String>(),
        ) {
            var expanded by remember { mutableStateOf(false) }
            // when no options available, I want ComboBox to be disabled
            val isEnabled by rememberUpdatedState { options.isNotEmpty() }
            val selectedOptionsList = remember { mutableStateListOf<String>() }

            //Initial setup of selected ids
            LaunchedEffect(selectedIds) {
                selectedOptionsList.clear()
                selectedOptionsList.addAll(selectedIds)
            }
            ExposedDropdownMenuBox(
                expanded = expanded,
                onExpandedChange = {
                    if (isEnabled()) {
                        expanded = !expanded
                    }
                },
                modifier = modifier,
            ) {
                val selectedSummary = when (selectedOptionsList.size) {
                    0 -> ""
                    1 -> options.firstOrNull { it == selectedOptionsList.first() } ?: ""
                    else -> "${selectedOptionsList.size} selected"
                }
                TextField(
                    enabled = isEnabled(),
                    modifier = Modifier.menuAnchor(),
                    readOnly = true,
                    value = selectedSummary,
                    onValueChange = {},
                    label = { Text(text = labelText) },
                    trailingIcon = {
                        ExposedDropdownMenuDefaults.TrailingIcon(expanded = expanded)
                    },
                    colors = ExposedDropdownMenuDefaults.textFieldColors(),
                )
                ExposedDropdownMenu(
                    expanded = expanded,
                    onDismissRequest = {
                        expanded = false
                    },
                ) {
                    for (option in options) {

                        //use derivedStateOf to evaluate if it is checked
                        val checked by remember {
                            derivedStateOf { option in selectedOptionsList }
                        }

                        DropdownMenuItem(
                            text = {
                                Row(verticalAlignment = Alignment.CenterVertically) {
                                    Checkbox(
                                        checked = checked,
                                        onCheckedChange = { newCheckedState ->
                                            if (newCheckedState) {
                                                onOptionAdded(option)
                                                selectedOptionsList.add(option) // Update local state
                                            } else {
                                                onOptionRemoved(option)
                                                selectedOptionsList.remove(option) // Update local state
                                            }
                                        },
                                    )
                                    Text(text = option)
                                }
                            },
                            onClick = {
                                if (checked) {
                                    onOptionRemoved(option)
                                    selectedOptionsList.remove(option)
                                } else {
                                    onOptionAdded(option)
                                    selectedOptionsList.add(option)
                                }
                            },
                            contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding,
                        )
                    }
                }
            }
        }

        @Composable
        fun SwitchWithLabel(
            label: String = "Switch Label",
            state: Boolean = false,
            onStateChange: (Boolean) -> Unit = {}
        ) {
            val interactionSource = remember { MutableInteractionSource() }
            Row(
                modifier = Modifier
                    .clickable(
                        interactionSource = interactionSource,
                        // This is for removing ripple when Row is clicked
                        indication = null,
                        role = Role.Switch,
                        onClick = {
                            onStateChange(!state)
                        }
                    )
                    .padding(start = dimensionResource(id=com.intuit.sdp.R.dimen._2sdp).value.dp
                    , end = dimensionResource(id=com.intuit.sdp.R.dimen._2sdp).value.dp),
                verticalAlignment = Alignment.CenterVertically

            ) {

                Text(text = label, style = getEloBoldStdSecondaryTextstart())
                Spacer(modifier = Modifier.padding(start = dimensionResource(id=com.intuit.sdp.R.dimen._5sdp).value.dp))
                Switch(
                    checked = state,
                    onCheckedChange = {
                        onStateChange(it)
                    }
                )
            }
        }

        @Composable
        fun BottomButtonBar(activity: Activity = Activity(), clearOutput: () -> Unit = {}) {

            Column(
                modifier = Modifier.fillMaxWidth()// Optional: Add vertical spacing
            ) {

                APIButton(
                    modifier = Modifier.fillMaxWidth(), // Make the button take full width below
                    text = "CLEAR OUTPUT",
                    onClick = clearOutput
                )

                Spacer(Modifier.padding(dimensionResource(id=com.intuit.sdp.R.dimen._2sdp)))

                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.spacedBy(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp))
                ) {
                    APIButton(
                        modifier = Modifier
                            .weight(1f), // Consider using weight for even distribution
                        text = "EXIT APP",
                        onClick = {
                            activity.finishAffinity()
                        }
                    )

                    APIButton(
                        modifier = Modifier
                            .weight(1f), // Consider using weight for even distribution
                        text = "MAIN MENU",
                        onClick = {
                            val intent = Intent(activity, MainActivity::class.java)
                            intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                            activity.startActivity(intent)
                        }
                    )
                }
            }





        }


        @Composable
        fun AccountManagerBtn(
            activity: Activity
        ) {
            val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()
            Box(
                modifier = Modifier
                    .background(color = Color.Transparent)
                    .padding(4.dp)
            )
            {
                Button(
                    shape = CutCornerShape(
                        topStart = 0f,
                        topEnd = 0f,
                        bottomEnd = 0f,
                        bottomStart = 0f
                    ),
                    modifier = if (globalDetails.authorized) {
                        Modifier
                            .background(color = Color.Green)
                            .fillMaxWidth()
                    } else {
                        Modifier
                            .background(color = Color.Red)
                            .fillMaxWidth()
                    },
                    colors = if (globalDetails.authorized) {
                        ButtonDefaults.buttonColors(containerColor = Color.Green)
                    } else {
                        ButtonDefaults.buttonColors(containerColor = Color.Red)
                    },
                    onClick = {
                        val intent = Intent(activity, AccountManagerActivity::class.java)
                        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                        activity.startActivity(intent)
                    }) {
                    val buttonText = when {
                        globalDetails.emailPass -> "Account Manager \n (verified token using email id and password)"
                        globalDetails.clientSecret -> "Account Manager \n (verified token using client id and secret key)"
                        globalDetails.tokenFromApi -> "Account Manager \n (verified token using Auth API)"
                        else -> "ACCOUNT MANAGER"
                    }
                    Text(text = buttonText, style = getEloBoldStdText(), modifier = Modifier.fillMaxWidth())
                }
            }
        }
    }


    @Composable
    fun AccountManagerBtn() {
        val globalDetails by GlobalClass.globalViewModel.globalState.collectAsState()
        Box(
            modifier = Modifier
                .wrapContentHeight()
                .background(color = Color.Transparent)
        )
        {
            Button(
                shape = CutCornerShape(
                    topStart = 0f,
                    topEnd = 0f,
                    bottomEnd = 0f,
                    bottomStart = 0f
                ),
                modifier = if (globalDetails.authorized) {
                    Modifier
                        .background(color = Color.Green)
                        .fillMaxWidth()
                        .wrapContentHeight()
                } else {
                    Modifier
                        .background(color = Color.Red)
                        .fillMaxWidth()
                        .wrapContentHeight()
                },
                colors = if (globalDetails.authorized) {
                    ButtonDefaults.buttonColors(containerColor = Color.Green)
                } else {
                    ButtonDefaults.buttonColors(containerColor = Color.Red)
                },
                onClick = {
                    /* val intent = Intent(activity, AccountManagerActivity::class.java)
                     intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                     activity.startActivity(intent)*/
                }) {
                val buttonText = when {
                    globalDetails.emailPass -> "Account Manager \n (verified token using email id and password)"
                    globalDetails.clientSecret -> "Account Manager \n (verified token using client id and secret key)"
                    globalDetails.tokenFromApi -> "Account Manager \n (verified token using Auth API)"
                    else -> "ACCOUNT MANAGER"
                }
                Text(text = buttonText, style = getEloBoldStdText(),modifier = Modifier.fillMaxWidth())
            }
        }

    }
}