/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.OnBackPressedCallback
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.eloview.sdk.infoManager.Info
import com.eloview.sdk.rest.ExecutorServiceManager
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloWideButton
import com.eloview.sdk.testapp.eloActivity.EloActivity
import com.eloview.sdk.testapp.infoActivity.InfoActivity
import com.eloview.sdk.testapp.memoryActivity.MemoryActivity
import com.eloview.sdk.testapp.networkActivity.NetworkActivity
import com.eloview.sdk.testapp.otaActivity.OtaActivity
import com.eloview.sdk.testapp.packageActivity.PackageActivity
import com.eloview.sdk.testapp.peripheralActivity.PeripheralActivity
import com.eloview.sdk.testapp.systemActivity.SystemActivity
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.ui.theme.getEloLargeBoldTextPrimary

import com.eloview.sdk.testapp.ui.theme.getEloSmallText

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    MainActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        activity = this
                    )
                }
            }
        }
        onBackPressedDispatcher.addCallback(this, object : OnBackPressedCallback(true) {
            override fun handleOnBackPressed() {
                Log.d("MainActivity", "handleOnBackPressed")
                exitOnBackPressed()
            }
        })
    }


    fun exitOnBackPressed() {
        this@MainActivity.finishAffinity()
    }

    override fun onDestroy() {
        super.onDestroy()
        Log.d("MainActivity", "onDestroy:")

        // Shutdown the ExecutorService when the activity is destroyed
        ExecutorServiceManager.shutdown()
    }
}

@Preview(showBackground = true)
@Composable
fun MainActivityScreen(
    modifier: Modifier = Modifier,
    activity: Activity = Activity()
) {
    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
            modifier = Modifier
                .fillMaxSize()
                .wrapContentHeight()
        ) {
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.End
            ) {
                Text(
                    modifier = Modifier.padding(dimensionResource(id = com.intuit.sdp.R.dimen._2sdp).value.dp),
                    text = Info().sdkVersion,
                    textAlign = TextAlign.End,
                    style = getEloSmallText()
                )
            }

            Spacer(modifier = Modifier.height(10.dp))

            Text(
                text = stringResource(R.string.str_app_lable),
                style = getEloLargeBoldTextPrimary(),
                modifier = Modifier.padding(bottom = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp)
            )


            Text(
                text = stringResource(R.string.str_main_menu),
                style = getEloLargeBoldTextPrimary(),
                modifier = Modifier.padding(bottom = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp)
            )

            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier
                    .padding(top=dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp,
                        start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                        end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {

                EloWideButton(
                    text = stringResource(R.string.str_ota),
                    onClick = {
                        // Launch OTAActivity.kt
                        val intent = Intent(activity, OtaActivity::class.java)
                        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                        activity.startActivity(intent)
                    }
                )

                EloWideButton(
                    text = stringResource(R.string.str_package),
                    onClick = {
                        // Launch PackageActivity.kt
                        val intent = Intent(activity, PackageActivity::class.java)
                        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                        activity.startActivity(intent)
                    }
                )

                EloWideButton(
                    text = stringResource(R.string.str_network),
                    onClick = {
                        // Launch NetworkActivity.kt
                        val intent = Intent(activity, NetworkActivity::class.java)
                        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                        activity.startActivity(intent)
                    }
                )
                EloWideButton(
                    text = stringResource(R.string.str_system),
                    onClick = {
                        // Launch SystemActivity.kt
                        val intent = Intent(activity, SystemActivity::class.java)
                        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                        activity.startActivity(intent)
                    }
                )

                EloWideButton(
                    text = stringResource(R.string.str_memory),
                    onClick = {
                        // Launch MemoryActivity.kt
                        val intent = Intent(activity, MemoryActivity::class.java)
                        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                        activity.startActivity(intent)
                    }
                )

                EloWideButton(
                    text = stringResource(R.string.str_info),
                    onClick = {
                        // Launch InfoActivity.kt
                        val intent = Intent(activity, InfoActivity::class.java)
                        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                        activity.startActivity(intent)
                    }
                )

                EloWideButton(
                   text = stringResource(R.string.str_peripheral),
                    onClick = {
                        // Launch PeripheralActivity.kt
                        val intent = Intent(activity, PeripheralActivity::class.java)
                        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                        activity.startActivity(intent)
                    }
                )

                EloWideButton(
                    text = stringResource(R.string.str_eloview),
                    onClick = {
                        // Launch EloActivity.kt
                        val intent = Intent(activity, EloActivity::class.java)
                        intent.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
                        activity.startActivity(intent)
                    }
                )

                EloWideButton(
                    text = stringResource(R.string.str_exit_app),
                    onClick = { activity.moveTaskToBack(true) },

                )
            }
        }
    }
}