/*
 * Copyright (c) 2014-2025. Elo Touch Solution
 * License Terms
 *     - License does not expire
 *     - Source-code or binary products cannot be resold or distributed
 *     - Should be used only on Elo products
 *     - Cannot modify source-code for any purpose (cannot create derivative works)
 */

package com.eloview.sdk.testapp

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.layout.wrapContentWidth
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.CutCornerShape
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ArrowDropDown
import androidx.compose.material3.Button
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.window.Popup
import androidx.compose.ui.window.PopupProperties
import com.eloview.sdk.accountManager.AccountManager
import com.eloview.sdk.testapp.common.CommonComposables.Companion.BottomButtonBar
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloOutputText
import com.eloview.sdk.testapp.common.CommonComposables.Companion.EloScreenTitleText
import com.eloview.sdk.testapp.common.Util
import com.eloview.sdk.testapp.networkActivity.ESTCertificateActivity.Companion.TAG
import com.eloview.sdk.testapp.ui.theme.EloViewHomeSDK20TestAppTheme
import com.eloview.sdk.testapp.ui.theme.getEloBoldStdText
import com.eloview.sdk.testapp.ui.theme.getEloHighSizePrimaryBoldText
import com.eloview.sdk.testapp.ui.theme.getEloHighSizePrimaryText
import com.eloview.sdk.testapp.ui.theme.getEloHighSizeWhiteBoldText
import com.eloview.sdk.testapp.ui.theme.getEloNormalStdPrimaryText
import com.eloview.sdk.testapp.ui.theme.getEloNormalStdSecondaryText
import com.eloview.sdk.testapp.ui.theme.getEloNormalStdText
import com.eloview.sdk.testapp.viewmodel.AccountManagerViewModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

class AccountManagerActivity : ComponentActivity() {
    companion object {
        var isTokenVerified = false
    }

    private lateinit var accountManagerViewModel: AccountManagerViewModel

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        accountManagerViewModel = AccountManagerViewModel(
            applicationContext,
            false
        )
        enableEdgeToEdge()
        setContent {
            EloViewHomeSDK20TestAppTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    AccountManagerActivityScreen(
                        modifier = Modifier.padding(innerPadding),
                        accountManagerViewModel = accountManagerViewModel,
                        activity = this
                    )
                }
            }
        }
    }

    @Deprecated("This method has been deprecated in favor of using the Activity Result API\n      which brings increased type safety via an {@link ActivityResultContract} and the prebuilt\n      contracts for common intents available in\n      {@link androidx.activity.result.contract.ActivityResultContracts}, provides hooks for\n      testing, and allow receiving results in separate, testable classes independent from your\n      activity. Use\n      {@link #registerForActivityResult(ActivityResultContract, ActivityResultCallback)}\n      with the appropriate {@link ActivityResultContract} and handling the result in the\n      {@link ActivityResultCallback#onActivityResult(Object) callback}.")
    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        try {
            if (requestCode == AccountManager.REQUEST_AUTH_LOGIN) {
                if (resultCode == RESULT_OK) {
                    AccountManager.instance.onOAuthActivityResult(
                        applicationContext,
                        data,
                        accountManagerViewModel.accessTokenHandler
                    )
                } else {
                    Log.e(TAG, "Activity ResultCode not ok")
                }
            } else if (requestCode == AccountManager.REQUEST_AUTH_LOGOUT) {
                // for logout the auth manager
                if (resultCode == RESULT_OK) {
                    accountManagerViewModel.setAccessToken("")
                    accountManagerViewModel.setEmailPass(false)
                    accountManagerViewModel.setTokenFromApi(false)
                    accountManagerViewModel.setClientSecret(false)
                    accountManagerViewModel.setOutput("Logout Successful")
                } else {
                    Log.e(TAG, "Activity ResultCode not ok")
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }

    }
}

@Preview(showBackground = true)
@Composable
fun AccountManagerActivityScreen(
    modifier: Modifier = Modifier,
    accountManagerViewModel: AccountManagerViewModel = AccountManagerViewModel(
        applicationContext = Activity(),
        true
    ),
    activity: Activity = Activity()
) {
    val accountManagerDetails by accountManagerViewModel.deviceManagerState.collectAsState()
    val email = remember { mutableStateOf("") }
    val password = remember { mutableStateOf("") }
    val clientId = remember { mutableStateOf("") }
    val secretKey = remember { mutableStateOf("") }
    Box(modifier = modifier, contentAlignment = Alignment.Center) {

        Column(
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
            modifier = Modifier.padding(top= dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp)
                .fillMaxSize()
                .wrapContentHeight()
        ){

            EloScreenTitleText("Account Manager API")

            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier =  Modifier. padding(top= dimensionResource(id=com.intuit.sdp.R.dimen._10sdp).value.dp)
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            ) {
                // Retrieve the selectedIndex from the ViewModel
                val selectedIndex by accountManagerViewModel.selectedIndex

                // List of items for the dropdown
                val list = listOf(
                    "Production",
                    "QA",
                    "SFP",
                    "Polaris EDev",
                    "Polaris Staging",
                    "Polaris Production",
                    "Polaris QA"
                )

                // Load the correct properties file after app restart
                if (selectedIndex != 0) {
                    accountManagerViewModel.loadEnvironmentProperties(selectedIndex, list[selectedIndex], activity)
                }

                if (accountManagerDetails.spinnerEnabled) {
                    DropdownList(
                        itemList = list,
                        selectedIndex = selectedIndex,
                        modifier = Modifier.fillMaxWidth().padding( start = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp,
                            end = dimensionResource(id = com.intuit.sdp.R.dimen._16sdp).value.dp),
                        onItemClick = { index ->
                            // Handle item click
                            accountManagerViewModel.onSpinnerItemSelection(
                                index,
                                list,
                                activity
                            )
                        }
                    )
                }
                if (accountManagerDetails.textNoteEnabled) {
                    Text(
                        text = "Note: Please clear cache & data for browser and EloHome before changing any Polaris env.",
                        textAlign = TextAlign.Center,
                        fontSize = dimensionResource(id = com.intuit.sdp.R.dimen._8sdp).value.sp,
                        color = MaterialTheme.colorScheme.secondary,
                        fontWeight = FontWeight.Bold,
                        modifier = Modifier.padding(start = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp,end = dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp)
                    )

                 Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp))
                }
                if (accountManagerDetails.textEmailEnabled) {
                    Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp))

                    TextField(
                        modifier = Modifier.fillMaxWidth(),
                        textStyle = getEloNormalStdSecondaryText(),
                        value = email.value, onValueChange = {
                            email.value = it
                            accountManagerViewModel.setEmail(it)
                        },
                        placeholder = { Text("Enter Email") })
                }
                if (accountManagerDetails.textPasswordEnabled) {
                    Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp))

                    TextField(
                        modifier = Modifier.fillMaxWidth(),
                        textStyle = getEloNormalStdSecondaryText(),
                        keyboardOptions = KeyboardOptions(
                            keyboardType = KeyboardType.Password
                        ),
                        visualTransformation = PasswordVisualTransformation(),
                        value = password.value, onValueChange = {
                            password.value = it
                            accountManagerViewModel.setPassword(it)
                        },
                        placeholder = { Text("Enter Password") })
                }
                if (accountManagerDetails.buttonGetTokenEnabled || accountManagerDetails.buttonVerifyTokenEnabled) {

                    Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp))

                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        horizontalArrangement = Arrangement.Center
                    ) {
                        if (accountManagerDetails.buttonGetTokenEnabled) {
                            Button(
                                shape = CutCornerShape(
                                    topStart = 0f,
                                    topEnd = 0f,
                                    bottomEnd = 0f,
                                    bottomStart = 0f
                                ),
                                modifier = Modifier.weight(1f),
                                onClick = {
                                    Util.hideKeyboard(activity)
                                    AccountManager.instance.getEloAccessToken(
                                        activity.applicationContext,
                                        accountManagerDetails.email,
                                        accountManagerDetails.password,
                                        accountManagerViewModel.accessTokenHandler
                                    )

                                }) {
                                Text(text = "Get Token", style = getEloHighSizeWhiteBoldText())
                            }
                        }
                        if (accountManagerDetails.buttonVerifyTokenEnabled) {
                            Button(
                                shape = CutCornerShape(
                                    topStart = 0f,
                                    topEnd = 0f,
                                    bottomEnd = 0f,
                                    bottomStart = 0f
                                ),
                                modifier = Modifier.padding(start = dimensionResource(id=com.intuit.sdp.R.dimen._10sdp).value.dp).weight(1f),
                                onClick = {
                                    Util.hideKeyboard(activity)
                                    CoroutineScope(Dispatchers.IO).launch {
                                        AccountManager.instance.verifyEloToken(
                                            activity.applicationContext,
                                            accountManagerDetails.accessToken,
                                            accountManagerViewModel.accessTokenHandler
                                        )
                                    }
                                }) {
                                Text(text = "Verify Token", style = getEloHighSizeWhiteBoldText())
                            }
                        }

                    }
                    Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp))

                    HorizontalDivider(
                        thickness = 1.dp,
                        color = Color.LightGray,
                        modifier = Modifier.padding(top = dimensionResource(id = com.intuit.sdp.R.dimen._7sdp).value.dp, bottom = dimensionResource(id = com.intuit.sdp.R.dimen._7sdp).value.dp)
                    )
                    Text(text = "OR", style = getEloNormalStdSecondaryText())
                    HorizontalDivider(
                        thickness = 1.dp,
                        color = Color.LightGray,
                        modifier = Modifier.padding(top = dimensionResource(id = com.intuit.sdp.R.dimen._7sdp).value.dp, bottom = dimensionResource(id = com.intuit.sdp.R.dimen._7sdp).value.dp)
                    )
                }
                if (accountManagerDetails.buttonLogoutEnabled) {
                    Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp))
                    Button(
                        shape = CutCornerShape(
                            topStart = 0f,
                            topEnd = 0f,
                            bottomEnd = 0f,
                            bottomStart = 0f
                        ),
                        modifier = Modifier.padding(start = dimensionResource(id = com.intuit.sdp.R.dimen._8sdp).value.dp),
                        onClick = {
                            AccountManager.instance.logoutOAuth()
                        }) {
                        Text(text = "Logout",style = getEloHighSizeWhiteBoldText())
                    }
                    Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp))
                 }

                if (accountManagerDetails.textClientIdEnabled || accountManagerDetails.textSecretKeyEnabled || accountManagerDetails.buttonGetAndVerifyTokenEnabled) {

                    if (accountManagerDetails.textClientIdEnabled) {
                        Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp))
                        TextField(
                            modifier = Modifier.fillMaxWidth(),
                            textStyle = getEloNormalStdSecondaryText(),
                            value = clientId.value, onValueChange = {
                                clientId.value = it
                                accountManagerViewModel.setClientId(it)
                            },
                            placeholder = { Text("Enter Client ID") })
                    }
                    if (accountManagerDetails.textSecretKeyEnabled) {
                        Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp))
                        TextField(
                            modifier = Modifier.fillMaxWidth(),
                            textStyle = getEloNormalStdSecondaryText(),
                            value = secretKey.value, onValueChange = {
                                secretKey.value = it
                                accountManagerViewModel.setSecretKey(it)
                            },
                            placeholder = { Text("Enter Secret Key") })
                    }
                    if (accountManagerDetails.buttonGetAndVerifyTokenEnabled) {
                        Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp))
                        Button(
                            shape = CutCornerShape(
                                topStart = 0f,
                                topEnd = 0f,
                                bottomEnd = 0f,
                                bottomStart = 0f
                            ),
                            modifier = Modifier.padding(start = 10.dp),
                            onClick = {
                                AccountManager.instance.getEloOAuthToken(
                                    activity.applicationContext,
                                    accountManagerDetails.clientId,
                                    accountManagerDetails.secretKey,
                                    accountManagerViewModel.accessTokenHandler
                                )
                            }) {
                            Text(text = "Get and Verify Token",style = getEloHighSizeWhiteBoldText())
                        }
                        Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp))
                    }
                    HorizontalDivider(
                        thickness = 1.dp,
                        color = Color.LightGray,
                        modifier = Modifier.padding(top = dimensionResource(id = com.intuit.sdp.R.dimen._7sdp).value.dp, bottom = dimensionResource(id = com.intuit.sdp.R.dimen._7sdp).value.dp)
                    )
                    Text(text = "OR", style = getEloNormalStdSecondaryText())
                    HorizontalDivider(
                        thickness = 1.dp,
                        color = Color.LightGray,
                        modifier = Modifier.padding(top = dimensionResource(id = com.intuit.sdp.R.dimen._7sdp).value.dp, bottom = dimensionResource(id = com.intuit.sdp.R.dimen._7sdp).value.dp)
                    )
                }
                if (accountManagerDetails.buttonLoadAndVerifyTokenEnabled) {
                    Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._10sdp).value.dp))
                    Text(
                        text = "Load a file containing the credentials and verify",
                        style = getEloHighSizePrimaryBoldText(),
                        textAlign = TextAlign.Center
                    )
                    Spacer(modifier = Modifier.fillMaxWidth().height(dimensionResource(id = com.intuit.sdp.R.dimen._5sdp).value.dp))
                    Button(
                        shape = CutCornerShape(
                            topStart = 0f,
                            topEnd = 0f,
                            bottomEnd = 0f,
                            bottomStart = 0f
                        ),
                        modifier = Modifier.padding(start = 10.dp),
                        onClick = {
                            accountManagerViewModel.loadAndVerifyTokenFromFile()
                        }) {
                        Text(text = "Load and Verify Token", style = getEloHighSizeWhiteBoldText())
                    }
                }



                EloOutputText(accountManagerDetails.output)
                BottomButtonBar(activity) { accountManagerViewModel.setOutput("") }
            }
        }
    }
}


@Composable
fun DropdownList(
    itemList: List<String>,
    selectedIndex: Int,
    modifier: Modifier,
    onItemClick: (Int) -> Unit
) {
    val isDarkTheme = isSystemInDarkTheme()
    val backgroundColor = if (isDarkTheme) Color(0xFF2C2C2C) else Color.White
    val textColor = if (isDarkTheme) Color.White else Color.Black
    val dividerColor = if (isDarkTheme) Color.DarkGray else Color.LightGray
    val borderColor = if (isDarkTheme) Color.Gray else Color.Gray
    var showDropdown by rememberSaveable { mutableStateOf(false) }
    val scrollState = rememberScrollState()

    Column(
        modifier = Modifier,
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.Center
    ) {

        // button
        Box(
            modifier = modifier
                .clickable { showDropdown = true },
//            .clickable { showDropdown = !showDropdown },
            contentAlignment = Alignment.Center
        ) {
            Row(
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.Center
            ) {
                // Text showing the selected item
                Text(
                    text = itemList[selectedIndex],
                    color = textColor,
                    modifier = Modifier
                        .padding(10.dp),
                    style = getEloNormalStdSecondaryText()
                )

                // Arrow icon
                Icon(
                    imageVector = Icons.Default.ArrowDropDown, // Arrow icon from Material Icons
                    contentDescription = "Dropdown Arrow",
                    tint = textColor,
                    modifier = Modifier
                        .size(32.dp)
                        .padding(start = 8.dp)
                )
            }
        }

        // dropdown list
        Box {
            if (showDropdown) {
                Popup(
                    alignment = Alignment.TopCenter,
                    properties = PopupProperties(
                        excludeFromSystemGesture = true,
                    ),
                    // to dismiss on click outside
                    onDismissRequest = { showDropdown = false }
                ) {

                    Column(
                        modifier = modifier
                            .heightIn(max = 275.dp)
                            .fillMaxWidth()
                            .verticalScroll(state = scrollState)
                            .border(width = 1.dp, color = borderColor)
                            .background(backgroundColor),
                        horizontalAlignment = Alignment.CenterHorizontally,
                    ) {

                        itemList.onEachIndexed { index, item ->
                            if (index != 0) {
                                HorizontalDivider(thickness = 1.dp, color = dividerColor)
                            }
                            Box(
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .background(backgroundColor)
                                    .width(200.dp)
                                    .padding(7.dp)
                                    .clickable {
                                        onItemClick(index)
                                        showDropdown = !showDropdown
                                    },
                            ) {
                                Text(text = item,color = textColor,style = getEloNormalStdSecondaryText(), textAlign = TextAlign.Center)
                            }
                        }

                    }
                }
            }
        }
    }

}