package com.eloview.homesdk.testapp;

import static com.eloview.homesdk.testapp.Util.hideKeyboard;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.content.Context;
import android.content.Intent;
import android.graphics.Color;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.text.InputType;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.Switch;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;

import com.eloview.homesdk.accountManager.AccountManager;
import com.eloview.homesdk.networkManager.CertificateConfig;
import com.eloview.homesdk.networkManager.ESTConfig;
import com.eloview.homesdk.networkManager.Network;
import com.eloview.homesdk.networkManager.NetworkConfig;
import com.eloview.homesdk.networkManager.NetworkSecurityType;
import com.eloview.homesdk.networkManager.NetworkType;


public class NetworkActivity extends Activity {

    private static final int CERTIFICATE_CODE = 100;
    private static final int LOCAL_CERTIFICATE_CODE = 101;
    private final String TAG = NetworkActivity.class.getSimpleName();
    private String accessToken = "";
    private TextView txt_result;
    private Button bt_account_manager;
    private boolean isVerificationRequired = true;
    private final Handler networkHandler = new Handler(new Handler.Callback() {
        @RequiresApi(api = Build.VERSION_CODES.N)
        @Override
        public boolean handleMessage(@NonNull Message msg) {
            Bundle b = msg.getData();
            String key = Integer.toString(msg.what);
            String val = b.getString(key);
            switch (msg.what) {
                case Network.SET_NETWORK_RESPONSE_CODE:
                    Log.i(TAG, "SET_NETWORK_RESPONSE_CODE = " + val);
                    break;
                case Network.GET_NETWORK_RESPONSE_CODE:
                    Log.i(TAG, "GET_NETWORK_RESPONSE_CODE = " + val);
                    val = "GET_NETWORK_RESPONSE_CODE \n\n" + val;
                    break;
                case Network.GET_WIFI_MAC_ADDRESS_CODE:
                    Log.i(TAG, "GET_WIFI_MAC_ADDRESS_CODE = " + val);
                    val = "GET_WIFI_MAC_ADDRESS_CODE \n\n" + val;
                    break;
                case Network.GET_ETH_MAC_ADDRESS_CODE:
                    Log.i(TAG, "GET_ETH_MAC_ADDRESS_CODE = " + val);
                    val = "GET_ETH_MAC_ADDRESS_CODE \n\n" + val;
                    break;
                case AccountManager.TOKEN_VERIFY_FAIL:
                    val = "INVALID TOKEN\n" + val;
                    txt_result.setText(val);
                    Toast.makeText(NetworkActivity.this, getResources().getString(R.string.invalid_token_msg), Toast.LENGTH_LONG).show();
                    bt_account_manager.setBackgroundColor(Color.RED);
                    break;
                case AccountManager.TOKEN_VERIFY_SUCCESS:
                    bt_account_manager.setBackgroundColor(Color.GREEN);
                    if (AccountManagerActivity.emailPass) {
                        bt_account_manager.setText(getResources().getString(R.string.verified_by_email));
                    } else if (AccountManagerActivity.clientSecret) {
                        bt_account_manager.setText(getResources().getString(R.string.verified_by_auth));
                    } else if (AccountManagerActivity.tokenFromApi) {
                        bt_account_manager.setText(getResources().getString(R.string.verified_by_auth_api));
                    }
                    break;
                case Network.SET_EST_CERTIFICATE_CODE:
                    isVerificationRequired = true;
                    val = "SET_EST_SERVER_CODE\n" + val;
                    break;
                case Network.INSTALL_CERTIFICATE_CODE:
                    isVerificationRequired = true;
                    val = "INSTALL_CERTIFICATE_CODE\n" + val;
                    break;
                case Network.REMOVE_EST_CERTIFICATE_CODE:
                    val = "REMOVE_EST_CERTIFICATE_CODE\n" + val;
                    break;
                case Network.DHCP_SERVER_NAME:
                    val = "DHCP_SERVER_NAME\n" + val;
                    break;
            }
            txt_result.setText(val);
            return false;
        }
    });

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_network);
        txt_result = findViewById(R.id.txt_result);
        bt_account_manager = findViewById(R.id.btAccManagerNetwork);
    }

    @SuppressLint("NewApi")
    @Override
    protected void onStop() {
        super.onStop();
        Log.e(TAG, "onStop");
    }

    @Override
    protected void onResume() {
        super.onResume();
        accessToken = AccountManagerActivity.accessToken;
        setEthswitchstate();
        if (isVerificationRequired) {
            AccountManager.instance.verifyEloToken(NetworkActivity.this, accessToken, networkHandler);
        }
    }

    public void setNetwork(View view) {
        showNetworkConfigDialog();
    }

    private void setEthswitchstate() {
        Switch swEthFormat = findViewById(R.id.ethstate);
        Log.d(TAG, "ETHSTATE" + isEthOn(getApplicationContext()));
        swEthFormat.setChecked(isEthOn(getApplicationContext()));
    }

    public void getNetwork(View view) {
        clearOutput();
        Network.instance.getNetwork(getApplicationContext(), accessToken, networkHandler);
    }

    public void setEthState(View view) {
        Switch swTimeFormat = findViewById(R.id.ethstate);
        Network.instance.setEthState(NetworkActivity.this, accessToken, swTimeFormat.isChecked(), networkHandler);
    }

    public void getWifiMAC(View view) {
        clearOutput();
        Network.instance.getWiFiMac(getApplicationContext(), accessToken, networkHandler);
    }

    public void getEthMAC(View view) {
        clearOutput();
        Network.instance.getEthernetMac(getApplicationContext(), accessToken, networkHandler);
    }

    public void getDHCPRecord(View view) {
        clearOutput();
        Network.instance.getDHCPServerHostName(getApplicationContext(), accessToken, networkHandler);
    }

    public void returnMainNetwork(View view) {
        Intent intent = new Intent(NetworkActivity.this, MainActivity.class);
        intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
        startActivity(intent);
    }

    public void clearOutputNetwork(View view) {
        txt_result.setText("");
    }

    private void clearOutput() {
        txt_result.setText("");
    }

    public void exitAppNetwork(View view) {
        moveTaskToBack(true);
    }

    public void accountManagerNetwork(View view) {
        Intent intent = new Intent(NetworkActivity.this, AccountManagerActivity.class);
        startActivity(intent);
    }

    @Override
    protected void onPause() {
        super.onPause();
        Network.instance.unregisterNetworkAPIListener(NetworkActivity.this);
        Network.instance.unregisterCertificatesListener(NetworkActivity.this);
        Network.instance.unregisterDHCPRecordsAPIListener(NetworkActivity.this);
        Network.instance.unregisterRemoveCertificateAPIListener(NetworkActivity.this);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        Network.instance.unregisterNetworkAPIListener(NetworkActivity.this);
        Network.instance.unregisterCertificatesListener(NetworkActivity.this);
        Network.instance.unregisterDHCPRecordsAPIListener(NetworkActivity.this);
        Network.instance.unregisterRemoveCertificateAPIListener(NetworkActivity.this);
    }

    private void showNetworkConfigDialog() {
        final Dialog networkDialog = new Dialog(NetworkActivity.this, R.style.CustomTheme);
        networkDialog.requestWindowFeature(Window.FEATURE_NO_TITLE);
        networkDialog.setContentView(R.layout.dialog_network_config);
        if (networkDialog.getWindow() != null) {
            networkDialog.getWindow().addFlags(WindowManager.LayoutParams.FLAG_DIM_BEHIND);
            networkDialog.getWindow().setLayout(WindowManager.LayoutParams.MATCH_PARENT, WindowManager.LayoutParams.MATCH_PARENT);
            networkDialog.getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_RESIZE);
            networkDialog.getWindow().setDimAmount(0.3f);
        }
        networkDialog.setCancelable(true);

//        RadioGroup rgNetworkType = networkDialog.findViewById(R.id.rgNetworkType);
        final RadioButton rbWiFi = networkDialog.findViewById(R.id.rbWiFi);

        final RadioGroup rgNetworkSecurityType = networkDialog.findViewById(R.id.rgNetworkSecurityType);
        RadioButton rbOpen = networkDialog.findViewById(R.id.rbOpen);
        String isDeviceSecured = Util.readSystemProperty("persist.vendor.payment.secured");
        if ("1".equals(isDeviceSecured)) {
            rbOpen.setEnabled(false);
        }

        final EditText etName = networkDialog.findViewById(R.id.etName);
        final EditText etAlias = networkDialog.findViewById(R.id.etAlias);
        final EditText etPassword = networkDialog.findViewById(R.id.etPassword);
        final EditText etDns1 = networkDialog.findViewById(R.id.etDns1);
        final EditText etDns2 = networkDialog.findViewById(R.id.etDns2);
        final EditText etGateway = networkDialog.findViewById(R.id.etGateway);
        final EditText etStaticIp = networkDialog.findViewById(R.id.etStaticIp);
        final EditText etNetworkPrefix = networkDialog.findViewById(R.id.etNetworkPrefix);
        final EditText etProxyHost = networkDialog.findViewById(R.id.etProxyHost);
        final EditText etProxyPort = networkDialog.findViewById(R.id.etProxyPort);

        final CheckBox cbPoorConnection = networkDialog.findViewById(R.id.cbPoorConnection);
        final CheckBox cbDefaultExclude = networkDialog.findViewById(R.id.cbDefaultExclude);

        Button btnCancel = networkDialog.findViewById(R.id.btnCancel);
        Button btnApply = networkDialog.findViewById(R.id.btnApply);

        btnApply.setOnClickListener(v -> {
            NetworkConfig networkConfig;
            if (rbWiFi.isChecked()) {
                networkConfig = new NetworkConfig(NetworkType.WiFi);
                // get selected radio button from radioGroup
                int selectedId = rgNetworkSecurityType.getCheckedRadioButtonId();
                if (selectedId != -1) {
                    if (selectedId == R.id.rbWpa) {
                        networkConfig.setSecurityType(NetworkSecurityType.WPA);
                    } else if (selectedId == R.id.rbWpa2) {
                        networkConfig.setSecurityType(NetworkSecurityType.WPA2);
                    } else if (selectedId == R.id.rbOpen) {
                        networkConfig.setSecurityType(NetworkSecurityType.OPEN);
                    } else if (selectedId == R.id.rbEAP) {
                        networkConfig.setSecurityType(NetworkSecurityType.EAP_TLS);
                    }
                } else {
                    Toast.makeText(getApplicationContext(), "Select Network Type of WiFi", Toast.LENGTH_SHORT).show();
                }
            } else {
                networkConfig = new NetworkConfig(NetworkType.LAN);
            }

            String Name = etName.getText().toString().trim();
            if (!TextUtils.isEmpty(Name)) {
                networkConfig.setName(Name);
            }
            String Alias = etAlias.getText().toString().trim();
            if (!TextUtils.isEmpty(Alias)) {
                networkConfig.setCertAlias(Alias);
            }

            String Password = etPassword.getText().toString().trim();
            if (!TextUtils.isEmpty(Password)) {
                networkConfig.setPassword(Password);
            }
            String Gateway = etGateway.getText().toString().trim();
            if (!TextUtils.isEmpty(Gateway)) {
                networkConfig.setGateway(Gateway);
            }
            String StaticIp = etStaticIp.getText().toString().trim();
            if (!TextUtils.isEmpty(StaticIp)) {
                networkConfig.setStaticIp(StaticIp);
            }
            String NetWorkPrefix = etNetworkPrefix.getText().toString().trim();
            Log.d(TAG, "NetWorkPrefix" + NetWorkPrefix);
            if (!TextUtils.isEmpty(NetWorkPrefix)) {
                Log.d(TAG, "not empty");
                networkConfig.setNetworkPrefix(NetWorkPrefix);
            }
            String Dns1 = etDns1.getText().toString().trim();
            if (!TextUtils.isEmpty(Dns1)) {
                networkConfig.setDns1(Dns1);
            }
            String Dns2 = etDns2.getText().toString().trim();
            if (!TextUtils.isEmpty(Dns2)) {
                networkConfig.setDns2(Dns2);
            }
            String host = etProxyHost.getText().toString().trim();
            if (!TextUtils.isEmpty(host)) {
                networkConfig.setProxyHost(host);
            }
            if (!TextUtils.isEmpty(etProxyPort.getText().toString().trim())) {
                try {
                    int port = Integer.parseInt(etProxyPort.getText().toString().trim());
                    networkConfig.setProxyPort(port);
                } catch (Exception e) {
                    e.printStackTrace();
                    Toast.makeText(getApplicationContext(), "Invalid proxy port", Toast.LENGTH_SHORT).show();
                }
            }

            networkConfig.setAvoidPoorWifi(cbPoorConnection.isChecked());
            networkConfig.setDefaultExclude(cbDefaultExclude.isChecked());

            Log.d(TAG, "onClick: " + networkConfig);
            Network.instance.setNetwork(getApplicationContext(), accessToken, networkConfig, networkHandler);

            networkDialog.dismiss();
        });

        btnCancel.setOnClickListener(v -> networkDialog.dismiss());

        networkDialog.show();
    }

    public void setCertificate(View view) {
        isVerificationRequired = false;
        startActivityForResult(new Intent(NetworkActivity.this, ESTCertificateActivity.class), CERTIFICATE_CODE);
    }

    public void installLocalCertificate(View view) {
        isVerificationRequired = false;
        startActivityForResult(new Intent(NetworkActivity.this, CertificateActivity.class), LOCAL_CERTIFICATE_CODE);
    }

    public void removeCertificate(View view) {
        removeAliasDialog();
    }

    private void removeAliasDialog() {
        hideKeyboard(this);
        final AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setTitle("Enter Alias name");
        final EditText input = new EditText(this);
        input.setInputType(InputType.TYPE_CLASS_TEXT);
        input.setHint(getResources().getString(R.string.est_alias));
        builder.setView(input);
        builder.setPositiveButton("OK", (dialog, which) -> {
            String alias = input.getText().toString();
            Network.instance.removeCertificate(getApplicationContext(), accessToken, alias, networkHandler);
        });
        builder.setNegativeButton("Cancel", (dialog, which) -> dialog.cancel());
        builder.show();
    }

    private boolean isEthOn(Context context) {
        Log.d(TAG, "isEthOn");
        ConnectivityManager cm = (ConnectivityManager) context
                .getSystemService(Context.CONNECTIVITY_SERVICE);

        if (cm != null) {
            NetworkInfo activeNetwork = cm.getActiveNetworkInfo();

            boolean isConnected = activeNetwork != null &&
                    activeNetwork.isConnectedOrConnecting();
            if (isConnected) {
                Log.d(TAG, "network type >>" + activeNetwork.getType());
                return activeNetwork.getType() == ConnectivityManager.TYPE_ETHERNET;
            } else {
                Log.d(TAG, "No Internet Connection");
            }
        } else {
            Log.d(TAG, "Could not find connectivity manager");
        }
        return false;
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        Log.d(TAG, "onActivityResult requestCode = [" + requestCode + "], resultCode = [" + resultCode + "]");
        if (resultCode == RESULT_OK) {
            if (requestCode == CERTIFICATE_CODE) {
                ESTConfig estConfigModel = data.getParcelableExtra("est");
                applyCertificate(estConfigModel);
            } else if (requestCode == LOCAL_CERTIFICATE_CODE) {
                Log.d(TAG, "LOCAL_CERTIFICATE_CODE");
                String result = data.getStringExtra("result");
                if (result != null) {
                    txt_result.setText(result);
                } else {
                    CertificateConfig certificateConfig = data.getParcelableExtra("certificate");
                    applyLocalCertificate(certificateConfig);
                }
            }
        } else {
            Log.d(TAG, "ResultCode not ok");
            isVerificationRequired = true;
        }
    }

    private void applyCertificate(ESTConfig estConfigModel) {
        clearOutput();
        Network.instance.installCertificate(getApplicationContext(), accessToken, estConfigModel, networkHandler);
    }

    private void applyLocalCertificate(CertificateConfig certificateConfig) {
        clearOutput();
        Network.instance.installLocalCertificate(getApplicationContext(), accessToken, certificateConfig, networkHandler);
    }
}
